/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.astyanax.entitystore;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.netflix.astyanax.Keyspace;
import com.netflix.astyanax.MutationBatch;
import com.netflix.astyanax.Serializer;
import com.netflix.astyanax.connectionpool.exceptions.ConnectionException;
import com.netflix.astyanax.entitystore.EntityManager;
import com.netflix.astyanax.entitystore.EntityMapper;
import com.netflix.astyanax.entitystore.LifecycleEvents;
import com.netflix.astyanax.entitystore.MappingUtils;
import com.netflix.astyanax.entitystore.NativeQuery;
import com.netflix.astyanax.model.ColumnFamily;
import com.netflix.astyanax.model.ColumnList;
import com.netflix.astyanax.model.ConsistencyLevel;
import com.netflix.astyanax.model.CqlResult;
import com.netflix.astyanax.model.Row;
import com.netflix.astyanax.model.Rows;
import com.netflix.astyanax.query.ColumnFamilyQuery;
import com.netflix.astyanax.recipes.reader.AllRowsReader;
import com.netflix.astyanax.retry.RetryPolicy;
import com.netflix.astyanax.serializers.StringSerializer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import javax.persistence.PersistenceException;
import org.apache.commons.lang.StringUtils;

public class DefaultEntityManager<T, K>
implements EntityManager<T, K> {
    private final EntityMapper<T, K> entityMapper;
    private final Keyspace keyspace;
    private final ColumnFamily<K, String> columnFamily;
    private final ConsistencyLevel readConsitency;
    private final ConsistencyLevel writeConsistency;
    private final RetryPolicy retryPolicy;
    private final LifecycleEvents<T> lifecycleHandler;
    private final boolean autoCommit;
    private final ThreadLocal<MutationBatch> tlMutation = new ThreadLocal();

    private DefaultEntityManager(Builder<T, K> builder) {
        this.entityMapper = ((Builder)builder).entityMapper;
        this.keyspace = ((Builder)builder).keyspace;
        this.columnFamily = ((Builder)builder).columnFamily;
        this.readConsitency = ((Builder)builder).readConsitency;
        this.writeConsistency = ((Builder)builder).writeConsistency;
        this.retryPolicy = ((Builder)builder).retryPolicy;
        this.lifecycleHandler = ((Builder)builder).lifecycleHandler;
        this.autoCommit = ((Builder)builder).autoCommit;
    }

    @Override
    public void put(T entity) throws PersistenceException {
        try {
            this.lifecycleHandler.onPrePersist(entity);
            MutationBatch mb = this.newMutationBatch();
            this.entityMapper.fillMutationBatch(mb, this.columnFamily, entity);
            if (this.autoCommit) {
                mb.execute();
            }
            this.lifecycleHandler.onPostPersist(entity);
        }
        catch (Exception e) {
            throw new PersistenceException("failed to put entity ", (Throwable)e);
        }
    }

    @Override
    public T get(K id) throws PersistenceException {
        try {
            ColumnFamilyQuery<K, String> cfq = this.newQuery();
            ColumnList cl = (ColumnList)cfq.getKey(id).execute().getResult();
            if (cl.isEmpty()) {
                return null;
            }
            T entity = this.entityMapper.constructEntity(id, (ColumnList<String>)cl);
            this.lifecycleHandler.onPostLoad(entity);
            return entity;
        }
        catch (Exception e) {
            throw new PersistenceException("failed to get entity " + id, (Throwable)e);
        }
    }

    @Override
    public void delete(K id) throws PersistenceException {
        try {
            MutationBatch mb = this.getMutationBatch();
            mb.withRow(this.columnFamily, id).delete();
            if (this.autoCommit) {
                mb.execute();
            }
        }
        catch (Exception e) {
            throw new PersistenceException("failed to delete entity " + id, (Throwable)e);
        }
    }

    @Override
    public void remove(T entity) throws PersistenceException {
        Object id = null;
        try {
            this.lifecycleHandler.onPreRemove(entity);
            id = this.entityMapper.getEntityId(entity);
            MutationBatch mb = this.newMutationBatch();
            mb.withRow(this.columnFamily, id).delete();
            if (this.autoCommit) {
                mb.execute();
            }
            this.lifecycleHandler.onPostRemove(entity);
        }
        catch (Exception e) {
            throw new PersistenceException("failed to delete entity " + id, (Throwable)e);
        }
    }

    @Override
    public List<T> getAll() throws PersistenceException {
        final ArrayList entities = Lists.newArrayList();
        this.visitAll(new Function<T, Boolean>(){

            public synchronized Boolean apply(T entity) {
                entities.add(entity);
                try {
                    DefaultEntityManager.this.lifecycleHandler.onPostLoad(entity);
                }
                catch (Exception exception) {
                    // empty catch block
                }
                return true;
            }
        });
        return entities;
    }

    @Override
    public List<T> get(Collection<K> ids) throws PersistenceException {
        try {
            ColumnFamilyQuery<K, String> cfq = this.newQuery();
            Rows rows = (Rows)cfq.getRowSlice(ids).execute().getResult();
            ArrayList entities = Lists.newArrayListWithExpectedSize((int)rows.size());
            for (Row row : rows) {
                if (row.getColumns().isEmpty()) continue;
                T entity = this.entityMapper.constructEntity(row.getKey(), (ColumnList<String>)row.getColumns());
                this.lifecycleHandler.onPostLoad(entity);
                entities.add(entity);
            }
            return entities;
        }
        catch (Exception e) {
            throw new PersistenceException("failed to get entities " + ids, (Throwable)e);
        }
    }

    @Override
    public void delete(Collection<K> ids) throws PersistenceException {
        MutationBatch mb = this.getMutationBatch();
        try {
            for (K id : ids) {
                mb.withRow(this.columnFamily, id).delete();
            }
            if (this.autoCommit) {
                mb.execute();
            }
        }
        catch (Exception e) {
            throw new PersistenceException("failed to delete entities " + ids, (Throwable)e);
        }
    }

    @Override
    public void remove(Collection<T> entities) throws PersistenceException {
        MutationBatch mb = this.getMutationBatch();
        try {
            for (T entity : entities) {
                this.lifecycleHandler.onPreRemove(entity);
                K id = this.entityMapper.getEntityId(entity);
                mb.withRow(this.columnFamily, id).delete();
            }
            mb.execute();
            for (T entity : entities) {
                this.lifecycleHandler.onPostRemove(entity);
            }
        }
        catch (Exception e) {
            throw new PersistenceException("failed to delete entities ", (Throwable)e);
        }
    }

    @Override
    public void put(Collection<T> entities) throws PersistenceException {
        MutationBatch mb = this.getMutationBatch();
        try {
            for (T entity : entities) {
                this.lifecycleHandler.onPrePersist(entity);
                this.entityMapper.fillMutationBatch(mb, this.columnFamily, entity);
            }
            if (this.autoCommit) {
                mb.execute();
            }
            for (T entity : entities) {
                this.lifecycleHandler.onPostPersist(entity);
            }
        }
        catch (Exception e) {
            throw new PersistenceException("failed to put entities ", (Throwable)e);
        }
    }

    @Override
    public void visitAll(final Function<T, Boolean> callback) throws PersistenceException {
        try {
            new AllRowsReader.Builder(this.keyspace, this.columnFamily).withIncludeEmptyRows(Boolean.valueOf(false)).forEachRow(new Function<Row<K, String>, Boolean>(){

                public Boolean apply(Row<K, String> row) {
                    if (row.getColumns().isEmpty()) {
                        return true;
                    }
                    Object entity = DefaultEntityManager.this.entityMapper.constructEntity(row.getKey(), (ColumnList<String>)row.getColumns());
                    try {
                        DefaultEntityManager.this.lifecycleHandler.onPostLoad(entity);
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                    return (Boolean)callback.apply(entity);
                }
            }).build().call();
        }
        catch (Exception e) {
            throw new PersistenceException("Failed to fetch all entites", (Throwable)e);
        }
    }

    @Override
    public List<T> find(String cql) throws PersistenceException {
        Preconditions.checkArgument((boolean)StringUtils.left((String)cql, (int)6).equalsIgnoreCase("SELECT"), (Object)"CQL must be SELECT statement");
        try {
            CqlResult results = (CqlResult)this.newQuery().withCql(cql).execute().getResult();
            ArrayList entities = Lists.newArrayListWithExpectedSize((int)results.getRows().size());
            for (Row row : results.getRows()) {
                if (row.getColumns().isEmpty()) continue;
                T entity = this.entityMapper.constructEntity(row.getKey(), (ColumnList<String>)row.getColumns());
                this.lifecycleHandler.onPostLoad(entity);
                entities.add(entity);
            }
            return entities;
        }
        catch (Exception e) {
            throw new PersistenceException("Failed to execute cql query", (Throwable)e);
        }
    }

    private MutationBatch newMutationBatch() {
        MutationBatch mb = this.keyspace.prepareMutationBatch();
        if (this.writeConsistency != null) {
            mb.withConsistencyLevel(this.writeConsistency);
        }
        if (this.retryPolicy != null) {
            mb.withRetryPolicy(this.retryPolicy);
        }
        return mb;
    }

    private MutationBatch getMutationBatch() {
        if (this.autoCommit) {
            return this.newMutationBatch();
        }
        MutationBatch mb = this.tlMutation.get();
        if (mb == null) {
            mb = this.newMutationBatch();
            this.tlMutation.set(mb);
        }
        return mb;
    }

    private ColumnFamilyQuery<K, String> newQuery() {
        ColumnFamilyQuery cfq = this.keyspace.prepareQuery(this.columnFamily);
        if (this.readConsitency != null) {
            cfq.setConsistencyLevel(this.readConsitency);
        }
        if (this.retryPolicy != null) {
            cfq.withRetryPolicy(this.retryPolicy);
        }
        return cfq;
    }

    @Override
    public void createStorage(Map<String, Object> options) throws PersistenceException {
        try {
            this.keyspace.createColumnFamily(this.columnFamily, options);
        }
        catch (ConnectionException e) {
            throw new PersistenceException("Unable to create column family " + this.columnFamily.getName(), (Throwable)e);
        }
    }

    @Override
    public void deleteStorage() throws PersistenceException {
        try {
            this.keyspace.dropColumnFamily(this.columnFamily);
        }
        catch (ConnectionException e) {
            throw new PersistenceException("Unable to drop column family " + this.columnFamily.getName(), (Throwable)e);
        }
    }

    @Override
    public void truncate() throws PersistenceException {
        try {
            this.keyspace.truncateColumnFamily(this.columnFamily);
        }
        catch (ConnectionException e) {
            throw new PersistenceException("Unable to drop column family " + this.columnFamily.getName(), (Throwable)e);
        }
    }

    @Override
    public void commit() throws PersistenceException {
        MutationBatch mb = this.tlMutation.get();
        if (mb != null) {
            try {
                mb.execute();
            }
            catch (ConnectionException e) {
                throw new PersistenceException("Failed to commit mutation batch", (Throwable)e);
            }
        }
    }

    @Override
    public NativeQuery<T, K> createNativeQuery() {
        throw new UnsupportedOperationException("Not implemented yet");
    }

    public static class Builder<T, K> {
        private Class<T> clazz = null;
        private EntityMapper<T, K> entityMapper = null;
        private Keyspace keyspace = null;
        private ColumnFamily<K, String> columnFamily = null;
        private ConsistencyLevel readConsitency = null;
        private ConsistencyLevel writeConsistency = null;
        private Integer ttl = null;
        private RetryPolicy retryPolicy = null;
        private LifecycleEvents<T> lifecycleHandler = null;
        private String columnFamilyName = null;
        private boolean autoCommit = true;

        public Builder<T, K> withEntityType(Class<T> clazz) {
            Preconditions.checkNotNull(clazz);
            this.clazz = clazz;
            return this;
        }

        public Builder<T, K> withKeyspace(Keyspace keyspace) {
            Preconditions.checkNotNull((Object)keyspace);
            this.keyspace = keyspace;
            return this;
        }

        public Builder<T, K> withColumnFamily(ColumnFamily<K, String> columnFamily) {
            Preconditions.checkState((this.columnFamilyName == null && this.columnFamily == null ? 1 : 0) != 0, (Object)"withColumnFamily called multiple times");
            Preconditions.checkNotNull(columnFamily);
            this.columnFamily = columnFamily;
            return this;
        }

        public Builder<T, K> withColumnFamily(String columnFamilyName) {
            Preconditions.checkState((this.columnFamilyName == null && this.columnFamily == null ? 1 : 0) != 0, (Object)"withColumnFamily called multiple times");
            Preconditions.checkNotNull((Object)columnFamilyName);
            this.columnFamilyName = columnFamilyName;
            return this;
        }

        public Builder<T, K> withReadConsistency(ConsistencyLevel level) {
            Preconditions.checkNotNull((Object)level);
            this.readConsitency = level;
            return this;
        }

        public Builder<T, K> withWriteConsistency(ConsistencyLevel level) {
            Preconditions.checkNotNull((Object)level);
            this.writeConsistency = level;
            return this;
        }

        public Builder<T, K> withConsistency(ConsistencyLevel level) {
            Preconditions.checkNotNull((Object)level);
            this.readConsitency = level;
            this.writeConsistency = level;
            return this;
        }

        public Builder<T, K> withTTL(Integer ttl) {
            this.ttl = ttl;
            return this;
        }

        public Builder<T, K> withRetryPolicy(RetryPolicy policy) {
            Preconditions.checkNotNull((Object)policy);
            this.retryPolicy = policy;
            return this;
        }

        public Builder<T, K> withAutoCommit(boolean autoCommit) {
            this.autoCommit = autoCommit;
            return this;
        }

        public DefaultEntityManager<T, K> build() {
            Preconditions.checkNotNull(this.clazz, (Object)"withEntityType(...) is not set");
            Preconditions.checkNotNull((Object)this.keyspace, (Object)"withKeyspace(...) is not set");
            this.entityMapper = new EntityMapper(this.clazz, this.ttl);
            this.lifecycleHandler = new LifecycleEvents<T>(this.clazz);
            if (this.columnFamily == null) {
                if (this.columnFamilyName == null) {
                    this.columnFamilyName = this.entityMapper.getEntityName();
                }
                this.columnFamily = new ColumnFamily(this.columnFamilyName, MappingUtils.getSerializerForField(this.entityMapper.getId()), (Serializer)StringSerializer.get());
            }
            return new DefaultEntityManager(this);
        }
    }
}

