package org.apache.maven.ear.j2ee;

/* ====================================================================
 *   Copyright 2001-2006 The Apache Software Foundation.
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 * ====================================================================
 */

import java.io.IOException;
import java.io.InputStream;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;


/**
 * A class to resolve external entity definitions for j2ee artifacts.
 *
 * @author  dion
 */
public class J2EEEntityResolver implements EntityResolver
{
    /** log for debug output */
    private static final Log LOG =
        LogFactory.getLog( J2EEEntityResolver.class );

    /** list of j2ee dtds that are being made available */
    public static final String[] J2EE_DTDS =
        new String[]
        {
            "-//Sun Microsystems, Inc.//DTD J2EE Application Client 1.3//EN",
            "-//Sun Microsystems, Inc.//DTD J2EE Application 1.3//EN",
            "-//Sun Microsystems, Inc.//DTD Enterprise JavaBeans 2.0//EN",
            "-//Sun Microsystems, Inc.//DTD Web Application 2.2//EN",
            "-//Sun Microsystems, Inc.//DTD Web Application 2.3//EN",
            "-//Sun Microsystems, Inc.//DTD JSP Tag Library 1.2//EN"
        };
    /** list of j2ee resources that are being made available */
    public static final String[] J2EE_RESOURCES =
        new String[]
        {
            "/plugin-resources/application-client_1_3.dtd",
            "/plugin-resources/application_1_3.dtd",
            "/plugin-resources/ejb-jar_2_0.dtd",
            "/plugin-resources/web-app_2.2.dtd",
            "/plugin-resources/web-app_2_3.dtd",
            "/plugin-resources/web-jsptaglibrary_1_2.dtd"
        };

    /** map of ids to resource names */
    private Map idToResource = new HashMap();

    /** Creates a new instance of EntityResolver */
    public J2EEEntityResolver()
    {
        for ( int i = 0; i < J2EE_DTDS.length; i++ )
        {
            idToResource.put( J2EE_DTDS[i], J2EE_RESOURCES[i] );
        }
    }

    /** resolve the entity given by the provided Ids
     * @param publicId the public id of the entity
     * @param systemId the 'system location' (typically a URL) of the entity
     * @return an {@link InputSource input source} for retrieval of the entity
     * @throws IOException when an I/O error occurs retrieving the entity
     * @throws SAXException if there are any problems
     */
    public InputSource resolveEntity( String publicId, String systemId )
        throws SAXException, IOException
    {
        LOG.debug( "resolving entity with publicId='" + publicId
            + ", systemId='" + systemId + "'" );

        if ( publicId != null )
        {
            String resource = (String) idToResource.get( publicId );

            LOG.debug( "resource found in map ='" + resource + "'" );

            if ( resource != null )
            {
                InputStream in = getClass().getResourceAsStream( resource );

                LOG.debug( "input stream ='" + in + "'" );

                if ( in != null )
                {
                    return new InputSource( in );
                }
            }
        }

        return null;
    }

    /** Getter for publicId to resource name map.
     * @return Value of property idToResource.
     */
    protected Map getIdToResource()
    {
        return idToResource;
    }

    /** Setter for publicId to resource name map.
     * @param newIdToResource New value of property idToResource.
     */
    protected void setIdToResource( Map newIdToResource )
    {
        this.idToResource = newIdToResource;
    }
}
