package com.ibm.as400.access;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;

/* 
 * Class used for seamless failover 
 * The order of the methods is the same as AS400JDBCStatement so we can verify 
 * that the superclass methods will not be run. 
 */
public class AS400JDBCStatementRedirect 
extends AS400JDBCStatement {
   AS400JDBCStatement stmt_; 
   AS400JDBCConnection connection_; 
   
   AS400JDBCStatementRedirect(AS400JDBCStatement stmt) throws SQLException {
     super(); 
     stmt_ = stmt; 
     connection_ = (AS400JDBCConnection) stmt.getConnection(); 
   }
  
   public void addBatch(String sql) throws SQLException {
     stmt_.addBatch(sql); 
   }

   public void cancel() throws SQLException {
     stmt_.cancel(); 
   }

   void checkOpen() throws SQLException {
     stmt_.checkOpen(); 
   }

   public void clearBatch() throws SQLException {
     stmt_.clearBatch(); 
   }

   public void clearWarnings() throws SQLException {
     stmt_.clearWarnings(); 
   }

   
   public void close() throws SQLException {
     
     int retryCount = AS400JDBCConnectionRedirect.SEAMLESS_RETRY_COUNT; 
     while(true) {
       
       try {
          stmt_.close(); 
          return; 
       } catch (AS400JDBCTransientException e) {
         if (connection_.canSeamlessFailover()) {
           retryCount--; 
           if (retryCount >= 0 ) {
             // Keep retrying
           } else {
             throw e; 
           }
         } else {
           throw e; 
         }
       }
     } /* retry */

   }

   void finishClosing() throws SQLException {
     stmt_.finishClosing(); 
   }
   
   void closeResultSet(int reuseFlag) throws SQLException {
     stmt_.closeResultSet(reuseFlag);
   }
   
   /* Note: commonExecute not needed */
   /* Note: commonExecuteAfter not needed */ 
   /* Note: commonExecuteBefore not needed */
   /* Note: commonPrepare not needed */
   /* Note: commonAfter not needed */
   /* Note: commonBefore not needed */
   /* Note: commonBypass not needed */
   
   public boolean execute(String sql) throws SQLException {
     int retryCount = AS400JDBCConnectionRedirect.SEAMLESS_RETRY_COUNT; 
     while (true) {
       try {
          return stmt_.execute(sql);  
       } catch (AS400JDBCTransientException e) {
         if (connection_.canSeamlessFailover()) {
           retryCount--; 
           if (retryCount >= 0 ) {
             // Keep looping 
           } else {
             throw e; 
           }
         } else {
           throw e; 
         }
       }
     } /* retry */
   }

   public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
     int retryCount = AS400JDBCConnectionRedirect.SEAMLESS_RETRY_COUNT; 
     while (true) {
       try {
          return stmt_.execute(sql, autoGeneratedKeys); 
       } catch (AS400JDBCTransientException e) {
         if (connection_.canSeamlessFailover()) {
           retryCount--; 
           if (retryCount >= 0 ) {
             // keep retrying 
           } else {
             throw e; 
           }
         } else {
           throw e; 
         }
       }
     } /* retry */
   }

   public boolean execute(String sql, int[] columnIndexes) throws SQLException {
     int retryCount = AS400JDBCConnectionRedirect.SEAMLESS_RETRY_COUNT; 
     while (true) {
       try {
          return stmt_.execute(sql, columnIndexes); 
       } catch (AS400JDBCTransientException e) {
         if (connection_.canSeamlessFailover()) {
           retryCount--; 
           if (retryCount >= 0 ) {
             // We will continue looping
           } else {
             throw e; 
           }
         } else {
           throw e; 
         }
       }
     } /* retry */
   }

   public boolean execute(String sql, String[] columnNames) throws SQLException {
     
     int retryCount = AS400JDBCConnectionRedirect.SEAMLESS_RETRY_COUNT; 
     while(true) {
       
       try {
          return stmt_.execute(sql, columnNames); 
       } catch (AS400JDBCTransientException e) {
         if (connection_.canSeamlessFailover()) {
           retryCount--; 
           if (retryCount >= 0 ) {
             // Keep retrying
           } else {
             throw e; 
           }
         } else {
           throw e; 
         }
       }
     } /* retry */
   }

   public int[] executeBatch() throws SQLException {
     return stmt_.executeBatch(); 
   }
  
   
  public ResultSet executeQuery(String sql) throws SQLException {
    
    int retryCount = AS400JDBCConnectionRedirect.SEAMLESS_RETRY_COUNT; 
    while(true) {
      
      try {
        AS400JDBCResultSet rs = (AS400JDBCResultSet) stmt_.executeQuery(sql);
        if (rs != null) { 
           rs.setStatement(this);
        }
        return  rs; 
      } catch (AS400JDBCTransientException e) {
        if (connection_.canSeamlessFailover()) {
          retryCount--; 
          if (retryCount >= 0 ) {
            // Keep retrying
          } else {
            throw e; 
          }
        } else {
          throw e; 
        }
      }
    } /* retry */
  }

  
  
  

  
  public int executeUpdate(String sql) throws SQLException {
    
    int retryCount = AS400JDBCConnectionRedirect.SEAMLESS_RETRY_COUNT; 
    while(true) {
      
      try {
        return stmt_.executeUpdate(sql); 
      } catch (AS400JDBCTransientException e) {
        if (connection_.canSeamlessFailover()) {
          retryCount--; 
          if (retryCount >= 0 ) {
            // Keep retrying
          } else {
            throw e; 
          }
        } else {
          throw e; 
        }
      }
    } /* retry */
  }

  public int executeUpdate(String sql, int autoGeneratedKeys)
      throws SQLException {
    
    int retryCount = AS400JDBCConnectionRedirect.SEAMLESS_RETRY_COUNT; 
    while(true) {
      
      try {
         return stmt_.executeUpdate(sql,autoGeneratedKeys); 
      } catch (AS400JDBCTransientException e) {
        if (connection_.canSeamlessFailover()) {
          retryCount--; 
          if (retryCount >= 0 ) {
            // Keep retrying
          } else {
            throw e; 
          }
        } else {
          throw e; 
        }
      }
    } /* retry */
  }

  public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
    
    int retryCount = AS400JDBCConnectionRedirect.SEAMLESS_RETRY_COUNT; 
    while(true) {
      
      try {
         return stmt_.executeUpdate(sql, columnIndexes); 
      } catch (AS400JDBCTransientException e) {
        if (connection_.canSeamlessFailover()) {
          retryCount--; 
          if (retryCount >= 0 ) {
            // Keep retrying
          } else {
            throw e; 
          }
        } else {
          throw e; 
        }
      }
    } /* retry */
  }

  public int executeUpdate(String sql, String[] columnNames)
      throws SQLException {
    
    int retryCount = AS400JDBCConnectionRedirect.SEAMLESS_RETRY_COUNT; 
    while(true) {
      
      try {
         return stmt_.executeUpdate(sql, columnNames); 
      } catch (AS400JDBCTransientException e) {
        if (connection_.canSeamlessFailover()) {
          retryCount--; 
          if (retryCount >= 0 ) {
            // Keep retrying
          } else {
            throw e; 
          }
        } else {
          throw e; 
        }
      }
    } /* retry */
  }

  /* Note: finalize not used */

  int getBlockingFactor(String cursorSensitivityProperty, JDSQLStatement sqlStatement, int rowLength) { 
    return stmt_.getBlockingFactor(cursorSensitivityProperty, sqlStatement, rowLength);
  }
  
  public Connection getConnection() throws SQLException {
    return stmt_.getConnection(); 
  }

  String getCursorName() {
    return stmt_.getCursorName(); 
  }
  
  public int getFetchDirection() throws SQLException {
    return stmt_.getFetchDirection(); 
  }

  public int getFetchSize() throws SQLException {
    return stmt_.getFetchSize(); 
  }

  public ResultSet getGeneratedKeys() throws SQLException {
    return stmt_.getGeneratedKeys(); 
  }

  public int getMaxFieldSize() throws SQLException {
    return stmt_.getMaxFieldSize();
  }

  public int getMaxRows() throws SQLException {
    return stmt_.getMaxRows(); 
  }

  
  public boolean getMoreResults() throws SQLException {
    return stmt_.getMoreResults(); 
  }


  public boolean getMoreResults(int current) throws SQLException {
    return stmt_.getMoreResults(current); 
  }

  public int getPositionOfSyntaxError() throws SQLException {
    return stmt_.getPositionOfSyntaxError(); 
  }

  public int getQueryTimeout() throws SQLException {
    return stmt_.getQueryTimeout(); 
  }

  public ResultSet getResultSet() throws SQLException {
    AS400JDBCResultSet rs = (AS400JDBCResultSet) stmt_.getResultSet();
    if (rs != null) { 
       rs.setStatement(this);
    }
    return rs; 
  }

  public int getResultSetConcurrency() throws SQLException {
    return stmt_.getResultSetConcurrency(); 
  }

  public int getResultSetHoldability() throws SQLException {
    return stmt_.getResultSetHoldability(); 
  }

  public int getResultSetType() throws SQLException {
    return stmt_.getResultSetType(); 
  }

  public int getUpdateCount() throws SQLException {
    return stmt_.getUpdateCount(); 
  }

  public SQLWarning getWarnings() throws SQLException {
    return stmt_.getWarnings(); 
  }
  
  void setHoldStatement(boolean hold) {
    stmt_.setHoldStatement(hold);
  }
  
  boolean isHoldStatement() { 
    return stmt_.isHoldStatement(); 
  }
  
  void setAssociatedWithLocators(boolean hasLocator) {
    stmt_.setAssociatedWithLocators(hasLocator);
  }
  
  boolean isAssociatedWithLocators() { 
    return stmt_.isAssociatedWithLocators(); 
  }
  
  
  public boolean isClosed() throws SQLException {
    return stmt_.isClosed(); 
  }

  void markCursorClosed(boolean isRollback) throws SQLException {
    stmt_.markCursorClosed(isRollback);
  }
  
  void notifyClose () throws SQLException   {
    stmt_.notifyClose(); 
 }
  
  void postWarning(SQLWarning sqlWarning) { 
    stmt_.postWarning(sqlWarning);
  }

  public void setCursorName(String name) throws SQLException {
    stmt_.setCursorName(name); 
  }

  public void setEscapeProcessing(boolean enable) throws SQLException {
    stmt_.setEscapeProcessing(enable);
  }

  public void setFetchDirection(int direction) throws SQLException {
    stmt_.setFetchDirection(direction); 
  }

  public void setFetchSize(int rows) throws SQLException {
    stmt_.setFetchSize(rows);
  }

  public void setMaxFieldSize(int max) throws SQLException {
    stmt_.setMaxFieldSize(max); 
  }

  public void setMaxRows(int max) throws SQLException {
    stmt_.setMaxRows(max); 
  }

  public void setQueryTimeout(int seconds) throws SQLException {
    stmt_.setQueryTimeout(seconds); 
  }

  public String toString() {
    return stmt_.toString(); 
  }

  /* Note: verifyGeneratedKeysParameter not needed */ 
  
  
  public void setPoolable(boolean poolable) throws SQLException {
    stmt_.setPoolable(poolable); 
  }

  public boolean isPoolable() throws SQLException {
    return stmt_.isPoolable(); 
  }

  protected java.lang.String[] getValidWrappedList() {
    return stmt_.getValidWrappedList(); 
  }
  
  public void closeOnCompletion() throws SQLException {
    ((AS400JDBCStatement)stmt_).closeOnCompletion(); 
  }

  public boolean isCloseOnCompletion() throws SQLException {
    return ((AS400JDBCStatement)stmt_).isCloseOnCompletion(); 
  }

  protected void startCancelThread() {
    stmt_.startCancelThread(); 
  }

  protected void endCancelThread() {
    stmt_.endCancelThread(); 
  }

  public   long getLargeUpdateCount() throws SQLException {
    return stmt_.getLargeUpdateCount();  
  }

  public void setLargeMaxRows(long max) throws SQLException {
    stmt_.setLargeMaxRows(max); 
  }

  public long getLargeMaxRows() throws SQLException {
    return stmt_.getLargeMaxRows(); 
  }


      
    
    public long[] executeLargeBatch()    throws SQLException  {
      return stmt_.executeLargeBatch(); 
    }

   public long executeLargeUpdate(String sql)   throws SQLException {
     
     int retryCount = AS400JDBCConnectionRedirect.SEAMLESS_RETRY_COUNT; 
     while(true) {
       
       try {
         return stmt_.executeLargeUpdate(sql); 
       } catch (AS400JDBCTransientException e) {
         if (connection_.canSeamlessFailover()) {
           retryCount--; 
           if (retryCount >= 0 ) {
             // Keep retrying
           } else {
             throw e; 
           }
         } else {
           throw e; 
         }
       }
     } /* retry */

   }


    public long executeLargeUpdate(String sql,
                                    int autoGeneratedKeys)
                             throws SQLException {
      
      int retryCount = AS400JDBCConnectionRedirect.SEAMLESS_RETRY_COUNT; 
      while(true) {
        
        try {
          return stmt_.executeLargeUpdate(sql, autoGeneratedKeys); 
        } catch (AS400JDBCTransientException e) {
          if (connection_.canSeamlessFailover()) {
            retryCount--; 
            if (retryCount >= 0 ) {
              // Keep retrying
            } else {
              throw e; 
            }
          } else {
            throw e; 
          }
        }
      } /* retry */
    }


    public long executeLargeUpdate(String sql,
                                    int[] columnIndexes) 
                             throws SQLException {

      
      int retryCount = AS400JDBCConnectionRedirect.SEAMLESS_RETRY_COUNT; 
      while(true) {
        
        try {
          return stmt_.executeLargeUpdate(sql, columnIndexes); 
        } catch (AS400JDBCTransientException e) {
          if (connection_.canSeamlessFailover()) {
            retryCount--; 
            if (retryCount >= 0 ) {
              // Keep retrying
            } else {
              throw e; 
            }
          } else {
            throw e; 
          }
        }
      } /* retry */

    
    }

    public long executeLargeUpdate(String sql,
                                    String[] columnNames)  throws SQLException {
      
      int retryCount = AS400JDBCConnectionRedirect.SEAMLESS_RETRY_COUNT; 
      while(true) {
        
        try {
          return stmt_.executeLargeUpdate(sql, columnNames); 
        } catch (AS400JDBCTransientException e) {
          if (connection_.canSeamlessFailover()) {
            retryCount--; 
            if (retryCount >= 0 ) {
              // Keep retrying
            } else {
              throw e; 
            }
          } else {
            throw e; 
          }
        }
      } /* retry */
      
    }




    JDSQLStatement getJDSQLStatement() {
      return stmt_.getJDSQLStatement(); 
    }
    
    void setConnectionReset(boolean reset) {
      stmt_.setConnectionReset(reset);
    }
   
    JDCursor getCursor() {
      return stmt_.getCursor(); 
    }
    
    AS400JDBCStatementLock getInternalLock() {
      return stmt_.getInternalLock(); 
    }

    
    int getInternalResultSetHoldability() {
      return stmt_.getInternalResultSetHoldability(); 
    }

    boolean isQueryRunning() {
      return stmt_.isQueryRunning();  
    }

  int getInternalQueryTimeout() {

    return stmt_.getInternalQueryTimeout();

  }
    
  // 
  // Note:  If you add methods here, they also need to be added to 
  // AS400JDBCPreparedStatementRedirect.
  // 
  
  
}
