/*
 * Decompiled with CFR 0.152.
 */
package org.openmetadata.service.secrets;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.openmetadata.schema.EntityInterface;
import org.openmetadata.schema.ServiceConnectionEntityInterface;
import org.openmetadata.schema.ServiceEntityInterface;
import org.openmetadata.schema.entity.services.ingestionPipelines.IngestionPipeline;
import org.openmetadata.schema.entity.services.ingestionPipelines.PipelineType;
import org.openmetadata.schema.entity.teams.User;
import org.openmetadata.service.Entity;
import org.openmetadata.service.exception.SecretsManagerMigrationException;
import org.openmetadata.service.jdbi3.ChangeEventRepository;
import org.openmetadata.service.jdbi3.IngestionPipelineRepository;
import org.openmetadata.service.jdbi3.ListFilter;
import org.openmetadata.service.jdbi3.ServiceEntityRepository;
import org.openmetadata.service.jdbi3.UserRepository;
import org.openmetadata.service.resources.CollectionRegistry;
import org.openmetadata.service.resources.events.EventResource;
import org.openmetadata.service.resources.services.ServiceEntityResource;
import org.openmetadata.service.resources.services.ingestionpipelines.IngestionPipelineResource;
import org.openmetadata.service.resources.teams.UserResource;
import org.openmetadata.service.secrets.SecretsManager;
import org.openmetadata.service.secrets.SecretsManagerFactory;
import org.openmetadata.service.util.EntityUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SecretsManagerMigrationService {
    private static final Logger LOG = LoggerFactory.getLogger(SecretsManagerMigrationService.class);
    private final SecretsManager newSecretManager;
    private final SecretsManager oldSecretManager;
    private final Map<Class<? extends ServiceConnectionEntityInterface>, ServiceEntityRepository<?, ?>> connectionTypeRepositoriesMap;
    private final ChangeEventRepository changeEventRepository;
    private final IngestionPipelineRepository ingestionPipelineRepository;
    private final UserRepository userRepository;

    public SecretsManagerMigrationService(SecretsManager secretsManager, String clusterName) {
        this.newSecretManager = secretsManager;
        this.connectionTypeRepositoriesMap = this.retrieveConnectionTypeRepositoriesMap();
        this.changeEventRepository = this.retrieveChangeEventRepository();
        this.ingestionPipelineRepository = this.retrieveIngestionPipelineRepository();
        this.userRepository = this.retrieveUserRepository();
        this.oldSecretManager = SecretsManagerFactory.createSecretsManager(null, clusterName);
    }

    public void migrateServicesToSecretManagerIfNeeded() {
        if (!this.newSecretManager.isLocal()) {
            this.migrateServices();
            this.migrateIngestionPipelines();
            this.migrateBotUsersCredentials();
        } else {
            LOG.info("Local secrets manager does not need to check if migration is needed.");
        }
    }

    private void migrateBotUsersCredentials() {
        LOG.info(String.format("Checking if bot users credentials migration is needed for secrets manager: [%s]", this.newSecretManager.getSecretsManagerProvider().value()));
        List<User> notStoredUsers = this.retrieveNotStoredUsers();
        if (!notStoredUsers.isEmpty()) {
            notStoredUsers.forEach(this::migrateBotUser);
            this.deleteChangeEventsFor("user");
        } else {
            LOG.info(String.format("All bot users credentials are already safely stored in [%s] secrets manager", this.newSecretManager.getSecretsManagerProvider().value()));
        }
    }

    private void migrateServices() {
        LOG.info(String.format("Checking if services migration is needed for secrets manager: [%s]", this.newSecretManager.getSecretsManagerProvider().value()));
        List<ServiceEntityInterface> notStoredServices = this.retrieveNotStoredServices();
        if (!notStoredServices.isEmpty()) {
            notStoredServices.forEach(this::migrateService);
            this.deleteChangeEventsForServices();
        } else {
            LOG.info(String.format("All services are already safely stored in [%s] secrets manager", this.newSecretManager.getSecretsManagerProvider().value()));
        }
    }

    private void migrateIngestionPipelines() {
        LOG.info(String.format("Checking if ingestion pipelines migration is needed for secrets manager: [%s]", this.newSecretManager.getSecretsManagerProvider().value()));
        List<IngestionPipeline> notStoredIngestionPipelines = this.retrieveNotStoredIngestionPipelines();
        if (!notStoredIngestionPipelines.isEmpty()) {
            notStoredIngestionPipelines.forEach(this::migrateIngestionPipelines);
            this.deleteChangeEventsFor("ingestionPipeline");
        } else {
            LOG.info(String.format("All ingestion pipelines are already safely stored in [%s] secrets manager", this.newSecretManager.getSecretsManagerProvider().value()));
        }
    }

    private void migrateService(ServiceEntityInterface serviceEntityInterface) {
        ServiceEntityRepository<?, ?> repository = this.connectionTypeRepositoriesMap.get(serviceEntityInterface.getConnection().getClass());
        try {
            ServiceEntityInterface service = (ServiceEntityInterface)repository.dao.findEntityById(serviceEntityInterface.getId());
            service.getConnection().setConfig(this.oldSecretManager.encryptOrDecryptServiceConnectionConfig(service.getConnection().getConfig(), service.getServiceType().value(), service.getName(), repository.getServiceType(), false));
            service.getConnection().setConfig(this.newSecretManager.encryptOrDecryptServiceConnectionConfig(service.getConnection().getConfig(), service.getServiceType().value(), service.getName(), repository.getServiceType(), true));
            Thread.sleep(100L);
            repository.dao.update((EntityInterface)service);
        }
        catch (IOException | InterruptedException e) {
            throw new SecretsManagerMigrationException(e.getMessage(), e.getCause());
        }
    }

    private List<ServiceEntityInterface> retrieveNotStoredServices() {
        return this.connectionTypeRepositoriesMap.values().stream().map(this::retrieveServices).flatMap(Collection::stream).collect(Collectors.toList());
    }

    private List<ServiceEntityInterface> retrieveServices(ServiceEntityRepository<?, ?> serviceEntityRepository) {
        try {
            return serviceEntityRepository.listAfter(null, EntityUtil.Fields.EMPTY_FIELDS, new ListFilter(), serviceEntityRepository.dao.listCount(new ListFilter()), null).getData().stream().map(ServiceEntityInterface.class::cast).filter(service -> !Objects.isNull(service.getConnection()) && !Objects.isNull(service.getConnection().getConfig())).collect(Collectors.toList());
        }
        catch (IOException e) {
            throw new SecretsManagerMigrationException(e.getMessage(), e.getCause());
        }
    }

    private void migrateIngestionPipelines(IngestionPipeline ingestionPipeline) {
        try {
            IngestionPipeline ingestion = (IngestionPipeline)this.ingestionPipelineRepository.dao.findEntityById(ingestionPipeline.getId());
            if (this.hasSecurityConfig(ingestionPipeline)) {
                ingestion.getOpenMetadataServerConnection().setSecurityConfig(null);
            }
            if (this.hasDbtConfig(ingestionPipeline)) {
                this.oldSecretManager.encryptOrDecryptDbtConfigSource(ingestionPipeline, false);
                this.newSecretManager.encryptOrDecryptDbtConfigSource(ingestionPipeline, true);
            }
            this.ingestionPipelineRepository.dao.update((EntityInterface)ingestion);
        }
        catch (IOException e) {
            throw new SecretsManagerMigrationException(e.getMessage(), e.getCause());
        }
    }

    private List<IngestionPipeline> retrieveNotStoredIngestionPipelines() {
        try {
            return this.ingestionPipelineRepository.listAfter(null, EntityUtil.Fields.EMPTY_FIELDS, new ListFilter(), this.ingestionPipelineRepository.dao.listCount(new ListFilter()), null).getData().stream().filter(ingestionPipeline -> this.hasSecurityConfig((IngestionPipeline)ingestionPipeline) || this.hasDbtConfig((IngestionPipeline)ingestionPipeline)).collect(Collectors.toList());
        }
        catch (IOException e) {
            throw new SecretsManagerMigrationException(e.getMessage(), e.getCause());
        }
    }

    private void migrateBotUser(User botUser) {
        try {
            User user = (User)this.userRepository.dao.findEntityById(botUser.getId());
            Object authConfig = this.oldSecretManager.encryptOrDecryptBotUserCredentials(botUser.getName(), user.getAuthenticationMechanism().getConfig(), false);
            authConfig = this.newSecretManager.encryptOrDecryptBotUserCredentials(botUser.getName(), authConfig, true);
            user.getAuthenticationMechanism().setConfig(authConfig);
            this.userRepository.dao.update((EntityInterface)user);
        }
        catch (IOException e) {
            throw new SecretsManagerMigrationException(e.getMessage(), e.getCause());
        }
    }

    private List<User> retrieveNotStoredUsers() {
        try {
            return this.userRepository.listAfter(null, new EntityUtil.Fields(List.of("authenticationMechanism")), new ListFilter(), this.userRepository.dao.listCount(new ListFilter()), null).getData().stream().filter(this::isBotWithCredentials).collect(Collectors.toList());
        }
        catch (IOException e) {
            throw new SecretsManagerMigrationException(e.getMessage(), e.getCause());
        }
    }

    private boolean isBotWithCredentials(User user) {
        return Boolean.TRUE.equals(user.getIsBot()) && user.getAuthenticationMechanism() != null && user.getAuthenticationMechanism().getConfig() != null;
    }

    private boolean hasSecurityConfig(IngestionPipeline ingestionPipeline) {
        return !Objects.isNull(ingestionPipeline.getOpenMetadataServerConnection()) && !Objects.isNull(ingestionPipeline.getOpenMetadataServerConnection().getSecurityConfig());
    }

    private boolean hasDbtConfig(IngestionPipeline ingestionPipeline) {
        return ingestionPipeline.getService().getType().equals("databaseService") && ingestionPipeline.getPipelineType().equals((Object)PipelineType.METADATA);
    }

    private void deleteChangeEventsForServices() {
        this.connectionTypeRepositoriesMap.values().stream().map(ServiceEntityRepository::getServiceType).forEach(serviceType -> {
            try {
                this.deleteChangeEventsFor(Entity.class.getField(serviceType.value().toUpperCase(Locale.ROOT) + "_SERVICE").get(Entity.class).toString());
            }
            catch (IllegalAccessException | NoSuchFieldException e) {
                throw new SecretsManagerMigrationException(e.getMessage(), e.getCause());
            }
        });
    }

    private void deleteChangeEventsFor(String entityType) {
        this.changeEventRepository.deleteAll(entityType);
    }

    private Map<Class<? extends ServiceConnectionEntityInterface>, ServiceEntityRepository<?, ?>> retrieveConnectionTypeRepositoriesMap() {
        Map<Class<ServiceConnectionEntityInterface>, ServiceEntityRepository<?, ?>> connectionTypeRepositoriesMap = CollectionRegistry.getInstance().getCollectionMap().values().stream().map(this::retrieveServiceRepository).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toMap(ServiceEntityRepository::getServiceConnectionClass, Function.identity()));
        if (connectionTypeRepositoriesMap.isEmpty()) {
            throw new SecretsManagerMigrationException("Unexpected error: ServiceRepository not found.");
        }
        return connectionTypeRepositoriesMap;
    }

    private ChangeEventRepository retrieveChangeEventRepository() {
        return CollectionRegistry.getInstance().getCollectionMap().values().stream().map(collectionDetails -> this.retrieveResource((CollectionRegistry.CollectionDetails)collectionDetails, (Class)EventResource.class)).filter(Optional::isPresent).map(res -> ((EventResource)res.get()).getDao()).findFirst().orElseThrow(() -> new SecretsManagerMigrationException("Unexpected error: ChangeEventRepository not found."));
    }

    private UserRepository retrieveUserRepository() {
        return CollectionRegistry.getInstance().getCollectionMap().values().stream().map(collectionDetails -> this.retrieveResource((CollectionRegistry.CollectionDetails)collectionDetails, (Class)UserResource.class)).filter(Optional::isPresent).map(res -> ((UserResource)res.get()).getUserRepository()).findFirst().orElseThrow(() -> new SecretsManagerMigrationException("Unexpected error: IngestionPipelineRepository not found."));
    }

    private IngestionPipelineRepository retrieveIngestionPipelineRepository() {
        return CollectionRegistry.getInstance().getCollectionMap().values().stream().map(collectionDetails -> this.retrieveResource((CollectionRegistry.CollectionDetails)collectionDetails, (Class)IngestionPipelineResource.class)).filter(Optional::isPresent).map(res -> ((IngestionPipelineResource)res.get()).getIngestionPipelineRepository()).findFirst().orElseThrow(() -> new SecretsManagerMigrationException("Unexpected error: IngestionPipelineRepository not found."));
    }

    private <T> Optional<T> retrieveResource(CollectionRegistry.CollectionDetails collectionDetails, Class<T> clazz) {
        Class<?> collectionDetailsClass = this.extractCollectionDetailsClass(collectionDetails);
        if (clazz.equals(collectionDetailsClass)) {
            return Optional.of(clazz.cast(collectionDetails.getResource()));
        }
        return Optional.empty();
    }

    private Optional<ServiceEntityRepository<?, ?>> retrieveServiceRepository(CollectionRegistry.CollectionDetails collectionDetails) {
        Class<?> collectionDetailsClass = this.extractCollectionDetailsClass(collectionDetails);
        if (ServiceEntityResource.class.isAssignableFrom(collectionDetailsClass)) {
            return Optional.of(((ServiceEntityResource)collectionDetails.getResource()).getServiceEntityRepository());
        }
        return Optional.empty();
    }

    private Class<?> extractCollectionDetailsClass(CollectionRegistry.CollectionDetails collectionDetails) {
        Class<?> collectionDetailsClass;
        try {
            collectionDetailsClass = Class.forName(collectionDetails.getResourceClass());
        }
        catch (ClassNotFoundException e) {
            throw new SecretsManagerMigrationException(e.getMessage(), e.getCause());
        }
        return collectionDetailsClass;
    }
}

