/*
 * Decompiled with CFR 0.152.
 */
package org.openmetadata.service.elasticsearch;

import java.io.IOException;
import java.io.InputStream;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Date;
import java.util.EnumMap;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequest;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.client.indices.CreateIndexRequest;
import org.elasticsearch.client.indices.CreateIndexResponse;
import org.elasticsearch.client.indices.GetIndexRequest;
import org.elasticsearch.client.indices.PutMappingRequest;
import org.elasticsearch.common.xcontent.XContentType;
import org.openmetadata.schema.settings.EventPublisherJob;
import org.openmetadata.schema.settings.FailureDetails;
import org.openmetadata.service.jdbi3.CollectionDAO;
import org.openmetadata.service.util.JsonUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ElasticSearchIndexDefinition {
    private static final Logger LOG = LoggerFactory.getLogger(ElasticSearchIndexDefinition.class);
    private static final String REASON_TRACE = "Reason: [%s] , Trace : [%s]";
    public static final String ENTITY_REPORT_DATA = "entityReportData";
    public static final String WEB_ANALYTIC_ENTITY_VIEW_REPORT_DATA = "webAnalyticEntityViewReportData";
    public static final String WEB_ANALYTIC_USER_ACTIVITY_REPORT_DATA = "webAnalyticUserActivityReportData";
    private final CollectionDAO dao;
    final EnumMap<ElasticSearchIndexType, ElasticSearchIndexStatus> elasticSearchIndexes = new EnumMap(ElasticSearchIndexType.class);
    private final RestHighLevelClient client;

    public ElasticSearchIndexDefinition(RestHighLevelClient client, CollectionDAO dao) {
        this.dao = dao;
        this.client = client;
        for (ElasticSearchIndexType elasticSearchIndexType : ElasticSearchIndexType.values()) {
            this.elasticSearchIndexes.put(elasticSearchIndexType, ElasticSearchIndexStatus.NOT_CREATED);
        }
    }

    public void createIndexes() {
        for (ElasticSearchIndexType elasticSearchIndexType : ElasticSearchIndexType.values()) {
            this.createIndex(elasticSearchIndexType);
        }
    }

    public void updateIndexes() {
        for (ElasticSearchIndexType elasticSearchIndexType : ElasticSearchIndexType.values()) {
            this.updateIndex(elasticSearchIndexType);
        }
    }

    public void dropIndexes() {
        for (ElasticSearchIndexType elasticSearchIndexType : ElasticSearchIndexType.values()) {
            this.deleteIndex(elasticSearchIndexType);
        }
    }

    public boolean createIndex(ElasticSearchIndexType elasticSearchIndexType) {
        try {
            GetIndexRequest gRequest = new GetIndexRequest(new String[]{elasticSearchIndexType.indexName});
            gRequest.local(false);
            boolean exists = this.client.indices().exists(gRequest, RequestOptions.DEFAULT);
            if (!exists) {
                String elasticSearchIndexMapping = this.getIndexMapping(elasticSearchIndexType);
                CreateIndexRequest request = new CreateIndexRequest(elasticSearchIndexType.indexName);
                request.source(elasticSearchIndexMapping, XContentType.JSON);
                CreateIndexResponse createIndexResponse = this.client.indices().create(request, RequestOptions.DEFAULT);
                LOG.info("{} Created {}", (Object)elasticSearchIndexType.indexName, (Object)createIndexResponse.isAcknowledged());
            }
            this.setIndexStatus(elasticSearchIndexType, ElasticSearchIndexStatus.CREATED);
        }
        catch (Exception e) {
            this.setIndexStatus(elasticSearchIndexType, ElasticSearchIndexStatus.FAILED);
            this.updateElasticSearchFailureStatus(this.getContext("Creating Index", elasticSearchIndexType.indexName), String.format(REASON_TRACE, e.getMessage(), ExceptionUtils.getStackTrace((Throwable)e)));
            LOG.error("Failed to create Elastic Search indexes due to", (Throwable)e);
            return false;
        }
        return true;
    }

    private String getContext(String type, String info) {
        return String.format("Failed While : %s \n Additional Info:  %s ", type, info);
    }

    private void updateIndex(ElasticSearchIndexType elasticSearchIndexType) {
        try {
            GetIndexRequest gRequest = new GetIndexRequest(new String[]{elasticSearchIndexType.indexName});
            gRequest.local(false);
            boolean exists = this.client.indices().exists(gRequest, RequestOptions.DEFAULT);
            String elasticSearchIndexMapping = this.getIndexMapping(elasticSearchIndexType);
            if (exists) {
                PutMappingRequest request = new PutMappingRequest(new String[]{elasticSearchIndexType.indexName});
                request.source(elasticSearchIndexMapping, XContentType.JSON);
                AcknowledgedResponse putMappingResponse = this.client.indices().putMapping(request, RequestOptions.DEFAULT);
                LOG.info("{} Updated {}", (Object)elasticSearchIndexType.indexName, (Object)putMappingResponse.isAcknowledged());
            } else {
                CreateIndexRequest request = new CreateIndexRequest(elasticSearchIndexType.indexName);
                request.source(elasticSearchIndexMapping, XContentType.JSON);
                CreateIndexResponse createIndexResponse = this.client.indices().create(request, RequestOptions.DEFAULT);
                LOG.info("{} Created {}", (Object)elasticSearchIndexType.indexName, (Object)createIndexResponse.isAcknowledged());
            }
            this.setIndexStatus(elasticSearchIndexType, ElasticSearchIndexStatus.CREATED);
        }
        catch (Exception e) {
            this.setIndexStatus(elasticSearchIndexType, ElasticSearchIndexStatus.FAILED);
            this.updateElasticSearchFailureStatus(this.getContext("Updating Index", elasticSearchIndexType.indexName), String.format(REASON_TRACE, e.getMessage(), ExceptionUtils.getStackTrace((Throwable)e)));
            LOG.error("Failed to update Elastic Search indexes due to", (Throwable)e);
        }
    }

    public void deleteIndex(ElasticSearchIndexType elasticSearchIndexType) {
        try {
            GetIndexRequest gRequest = new GetIndexRequest(new String[]{elasticSearchIndexType.indexName});
            gRequest.local(false);
            boolean exists = this.client.indices().exists(gRequest, RequestOptions.DEFAULT);
            if (exists) {
                DeleteIndexRequest request = new DeleteIndexRequest(elasticSearchIndexType.indexName);
                AcknowledgedResponse deleteIndexResponse = this.client.indices().delete(request, RequestOptions.DEFAULT);
                LOG.info("{} Deleted {}", (Object)elasticSearchIndexType.indexName, (Object)deleteIndexResponse.isAcknowledged());
            }
        }
        catch (IOException e) {
            this.updateElasticSearchFailureStatus(this.getContext("Deleting Index", elasticSearchIndexType.indexName), String.format(REASON_TRACE, e.getMessage(), ExceptionUtils.getStackTrace((Throwable)e)));
            LOG.error("Failed to delete Elastic Search indexes due to", (Throwable)e);
        }
    }

    private void setIndexStatus(ElasticSearchIndexType indexType, ElasticSearchIndexStatus elasticSearchIndexStatus) {
        this.elasticSearchIndexes.put(indexType, elasticSearchIndexStatus);
    }

    public String getIndexMapping(ElasticSearchIndexType elasticSearchIndexType) throws IOException {
        InputStream in = ElasticSearchIndexDefinition.class.getResourceAsStream(elasticSearchIndexType.indexMappingFile);
        return new String(in.readAllBytes());
    }

    public static ElasticSearchIndexType getIndexMappingByEntityType(String type) {
        if (type.equalsIgnoreCase("table")) {
            return ElasticSearchIndexType.TABLE_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("dashboard")) {
            return ElasticSearchIndexType.DASHBOARD_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("pipeline")) {
            return ElasticSearchIndexType.PIPELINE_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("topic")) {
            return ElasticSearchIndexType.TOPIC_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("user")) {
            return ElasticSearchIndexType.USER_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("team")) {
            return ElasticSearchIndexType.TEAM_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("glossary")) {
            return ElasticSearchIndexType.GLOSSARY_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("mlmodel")) {
            return ElasticSearchIndexType.MLMODEL_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("glossaryTerm")) {
            return ElasticSearchIndexType.GLOSSARY_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("tag")) {
            return ElasticSearchIndexType.TAG_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase(ENTITY_REPORT_DATA)) {
            return ElasticSearchIndexType.ENTITY_REPORT_DATA_INDEX;
        }
        if (type.equalsIgnoreCase(WEB_ANALYTIC_ENTITY_VIEW_REPORT_DATA)) {
            return ElasticSearchIndexType.WEB_ANALYTIC_ENTITY_VIEW_REPORT_DATA_INDEX;
        }
        if (type.equalsIgnoreCase(WEB_ANALYTIC_USER_ACTIVITY_REPORT_DATA)) {
            return ElasticSearchIndexType.WEB_ANALYTIC_USER_ACTIVITY_REPORT_DATA_INDEX;
        }
        throw new RuntimeException("Failed to find index doc for type " + type);
    }

    private void updateElasticSearchFailureStatus(String failedFor, String failureMessage) {
        try {
            long updateTime = Date.from(LocalDateTime.now().atZone(ZoneId.systemDefault()).toInstant()).getTime();
            String recordString = this.dao.entityExtensionTimeSeriesDao().getExtension("eventPublisher:ElasticSearch:STREAM", "service.eventPublisher");
            EventPublisherJob lastRecord = JsonUtils.readValue(recordString, EventPublisherJob.class);
            long originalLastUpdate = lastRecord.getTimestamp();
            lastRecord.setStatus(EventPublisherJob.Status.ACTIVE_WITH_ERROR);
            lastRecord.setTimestamp(Long.valueOf(updateTime));
            lastRecord.setFailureDetails(new FailureDetails().withContext(failedFor).withLastFailedAt(Long.valueOf(updateTime)).withLastFailedReason(failureMessage));
            this.dao.entityExtensionTimeSeriesDao().update("eventPublisher:ElasticSearch:STREAM", "service.eventPublisher", JsonUtils.pojoToJson(lastRecord), originalLastUpdate);
        }
        catch (Exception e) {
            LOG.error("Failed to Update Elastic Search Job Info");
        }
    }

    public static enum ElasticSearchIndexType {
        TABLE_SEARCH_INDEX("table_search_index", "/elasticsearch/table_index_mapping.json"),
        TOPIC_SEARCH_INDEX("topic_search_index", "/elasticsearch/topic_index_mapping.json"),
        DASHBOARD_SEARCH_INDEX("dashboard_search_index", "/elasticsearch/dashboard_index_mapping.json"),
        PIPELINE_SEARCH_INDEX("pipeline_search_index", "/elasticsearch/pipeline_index_mapping.json"),
        USER_SEARCH_INDEX("user_search_index", "/elasticsearch/user_index_mapping.json"),
        TEAM_SEARCH_INDEX("team_search_index", "/elasticsearch/team_index_mapping.json"),
        GLOSSARY_SEARCH_INDEX("glossary_search_index", "/elasticsearch/glossary_index_mapping.json"),
        MLMODEL_SEARCH_INDEX("mlmodel_search_index", "/elasticsearch/mlmodel_index_mapping.json"),
        TAG_SEARCH_INDEX("tag_search_index", "/elasticsearch/tag_index_mapping.json"),
        ENTITY_REPORT_DATA_INDEX("entity_report_data_index", "/elasticsearch/entity_report_data_index.json"),
        WEB_ANALYTIC_ENTITY_VIEW_REPORT_DATA_INDEX("web_analytic_entity_view_report_data_index", "/elasticsearch/web_analytic_entity_view_report_data_index.json"),
        WEB_ANALYTIC_USER_ACTIVITY_REPORT_DATA_INDEX("web_analytic_user_activity_report_data_index", "/elasticsearch/web_analytic_user_activity_report_data_index.json");

        public final String indexName;
        public final String indexMappingFile;

        private ElasticSearchIndexType(String indexName, String indexMappingFile) {
            this.indexName = indexName;
            this.indexMappingFile = indexMappingFile;
        }
    }

    public static enum ElasticSearchIndexStatus {
        CREATED,
        NOT_CREATED,
        FAILED;

    }
}

