/*
 * Decompiled with CFR 0.152.
 */
package org.openmetadata.service.migration.api;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import org.jdbi.v3.core.Handle;
import org.jdbi.v3.core.Jdbi;
import org.openmetadata.service.jdbi3.MigrationDAO;
import org.openmetadata.service.migration.api.MigrationStep;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MigrationWorkflow {
    private static final Logger LOG = LoggerFactory.getLogger(MigrationWorkflow.class);
    private final List<MigrationStep> migrations;
    private final MigrationDAO migrationDAO;
    private final Jdbi jdbi;
    private boolean ignoreFileChecksum = false;

    public MigrationWorkflow(Jdbi jdbi, List<MigrationStep> migrationSteps, boolean ignoreFileChecksum) {
        this.jdbi = jdbi;
        this.migrationDAO = (MigrationDAO)jdbi.onDemand(MigrationDAO.class);
        this.ignoreFileChecksum = ignoreFileChecksum;
        migrationSteps.sort(Comparator.comparing(MigrationStep::getMigrationVersion));
        this.migrations = this.filterAndGetMigrationsToRun(migrationSteps);
    }

    private List<MigrationStep> filterAndGetMigrationsToRun(List<MigrationStep> migrations) {
        LOG.debug("Filtering Server Migrations");
        String maxMigration = migrations.get(migrations.size() - 1).getMigrationVersion();
        ArrayList<MigrationStep> result = new ArrayList<MigrationStep>();
        for (MigrationStep step : migrations) {
            String checksum = this.migrationDAO.getVersionMigrationChecksum(String.valueOf(step.getMigrationVersion()));
            if (checksum != null) {
                if (maxMigration.compareTo(step.getMigrationVersion()) < 0) {
                    result.add(step);
                    continue;
                }
                if (!this.ignoreFileChecksum && checksum.equals(step.getFileUuid())) continue;
                LOG.warn("[Migration Workflow] You are changing an older Migration File. This is not advised. Add your changes to latest Migrations.");
                result.add(step);
                continue;
            }
            result.add(step);
        }
        return result;
    }

    private void initializeMigrationWorkflow() {
    }

    public void runMigrationWorkflows() {
        try (Handle transactionHandler = this.jdbi.open();){
            LOG.info("[MigrationWorkflow] WorkFlow Started");
            try {
                for (MigrationStep step : this.migrations) {
                    LOG.info("[MigrationStep] Initialized, Version: {}, DatabaseType: {}, FileName: {}", new Object[]{step.getMigrationVersion(), step.getDatabaseConnectionType(), step.getMigrationFileName()});
                    step.initialize(transactionHandler);
                    LOG.info("[MigrationStep] Running PreDataSQLs, Version: {}, DatabaseType: {}, FileName: {}", new Object[]{step.getMigrationVersion(), step.getDatabaseConnectionType(), step.getMigrationFileName()});
                    step.preDDL();
                    LOG.info("[MigrationStep] Transaction Started");
                    transactionHandler.begin();
                    LOG.info("[MigrationStep] Running DataMigration, Version: {}, DatabaseType: {}, FileName: {}", new Object[]{step.getMigrationVersion(), step.getDatabaseConnectionType(), step.getMigrationFileName()});
                    step.runDataMigration();
                    LOG.info("[MigrationStep] Committing Transaction");
                    transactionHandler.commit();
                    LOG.info("[MigrationStep] Running TransactionalPostDataSQLs, Version: {}, DatabaseType: {}, FileName: {}", new Object[]{step.getMigrationVersion(), step.getDatabaseConnectionType(), step.getMigrationFileName()});
                    step.postDDL();
                    LOG.info("[MigrationStep] Update Migration Status, Version: {}, DatabaseType: {}, FileName: {}", new Object[]{step.getMigrationVersion(), step.getDatabaseConnectionType(), step.getMigrationFileName()});
                    this.updateMigrationStepInDB(step);
                }
            }
            catch (Exception e) {
                LOG.error("Encountered Exception in MigrationWorkflow", (Throwable)e);
                LOG.info("[MigrationWorkflow] Rolling Back Transaction");
                if (transactionHandler.isInTransaction()) {
                    transactionHandler.rollback();
                }
            }
        }
        LOG.info("[MigrationWorkflow] WorkFlow Completed");
    }

    public void closeMigrationWorkflow() {
    }

    public void updateMigrationStepInDB(MigrationStep step) {
        this.migrationDAO.upsertServerMigration(step.getMigrationVersion(), step.getMigrationFileName(), step.getFileUuid());
    }

    public void migrateSearchIndexes() {
    }
}

