/*
 * Decompiled with CFR 0.152.
 */
package org.openmetadata.service.util;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import freemarker.template.TemplateException;
import java.io.IOException;
import java.net.URI;
import java.time.Instant;
import java.util.HashSet;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import javax.ws.rs.container.ContainerResponseContext;
import javax.ws.rs.core.Response;
import org.openmetadata.schema.entity.feed.Thread;
import org.openmetadata.schema.entity.teams.Team;
import org.openmetadata.schema.entity.teams.User;
import org.openmetadata.schema.type.AnnouncementDetails;
import org.openmetadata.schema.type.Post;
import org.openmetadata.schema.type.Relationship;
import org.openmetadata.schema.utils.EntityInterfaceUtil;
import org.openmetadata.service.Entity;
import org.openmetadata.service.jdbi3.CollectionDAO;
import org.openmetadata.service.jdbi3.UserRepository;
import org.openmetadata.service.resources.feeds.MessageParser;
import org.openmetadata.service.socket.WebSocketManager;
import org.openmetadata.service.util.EmailUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NotificationHandler {
    private static final Logger LOG = LoggerFactory.getLogger(NotificationHandler.class);
    private final CollectionDAO dao;
    private final ObjectMapper mapper;
    private final ExecutorService threadScheduler;

    public NotificationHandler(CollectionDAO dao) {
        this.dao = dao;
        this.mapper = new ObjectMapper();
        this.threadScheduler = Executors.newFixedThreadPool(1);
    }

    public void processNotifications(ContainerResponseContext responseContext) {
        this.threadScheduler.submit(() -> {
            try {
                this.handleNotifications(responseContext);
            }
            catch (JsonProcessingException e) {
                LOG.error("[NotificationHandler] Failed to use mapper in converting to Json", (Throwable)e);
            }
        });
    }

    private void handleNotifications(ContainerResponseContext responseContext) throws JsonProcessingException {
        int responseCode = responseContext.getStatus();
        if (responseCode == Response.Status.CREATED.getStatusCode() && responseContext.getEntity() != null && responseContext.getEntity().getClass().equals(Thread.class)) {
            Thread thread = (Thread)responseContext.getEntity();
            switch (thread.getType()) {
                case Task: {
                    this.handleTaskNotification(thread);
                    break;
                }
                case Conversation: {
                    this.handleConversationNotification(thread);
                    break;
                }
                case Announcement: {
                    this.handleAnnouncementNotification(thread);
                }
            }
        }
    }

    private void handleTaskNotification(Thread thread) throws JsonProcessingException {
        String jsonThread = this.mapper.writeValueAsString((Object)thread);
        if (thread.getPostsCount() == 0) {
            List assignees = thread.getTask().getAssignees();
            HashSet<UUID> receiversList = new HashSet<UUID>();
            assignees.forEach(e -> {
                if ("user".equals(e.getType())) {
                    receiversList.add(e.getId());
                } else if ("team".equals(e.getType())) {
                    List<CollectionDAO.EntityRelationshipRecord> records = this.dao.relationshipDAO().findTo(e.getId().toString(), "team", Relationship.HAS.ordinal(), "user");
                    records.forEach(eRecord -> receiversList.add(eRecord.getId()));
                }
            });
            WebSocketManager.getInstance().sendToManyWithUUID(receiversList, "taskChannel", jsonThread);
            this.handleEmailNotifications(receiversList, thread);
        }
    }

    private void handleAnnouncementNotification(Thread thread) throws JsonProcessingException {
        String jsonThread = this.mapper.writeValueAsString((Object)thread);
        AnnouncementDetails announcementDetails = thread.getAnnouncement();
        Long currentTimestamp = Instant.now().getEpochSecond();
        if (announcementDetails.getStartTime() <= currentTimestamp && currentTimestamp <= announcementDetails.getEndTime()) {
            WebSocketManager.getInstance().broadCastMessageToAll("announcementChannel", jsonThread);
        }
    }

    private void handleConversationNotification(Thread thread) throws JsonProcessingException {
        List<MessageParser.EntityLink> mentions;
        String jsonThread = this.mapper.writeValueAsString((Object)thread);
        WebSocketManager.getInstance().broadCastMessageToAll("activityFeed", jsonThread);
        if (thread.getPostsCount() == 0) {
            mentions = MessageParser.getEntityLinks(thread.getMessage());
        } else {
            Post latestPost = (Post)thread.getPosts().get(thread.getPostsCount() - 1);
            mentions = MessageParser.getEntityLinks(latestPost.getMessage());
        }
        mentions.forEach(entityLink -> {
            String fqn = entityLink.getEntityFQN();
            if ("user".equals(entityLink.getEntityType())) {
                User user = (User)this.dao.userDAO().findEntityByName(EntityInterfaceUtil.quoteName((String)fqn));
                WebSocketManager.getInstance().sendToOne(user.getId(), "mentionChannel", jsonThread);
            } else if ("team".equals(entityLink.getEntityType())) {
                Team team = (Team)this.dao.teamDAO().findEntityByName(EntityInterfaceUtil.quoteName((String)fqn));
                List<CollectionDAO.EntityRelationshipRecord> records = this.dao.relationshipDAO().findTo(team.getId().toString(), "team", Relationship.HAS.ordinal(), "user");
                WebSocketManager.getInstance().sendToManyWithString(records, "mentionChannel", jsonThread);
            }
        });
    }

    private void handleEmailNotifications(HashSet<UUID> userList, Thread thread) {
        UserRepository repository = (UserRepository)Entity.getEntityRepository("user");
        URI urlInstance = thread.getHref();
        userList.forEach(id -> {
            try {
                User user = (User)repository.get(null, (UUID)id, repository.getFields("name,email,href"));
                EmailUtil.getInstance().sendTaskAssignmentNotificationToUser(user.getName(), user.getEmail(), String.format("%s/users/%s/tasks", EmailUtil.getInstance().buildBaseUrl(urlInstance), user.getName()), thread, EmailUtil.getInstance().getTaskAssignmentSubject(), "taskAssignment.ftl");
            }
            catch (IOException ex) {
                LOG.error("Task Email Notification Failed :", (Throwable)ex);
            }
            catch (TemplateException ex) {
                LOG.error("Task Email Notification Template Parsing Exception :", (Throwable)ex);
            }
        });
    }
}

