/*
 * Decompiled with CFR 0.152.
 */
package org.openmetadata.service.migration.api;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import org.jdbi.v3.core.Handle;
import org.jdbi.v3.core.Jdbi;
import org.openmetadata.service.jdbi3.MigrationDAO;
import org.openmetadata.service.migration.Migration;
import org.openmetadata.service.migration.api.MigrationStep;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MigrationWorkflow {
    private static final Logger LOG = LoggerFactory.getLogger(MigrationWorkflow.class);
    private final List<MigrationStep> migrations;
    private final MigrationDAO migrationDAO;
    private final Jdbi jdbi;
    private final boolean forceMigrations;

    public MigrationWorkflow(Jdbi jdbi, List<MigrationStep> migrationSteps, boolean forceMigrations) {
        this.jdbi = jdbi;
        this.migrationDAO = (MigrationDAO)jdbi.onDemand(MigrationDAO.class);
        this.forceMigrations = forceMigrations;
        migrationSteps.sort(Comparator.comparing(MigrationStep::getMigrationVersion));
        this.migrations = this.filterAndGetMigrationsToRun(migrationSteps);
    }

    public static void validateMigrationsForServer(Jdbi jdbi, List<MigrationStep> migrations) {
        if (!migrations.isEmpty()) {
            migrations.sort(Comparator.comparing(MigrationStep::getMigrationVersion));
            String maxMigration = migrations.get(migrations.size() - 1).getMigrationVersion();
            Optional<String> lastMigratedServer = Migration.lastMigratedServer(jdbi);
            if (lastMigratedServer.isEmpty()) {
                throw new IllegalStateException("Could not validate Server migrations in the database. Make sure you have run `./bootstrap/bootstrap_storage.sh migrate-all` at least once.");
            }
            if (lastMigratedServer.get().compareTo(maxMigration) < 0) {
                throw new IllegalStateException("There are pending migrations to be run on the database. Please backup your data and run `./bootstrap/bootstrap_storage.sh migrate-all`. You can find more information on upgrading OpenMetadata at https://docs.open-metadata.org/deployment/upgrade ");
            }
        } else {
            LOG.info("No Server Migration Files Found in the system.");
        }
    }

    private List<MigrationStep> filterAndGetMigrationsToRun(List<MigrationStep> migrations) {
        LOG.debug("Filtering Server Migrations");
        String maxMigration = migrations.get(migrations.size() - 1).getMigrationVersion();
        ArrayList<MigrationStep> result = new ArrayList<MigrationStep>();
        for (MigrationStep step : migrations) {
            String checksum = this.migrationDAO.getVersionMigrationChecksum(String.valueOf(step.getMigrationVersion()));
            if (checksum != null) {
                if (maxMigration.compareTo(step.getMigrationVersion()) < 0) {
                    result.add(step);
                    continue;
                }
                if (!this.forceMigrations && checksum.equals(step.getFileUuid())) continue;
                LOG.warn("[Migration Workflow] You are changing an older Migration File. This is not advised. Add your changes to latest Migrations.");
                result.add(step);
                continue;
            }
            result.add(step);
        }
        return result;
    }

    private void initializeMigrationWorkflow() {
    }

    public void runMigrationWorkflows() {
        try (Handle transactionHandler = this.jdbi.open();){
            LOG.info("[MigrationWorkflow] WorkFlow Started");
            try {
                for (MigrationStep step : this.migrations) {
                    LOG.info("[MigrationStep] Initialized, Version: {}, DatabaseType: {}, FileName: {}", new Object[]{step.getMigrationVersion(), step.getDatabaseConnectionType(), step.getMigrationFileName()});
                    step.initialize(transactionHandler);
                    LOG.info("[MigrationStep] Running PreDataSQLs, Version: {}, DatabaseType: {}, FileName: {}", new Object[]{step.getMigrationVersion(), step.getDatabaseConnectionType(), step.getMigrationFileName()});
                    step.preDDL();
                    LOG.info("[MigrationStep] Transaction Started");
                    LOG.info("[MigrationStep] Running DataMigration, Version: {}, DatabaseType: {}, FileName: {}", new Object[]{step.getMigrationVersion(), step.getDatabaseConnectionType(), step.getMigrationFileName()});
                    step.runDataMigration();
                    LOG.info("[MigrationStep] Running TransactionalPostDataSQLs, Version: {}, DatabaseType: {}, FileName: {}", new Object[]{step.getMigrationVersion(), step.getDatabaseConnectionType(), step.getMigrationFileName()});
                    step.postDDL();
                    LOG.info("[MigrationStep] Update Migration Status, Version: {}, DatabaseType: {}, FileName: {}", new Object[]{step.getMigrationVersion(), step.getDatabaseConnectionType(), step.getMigrationFileName()});
                    this.updateMigrationStepInDB(step);
                }
            }
            catch (Exception e) {
                LOG.error("Encountered Exception in MigrationWorkflow", (Throwable)e);
                LOG.info("[MigrationWorkflow] Rolling Back Transaction");
                throw e;
            }
        }
        LOG.info("[MigrationWorkflow] WorkFlow Completed");
    }

    public void closeMigrationWorkflow() {
    }

    public void updateMigrationStepInDB(MigrationStep step) {
        this.migrationDAO.upsertServerMigration(step.getMigrationVersion(), step.getMigrationFileName(), step.getFileUuid());
    }

    public void migrateSearchIndexes() {
    }
}

