/*
 * Decompiled with CFR 0.152.
 */
package org.openmetadata.service.migration.api;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;
import org.jdbi.v3.core.Handle;
import org.jdbi.v3.core.Jdbi;
import org.openmetadata.service.jdbi3.MigrationDAO;
import org.openmetadata.service.jdbi3.locator.ConnectionType;
import org.openmetadata.service.migration.api.MigrationProcess;
import org.openmetadata.service.migration.utils.MigrationFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MigrationWorkflow {
    private static final Logger LOG = LoggerFactory.getLogger(MigrationWorkflow.class);
    private final List<MigrationProcess> migrations;
    private final MigrationDAO migrationDAO;
    private final Jdbi jdbi;
    private final boolean forceMigrations;

    public MigrationWorkflow(Jdbi jdbi, String nativeSQLScriptRootPath, ConnectionType connectionType, boolean forceMigrations) {
        this.jdbi = jdbi;
        this.migrationDAO = (MigrationDAO)jdbi.onDemand(MigrationDAO.class);
        this.forceMigrations = forceMigrations;
        List<MigrationFile> availableMigrations = Arrays.stream(Objects.requireNonNull(new File(nativeSQLScriptRootPath).listFiles(File::isDirectory))).map(dir -> new MigrationFile((File)dir, this.migrationDAO, connectionType)).sorted().collect(Collectors.toList());
        this.migrations = this.filterAndGetMigrationsToRun(availableMigrations);
    }

    public void validateMigrationsForServer() {
        if (!this.migrations.isEmpty()) {
            throw new IllegalStateException("There are pending migrations to be run on the database. Please backup your data and run `./bootstrap/bootstrap_storage.sh migrate-all`. You can find more information on upgrading OpenMetadata at https://docs.open-metadata.org/deployment/upgrade ");
        }
    }

    private List<MigrationProcess> filterAndGetMigrationsToRun(List<MigrationFile> availableMigrations) {
        LOG.debug("Filtering Server Migrations");
        Optional<String> previousMaxMigration = this.migrationDAO.getMaxServerMigrationVersion();
        List<MigrationFile> applyMigrations = previousMaxMigration.isPresent() && !this.forceMigrations ? availableMigrations.stream().filter(migration -> migration.biggerThan((String)previousMaxMigration.get())).collect(Collectors.toList()) : availableMigrations;
        ArrayList<MigrationProcess> processes = new ArrayList<MigrationProcess>();
        try {
            for (MigrationFile file : applyMigrations) {
                file.parseSQLFiles();
                String clazzName = file.getMigrationProcessClassName();
                MigrationProcess process = (MigrationProcess)Class.forName(clazzName).getConstructor(MigrationFile.class).newInstance(file);
                processes.add(process);
            }
        }
        catch (Exception e) {
            LOG.error("Failed to list and add migrations to run due to ", (Throwable)e);
        }
        return processes;
    }

    private void initializeMigrationWorkflow() {
    }

    public void runMigrationWorkflows() {
        try (Handle transactionHandler = this.jdbi.open();){
            LOG.info("[MigrationWorkflow] WorkFlow Started");
            try {
                for (MigrationProcess process : this.migrations) {
                    LOG.info("[MigrationProcess] Initialized, Version: {}, DatabaseType: {}, FileName: {}", new Object[]{process.getVersion(), process.getDatabaseConnectionType(), process.getMigrationsPath()});
                    process.initialize(transactionHandler);
                    LOG.info("[MigrationProcess] Running Schema Changes, Version: {}, DatabaseType: {}, FileName: {}", new Object[]{process.getVersion(), process.getDatabaseConnectionType(), process.getSchemaChangesFilePath()});
                    process.runSchemaChanges();
                    LOG.info("[MigrationStep] Transaction Started");
                    LOG.info("[MigrationProcess] Running Data Migrations, Version: {}, DatabaseType: {}, FileName: {}", new Object[]{process.getVersion(), process.getDatabaseConnectionType(), process.getSchemaChangesFilePath()});
                    process.runDataMigration();
                    LOG.info("[MigrationProcess] Running Post DDL Scripts, Version: {}, DatabaseType: {}, FileName: {}", new Object[]{process.getVersion(), process.getDatabaseConnectionType(), process.getPostDDLScriptFilePath()});
                    process.runPostDDLScripts();
                    LOG.info("[MigrationStep] Update Migration Status, Version: {}, DatabaseType: {}, FileName: {}", new Object[]{process.getVersion(), process.getDatabaseConnectionType(), process.getMigrationsPath()});
                    this.updateMigrationStepInDB(process);
                }
            }
            catch (Exception e) {
                LOG.error("Encountered Exception in MigrationWorkflow", (Throwable)e);
                LOG.info("[MigrationWorkflow] Rolling Back Transaction");
                throw e;
            }
        }
        LOG.info("[MigrationWorkflow] WorkFlow Completed");
    }

    public void closeMigrationWorkflow() {
    }

    public void updateMigrationStepInDB(MigrationProcess step) {
        this.migrationDAO.upsertServerMigration(step.getVersion(), step.getMigrationsPath(), UUID.randomUUID().toString());
    }

    public void migrateSearchIndexes() {
    }
}

