/*
 * Decompiled with CFR 0.152.
 */
package org.openmetadata.service.security.auth;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.unboundid.ldap.sdk.Attribute;
import com.unboundid.ldap.sdk.BindResult;
import com.unboundid.ldap.sdk.Filter;
import com.unboundid.ldap.sdk.LDAPConnection;
import com.unboundid.ldap.sdk.LDAPConnectionOptions;
import com.unboundid.ldap.sdk.LDAPConnectionPool;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.ldap.sdk.SearchRequest;
import com.unboundid.ldap.sdk.SearchResult;
import com.unboundid.ldap.sdk.SearchResultEntry;
import com.unboundid.ldap.sdk.SearchScope;
import com.unboundid.util.ssl.SSLUtil;
import freemarker.template.TemplateException;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.net.SocketFactory;
import javax.net.ssl.TrustManager;
import javax.ws.rs.core.Response;
import org.openmetadata.common.utils.CommonUtil;
import org.openmetadata.schema.TokenInterface;
import org.openmetadata.schema.api.configuration.LoginConfiguration;
import org.openmetadata.schema.auth.LdapConfiguration;
import org.openmetadata.schema.auth.LoginRequest;
import org.openmetadata.schema.auth.RefreshToken;
import org.openmetadata.schema.auth.TokenType;
import org.openmetadata.schema.entity.teams.Role;
import org.openmetadata.schema.entity.teams.User;
import org.openmetadata.schema.services.connections.metadata.AuthProvider;
import org.openmetadata.schema.settings.SettingsType;
import org.openmetadata.schema.type.EntityReference;
import org.openmetadata.service.Entity;
import org.openmetadata.service.OpenMetadataApplicationConfig;
import org.openmetadata.service.auth.JwtResponse;
import org.openmetadata.service.exception.CustomExceptionMessage;
import org.openmetadata.service.exception.EntityNotFoundException;
import org.openmetadata.service.exception.UnhandledServerException;
import org.openmetadata.service.jdbi3.RoleRepository;
import org.openmetadata.service.jdbi3.TokenRepository;
import org.openmetadata.service.jdbi3.UserRepository;
import org.openmetadata.service.resources.settings.SettingsCache;
import org.openmetadata.service.security.AuthenticationException;
import org.openmetadata.service.security.auth.AuthenticatorHandler;
import org.openmetadata.service.security.auth.LoginAttemptCache;
import org.openmetadata.service.util.EmailUtil;
import org.openmetadata.service.util.JsonUtils;
import org.openmetadata.service.util.LdapUtil;
import org.openmetadata.service.util.TokenUtil;
import org.openmetadata.service.util.UserUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.util.CollectionUtils;

public class LdapAuthenticator
implements AuthenticatorHandler {
    private static final Logger LOG = LoggerFactory.getLogger(LdapAuthenticator.class);
    static final String LDAP_ERR_MSG = "[LDAP] Issue in creating a LookUp Connection ";
    private RoleRepository roleRepository;
    private UserRepository userRepository;
    private TokenRepository tokenRepository;
    private LoginAttemptCache loginAttemptCache;
    private LdapConfiguration ldapConfiguration;
    private LDAPConnectionPool ldapLookupConnectionPool;
    private LoginConfiguration loginConfiguration;

    @Override
    public void init(OpenMetadataApplicationConfig config) {
        if (!config.getAuthenticationConfiguration().getProvider().equals((Object)AuthProvider.LDAP) || config.getAuthenticationConfiguration().getLdapConfiguration() == null) {
            throw new IllegalStateException("Invalid or Missing Ldap Configuration.");
        }
        this.ldapLookupConnectionPool = this.getLdapConnectionPool(config.getAuthenticationConfiguration().getLdapConfiguration());
        this.userRepository = (UserRepository)Entity.getEntityRepository("user");
        this.roleRepository = (RoleRepository)Entity.getEntityRepository("role");
        this.tokenRepository = Entity.getTokenRepository();
        this.ldapConfiguration = config.getAuthenticationConfiguration().getLdapConfiguration();
        this.loginAttemptCache = new LoginAttemptCache();
        this.loginConfiguration = SettingsCache.getSetting(SettingsType.LOGIN_CONFIGURATION, LoginConfiguration.class);
    }

    private LDAPConnectionPool getLdapConnectionPool(LdapConfiguration ldapConfiguration) {
        LDAPConnectionPool connectionPool;
        try {
            if (Boolean.TRUE.equals(ldapConfiguration.getSslEnabled())) {
                LDAPConnectionOptions connectionOptions = new LDAPConnectionOptions();
                LdapUtil ldapUtil = new LdapUtil();
                SSLUtil sslUtil = new SSLUtil((TrustManager)ldapUtil.getLdapSSLConnection(ldapConfiguration, connectionOptions));
                LDAPConnection connection = new LDAPConnection((SocketFactory)sslUtil.createSSLSocketFactory(), connectionOptions, ldapConfiguration.getHost(), ldapConfiguration.getPort().intValue(), ldapConfiguration.getDnAdminPrincipal(), ldapConfiguration.getDnAdminPassword());
                connectionPool = new LDAPConnectionPool(connection, ldapConfiguration.getMaxPoolSize().intValue());
            } else {
                LDAPConnection conn = new LDAPConnection(ldapConfiguration.getHost(), ldapConfiguration.getPort().intValue(), ldapConfiguration.getDnAdminPrincipal(), ldapConfiguration.getDnAdminPassword());
                connectionPool = new LDAPConnectionPool(conn, ldapConfiguration.getMaxPoolSize().intValue());
            }
        }
        catch (LDAPException | GeneralSecurityException e) {
            LOG.error("[LDAP] Issue in creating a LookUp Connection", e);
            throw new IllegalStateException(LDAP_ERR_MSG, e);
        }
        return connectionPool;
    }

    @Override
    public JwtResponse loginUser(LoginRequest loginRequest) throws IOException, TemplateException {
        this.checkIfLoginBlocked(loginRequest.getEmail());
        User storedUser = this.lookUserInProvider(loginRequest.getEmail());
        this.validatePassword(storedUser.getEmail(), storedUser, loginRequest.getPassword());
        User omUser = this.checkAndCreateUser(storedUser.getEmail(), storedUser.getFullyQualifiedName(), storedUser.getName());
        return this.getJwtResponse(omUser, this.loginConfiguration.getJwtTokenExpiryTime().intValue());
    }

    private User checkAndCreateUser(String email, String userName, String userDn) throws IOException {
        try {
            User omUser = this.userRepository.getByName(null, userName, this.userRepository.getFields("id,name,email,roles"));
            this.getRoleForLdap(omUser, userDn, Boolean.TRUE);
            return omUser;
        }
        catch (EntityNotFoundException ex) {
            return this.userRepository.create(null, this.getUserForLdap(email, userName, userDn));
        }
        catch (LDAPException e) {
            LOG.error("An error occurs when reassigning roles for an LDAP user({}): {}", new Object[]{userName, e.getMessage(), e});
            throw new UnhandledServerException(e.getMessage());
        }
    }

    @Override
    public void checkIfLoginBlocked(String email) {
        if (this.loginAttemptCache.isLoginBlocked(email)) {
            throw new AuthenticationException("Failed Login Attempts Exceeded. Please try after some time.");
        }
    }

    @Override
    public void recordFailedLoginAttempt(String providedIdentity, User storedUser) throws TemplateException, IOException {
        this.loginAttemptCache.recordFailedLogin(providedIdentity);
        int failedLoginAttempt = this.loginAttemptCache.getUserFailedLoginCount(providedIdentity);
        if (failedLoginAttempt == this.loginConfiguration.getMaxLoginFailAttempts()) {
            EmailUtil.sendAccountStatus(storedUser, "Multiple Failed Login Attempts.", String.format("Someone is tried accessing your account. Login is Blocked for %s seconds.", this.loginConfiguration.getAccessBlockTime()));
        }
    }

    @Override
    public void validatePassword(String providedIdentity, User storedUser, String reqPassword) throws TemplateException, IOException {
        BindResult bindingResult;
        block4: {
            bindingResult = null;
            try {
                bindingResult = this.ldapLookupConnectionPool.bind(storedUser.getName(), reqPassword);
                if (Objects.equals(bindingResult.getResultCode().getName(), ResultCode.SUCCESS.getName())) {
                    return;
                }
            }
            catch (Exception ex) {
                if (bindingResult == null || !Objects.equals(bindingResult.getResultCode().getName(), ResultCode.INVALID_CREDENTIALS.getName())) break block4;
                this.recordFailedLoginAttempt(providedIdentity, storedUser);
                throw new CustomExceptionMessage(Response.Status.UNAUTHORIZED, "INVALID_USER_OR_PASSWORD", "You have entered an invalid email or password.");
            }
        }
        if (bindingResult != null) {
            throw new CustomExceptionMessage(Response.Status.INTERNAL_SERVER_ERROR, "INVALID_USER_OR_PASSWORD", bindingResult.getResultCode().getName());
        }
        throw new CustomExceptionMessage(Response.Status.INTERNAL_SERVER_ERROR, "INVALID_USER_OR_PASSWORD", "You have entered an invalid email or password.");
    }

    @Override
    public User lookUserInProvider(String email) {
        try {
            Filter emailFilter = Filter.createEqualityFilter((String)this.ldapConfiguration.getMailAttributeName(), (String)email);
            SearchRequest searchRequest = new SearchRequest(this.ldapConfiguration.getUserBaseDN(), SearchScope.SUB, emailFilter, new String[]{this.ldapConfiguration.getMailAttributeName()});
            SearchResult result = this.ldapLookupConnectionPool.search(searchRequest);
            if (result.getSearchEntries().size() == 1) {
                SearchResultEntry searchResultEntry = (SearchResultEntry)result.getSearchEntries().get(0);
                String userDN = searchResultEntry.getDN();
                Attribute emailAttr = searchResultEntry.getAttribute(this.ldapConfiguration.getMailAttributeName());
                if (!CommonUtil.nullOrEmpty((String)userDN) && emailAttr != null) {
                    return this.getUserForLdap(email).withName(userDN);
                }
                throw new CustomExceptionMessage(Response.Status.FORBIDDEN, "INVALID_USER_OR_PASSWORD", "Username or Email Attribute is incorrect. Please check Openmetadata Configuration.");
            }
            if (result.getSearchEntries().size() > 1) {
                throw new CustomExceptionMessage(Response.Status.INTERNAL_SERVER_ERROR, "Email corresponds to multiple entries in Directory.", "Email corresponds to multiple entries in Directory.");
            }
            throw new CustomExceptionMessage(Response.Status.INTERNAL_SERVER_ERROR, "Email corresponds to multiple entries in Directory.", "You have entered an invalid email or password.");
        }
        catch (LDAPException ex) {
            throw new CustomExceptionMessage(Response.Status.INTERNAL_SERVER_ERROR, "LDAP_ERROR", ex.getMessage());
        }
    }

    private User getUserForLdap(String email) {
        String userName = email.split("@")[0];
        return new User().withId(UUID.randomUUID()).withName(userName).withFullyQualifiedName(userName).withEmail(email).withIsBot(Boolean.valueOf(false)).withUpdatedBy(userName).withUpdatedAt(Long.valueOf(System.currentTimeMillis())).withIsEmailVerified(Boolean.valueOf(false)).withAuthenticationMechanism(null);
    }

    private User getUserForLdap(String email, String userName, String userDn) {
        User user = new User().withId(UUID.randomUUID()).withName(userName).withFullyQualifiedName(userName).withEmail(email).withIsBot(Boolean.valueOf(false)).withUpdatedBy(userName).withUpdatedAt(Long.valueOf(System.currentTimeMillis())).withIsEmailVerified(Boolean.valueOf(false)).withAuthenticationMechanism(null);
        try {
            this.getRoleForLdap(user, userDn, false);
        }
        catch (JsonProcessingException | LDAPException e) {
            LOG.error("Failed to assign roles from LDAP to OpenMetadata for the user {} due to {}", (Object)user.getName(), (Object)e.getMessage());
        }
        return user;
    }

    private void getRoleForLdap(User user, String userDn, Boolean reAssign) throws LDAPException, JsonProcessingException {
        try {
            Filter groupFilter = Filter.createEqualityFilter((String)this.ldapConfiguration.getGroupAttributeName(), (String)this.ldapConfiguration.getGroupAttributeValue());
            Filter groupMemberAttr = Filter.createEqualityFilter((String)this.ldapConfiguration.getGroupMemberAttributeName(), (String)userDn);
            Filter groupAndMemberFilter = Filter.createANDFilter((Filter[])new Filter[]{groupFilter, groupMemberAttr});
            SearchRequest searchRequest = new SearchRequest(this.ldapConfiguration.getGroupBaseDN(), SearchScope.SUB, groupAndMemberFilter, new String[]{this.ldapConfiguration.getAllAttributeName()});
            SearchResult searchResult = this.ldapLookupConnectionPool.search(searchRequest);
            if (CollectionUtils.isEmpty((Collection)searchResult.getSearchEntries())) {
                if (Boolean.TRUE.equals(reAssign)) {
                    user.setRoles(this.getReassignRoles(user, new ArrayList<EntityReference>(0), Boolean.FALSE));
                    UserUtil.addOrUpdateUser(user);
                }
                return;
            }
            Map<String, List<String>> roleMapping = JsonUtils.readValue(this.ldapConfiguration.getAuthRolesMapping(), new TypeReference<Map<String, List<String>>>(){});
            ArrayList<EntityReference> roleReferenceList = new ArrayList();
            boolean adminFlag = Boolean.FALSE;
            for (SearchResultEntry searchResultEntry : searchResult.getSearchEntries()) {
                String groupDN = searchResultEntry.getDN();
                if (!roleMapping.containsKey(groupDN) || CollectionUtils.isEmpty((Collection)roleMapping.get(groupDN))) continue;
                List<String> roles = roleMapping.get(groupDN);
                for (String roleName : roles) {
                    if (this.ldapConfiguration.getRoleAdminName().equals(roleName)) {
                        adminFlag = Boolean.TRUE;
                        continue;
                    }
                    try {
                        Role roleOm = (Role)this.roleRepository.getByName(null, roleName, this.roleRepository.getFields("id,name"));
                        EntityReference entityReference = new EntityReference();
                        BeanUtils.copyProperties((Object)roleOm, (Object)entityReference);
                        entityReference.setType("role");
                        roleReferenceList.add(entityReference);
                    }
                    catch (EntityNotFoundException ex) {
                        LOG.error("Role {} does not exist in OM Database", (Object)roleName);
                    }
                }
            }
            roleReferenceList = new ArrayList(roleReferenceList.stream().collect(Collectors.toMap(EntityReference::getName, Function.identity(), (entityReference1, entityReference2) -> entityReference1, LinkedHashMap::new)).values());
            user.setRoles(this.getReassignRoles(user, roleReferenceList, adminFlag));
            if (Boolean.TRUE.equals(reAssign)) {
                UserUtil.addOrUpdateUser(user);
            }
        }
        catch (Exception ex) {
            LOG.warn("Failed to get user's groups from LDAP server using the DN of the user {} due to {}", (Object)userDn, (Object)ex.getMessage());
        }
    }

    private List<EntityReference> getReassignRoles(User user, List<EntityReference> mapRoleList, Boolean adminFlag) {
        Set reassignRolesSet = this.ldapConfiguration.getAuthReassignRoles();
        if (!reassignRolesSet.contains(this.ldapConfiguration.getAllAttributeName())) {
            List oldRoleList;
            if (!CollectionUtils.isEmpty(mapRoleList)) {
                for (int i = mapRoleList.size() - 1; i >= 0; --i) {
                    EntityReference mappingRole = mapRoleList.get(i);
                    if (reassignRolesSet.contains(mappingRole.getName())) continue;
                    mapRoleList.remove(i);
                }
            }
            if (!CollectionUtils.isEmpty((Collection)(oldRoleList = user.getRoles()))) {
                for (EntityReference oldRole : oldRoleList) {
                    if (reassignRolesSet.contains(oldRole.getName())) continue;
                    mapRoleList.add(oldRole);
                }
            }
            if (reassignRolesSet.contains(this.ldapConfiguration.getRoleAdminName())) {
                user.setIsAdmin(adminFlag);
            }
        } else {
            user.setIsAdmin(adminFlag);
        }
        return mapRoleList;
    }

    @Override
    public RefreshToken createRefreshTokenForLogin(UUID currentUserId) {
        this.tokenRepository.deleteTokenByUserAndType(currentUserId, TokenType.REFRESH_TOKEN.toString());
        RefreshToken newRefreshToken = TokenUtil.getRefreshToken(currentUserId, UUID.randomUUID());
        this.tokenRepository.insertToken((TokenInterface)newRefreshToken);
        return newRefreshToken;
    }
}

