/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesis.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.kinesis.transform.StreamDescriptionSummaryMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the output for <a>DescribeStreamSummary</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StreamDescriptionSummary implements StructuredPojo,
        ToCopyableBuilder<StreamDescriptionSummary.Builder, StreamDescriptionSummary> {
    private final String streamName;

    private final String streamARN;

    private final String streamStatus;

    private final Integer retentionPeriodHours;

    private final Instant streamCreationTimestamp;

    private final List<EnhancedMetrics> enhancedMonitoring;

    private final String encryptionType;

    private final String keyId;

    private final Integer openShardCount;

    private final Integer consumerCount;

    private StreamDescriptionSummary(BuilderImpl builder) {
        this.streamName = builder.streamName;
        this.streamARN = builder.streamARN;
        this.streamStatus = builder.streamStatus;
        this.retentionPeriodHours = builder.retentionPeriodHours;
        this.streamCreationTimestamp = builder.streamCreationTimestamp;
        this.enhancedMonitoring = builder.enhancedMonitoring;
        this.encryptionType = builder.encryptionType;
        this.keyId = builder.keyId;
        this.openShardCount = builder.openShardCount;
        this.consumerCount = builder.consumerCount;
    }

    /**
     * <p>
     * The name of the stream being described.
     * </p>
     * 
     * @return The name of the stream being described.
     */
    public String streamName() {
        return streamName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the stream being described.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the stream being described.
     */
    public String streamARN() {
        return streamARN;
    }

    /**
     * <p>
     * The current status of the stream being described. The stream status is one of the following states:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The stream is being created. Kinesis Data Streams immediately returns and sets
     * <code>StreamStatus</code> to <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The stream is being deleted. The specified stream is in the <code>DELETING</code> state
     * until Kinesis Data Streams completes the deletion.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The stream exists and is ready for read and write operations or deletion. You should
     * perform read and write operations only on an <code>ACTIVE</code> stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - Shards in the stream are being merged or split. Read and write operations continue to
     * work while the stream is in the <code>UPDATING</code> state.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #streamStatus} will
     * return {@link StreamStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #streamStatusAsString}.
     * </p>
     * 
     * @return The current status of the stream being described. The stream status is one of the following states:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The stream is being created. Kinesis Data Streams immediately returns and sets
     *         <code>StreamStatus</code> to <code>CREATING</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The stream is being deleted. The specified stream is in the <code>DELETING</code>
     *         state until Kinesis Data Streams completes the deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The stream exists and is ready for read and write operations or deletion. You
     *         should perform read and write operations only on an <code>ACTIVE</code> stream.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - Shards in the stream are being merged or split. Read and write operations
     *         continue to work while the stream is in the <code>UPDATING</code> state.
     *         </p>
     *         </li>
     * @see StreamStatus
     */
    public StreamStatus streamStatus() {
        return StreamStatus.fromValue(streamStatus);
    }

    /**
     * <p>
     * The current status of the stream being described. The stream status is one of the following states:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The stream is being created. Kinesis Data Streams immediately returns and sets
     * <code>StreamStatus</code> to <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The stream is being deleted. The specified stream is in the <code>DELETING</code> state
     * until Kinesis Data Streams completes the deletion.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The stream exists and is ready for read and write operations or deletion. You should
     * perform read and write operations only on an <code>ACTIVE</code> stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - Shards in the stream are being merged or split. Read and write operations continue to
     * work while the stream is in the <code>UPDATING</code> state.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #streamStatus} will
     * return {@link StreamStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #streamStatusAsString}.
     * </p>
     * 
     * @return The current status of the stream being described. The stream status is one of the following states:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The stream is being created. Kinesis Data Streams immediately returns and sets
     *         <code>StreamStatus</code> to <code>CREATING</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The stream is being deleted. The specified stream is in the <code>DELETING</code>
     *         state until Kinesis Data Streams completes the deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The stream exists and is ready for read and write operations or deletion. You
     *         should perform read and write operations only on an <code>ACTIVE</code> stream.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - Shards in the stream are being merged or split. Read and write operations
     *         continue to work while the stream is in the <code>UPDATING</code> state.
     *         </p>
     *         </li>
     * @see StreamStatus
     */
    public String streamStatusAsString() {
        return streamStatus;
    }

    /**
     * <p>
     * The current retention period, in hours.
     * </p>
     * 
     * @return The current retention period, in hours.
     */
    public Integer retentionPeriodHours() {
        return retentionPeriodHours;
    }

    /**
     * <p>
     * The approximate time that the stream was created.
     * </p>
     * 
     * @return The approximate time that the stream was created.
     */
    public Instant streamCreationTimestamp() {
        return streamCreationTimestamp;
    }

    /**
     * <p>
     * Represents the current enhanced monitoring settings of the stream.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Represents the current enhanced monitoring settings of the stream.
     */
    public List<EnhancedMetrics> enhancedMonitoring() {
        return enhancedMonitoring;
    }

    /**
     * <p>
     * The encryption type used. This value is one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KMS</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NONE</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionType}
     * will return {@link EncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionTypeAsString}.
     * </p>
     * 
     * @return The encryption type used. This value is one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>KMS</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NONE</code>
     *         </p>
     *         </li>
     * @see EncryptionType
     */
    public EncryptionType encryptionType() {
        return EncryptionType.fromValue(encryptionType);
    }

    /**
     * <p>
     * The encryption type used. This value is one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KMS</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NONE</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionType}
     * will return {@link EncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionTypeAsString}.
     * </p>
     * 
     * @return The encryption type used. This value is one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>KMS</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NONE</code>
     *         </p>
     *         </li>
     * @see EncryptionType
     */
    public String encryptionTypeAsString() {
        return encryptionType;
    }

    /**
     * <p>
     * The GUID for the customer-managed AWS KMS key to use for encryption. This value can be a globally unique
     * identifier, a fully specified ARN to either an alias or a key, or an alias name prefixed by "alias/".You can also
     * use a master key owned by Kinesis Data Streams by specifying the alias <code>aws/kinesis</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ARN example: <code>arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias ARN example: <code> arn:aws:kms:us-east-1:123456789012:alias/MyAliasName</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Globally unique key ID example: <code>12345678-1234-1234-1234-123456789012</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias name example: <code>alias/MyAliasName</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Master key owned by Kinesis Data Streams: <code>alias/aws/kinesis</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The GUID for the customer-managed AWS KMS key to use for encryption. This value can be a globally unique
     *         identifier, a fully specified ARN to either an alias or a key, or an alias name prefixed by "alias/".You
     *         can also use a master key owned by Kinesis Data Streams by specifying the alias <code>aws/kinesis</code>
     *         .</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Key ARN example: <code>arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias ARN example: <code> arn:aws:kms:us-east-1:123456789012:alias/MyAliasName</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Globally unique key ID example: <code>12345678-1234-1234-1234-123456789012</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias name example: <code>alias/MyAliasName</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Master key owned by Kinesis Data Streams: <code>alias/aws/kinesis</code>
     *         </p>
     *         </li>
     */
    public String keyId() {
        return keyId;
    }

    /**
     * <p>
     * The number of open shards in the stream.
     * </p>
     * 
     * @return The number of open shards in the stream.
     */
    public Integer openShardCount() {
        return openShardCount;
    }

    /**
     * <p>
     * The number of enhanced fan-out consumers registered with the stream.
     * </p>
     * 
     * @return The number of enhanced fan-out consumers registered with the stream.
     */
    public Integer consumerCount() {
        return consumerCount;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(streamName());
        hashCode = 31 * hashCode + Objects.hashCode(streamARN());
        hashCode = 31 * hashCode + Objects.hashCode(streamStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(retentionPeriodHours());
        hashCode = 31 * hashCode + Objects.hashCode(streamCreationTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(enhancedMonitoring());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(keyId());
        hashCode = 31 * hashCode + Objects.hashCode(openShardCount());
        hashCode = 31 * hashCode + Objects.hashCode(consumerCount());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StreamDescriptionSummary)) {
            return false;
        }
        StreamDescriptionSummary other = (StreamDescriptionSummary) obj;
        return Objects.equals(streamName(), other.streamName()) && Objects.equals(streamARN(), other.streamARN())
                && Objects.equals(streamStatusAsString(), other.streamStatusAsString())
                && Objects.equals(retentionPeriodHours(), other.retentionPeriodHours())
                && Objects.equals(streamCreationTimestamp(), other.streamCreationTimestamp())
                && Objects.equals(enhancedMonitoring(), other.enhancedMonitoring())
                && Objects.equals(encryptionTypeAsString(), other.encryptionTypeAsString())
                && Objects.equals(keyId(), other.keyId()) && Objects.equals(openShardCount(), other.openShardCount())
                && Objects.equals(consumerCount(), other.consumerCount());
    }

    @Override
    public String toString() {
        return ToString.builder("StreamDescriptionSummary").add("StreamName", streamName()).add("StreamARN", streamARN())
                .add("StreamStatus", streamStatusAsString()).add("RetentionPeriodHours", retentionPeriodHours())
                .add("StreamCreationTimestamp", streamCreationTimestamp()).add("EnhancedMonitoring", enhancedMonitoring())
                .add("EncryptionType", encryptionTypeAsString()).add("KeyId", keyId()).add("OpenShardCount", openShardCount())
                .add("ConsumerCount", consumerCount()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StreamName":
            return Optional.ofNullable(clazz.cast(streamName()));
        case "StreamARN":
            return Optional.ofNullable(clazz.cast(streamARN()));
        case "StreamStatus":
            return Optional.ofNullable(clazz.cast(streamStatusAsString()));
        case "RetentionPeriodHours":
            return Optional.ofNullable(clazz.cast(retentionPeriodHours()));
        case "StreamCreationTimestamp":
            return Optional.ofNullable(clazz.cast(streamCreationTimestamp()));
        case "EnhancedMonitoring":
            return Optional.ofNullable(clazz.cast(enhancedMonitoring()));
        case "EncryptionType":
            return Optional.ofNullable(clazz.cast(encryptionTypeAsString()));
        case "KeyId":
            return Optional.ofNullable(clazz.cast(keyId()));
        case "OpenShardCount":
            return Optional.ofNullable(clazz.cast(openShardCount()));
        case "ConsumerCount":
            return Optional.ofNullable(clazz.cast(consumerCount()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        StreamDescriptionSummaryMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, StreamDescriptionSummary> {
        /**
         * <p>
         * The name of the stream being described.
         * </p>
         * 
         * @param streamName
         *        The name of the stream being described.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamName(String streamName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the stream being described.
         * </p>
         * 
         * @param streamARN
         *        The Amazon Resource Name (ARN) for the stream being described.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamARN(String streamARN);

        /**
         * <p>
         * The current status of the stream being described. The stream status is one of the following states:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The stream is being created. Kinesis Data Streams immediately returns and sets
         * <code>StreamStatus</code> to <code>CREATING</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The stream is being deleted. The specified stream is in the <code>DELETING</code>
         * state until Kinesis Data Streams completes the deletion.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The stream exists and is ready for read and write operations or deletion. You should
         * perform read and write operations only on an <code>ACTIVE</code> stream.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - Shards in the stream are being merged or split. Read and write operations continue to
         * work while the stream is in the <code>UPDATING</code> state.
         * </p>
         * </li>
         * </ul>
         * 
         * @param streamStatus
         *        The current status of the stream being described. The stream status is one of the following
         *        states:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The stream is being created. Kinesis Data Streams immediately returns and sets
         *        <code>StreamStatus</code> to <code>CREATING</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The stream is being deleted. The specified stream is in the
         *        <code>DELETING</code> state until Kinesis Data Streams completes the deletion.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The stream exists and is ready for read and write operations or deletion. You
         *        should perform read and write operations only on an <code>ACTIVE</code> stream.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - Shards in the stream are being merged or split. Read and write operations
         *        continue to work while the stream is in the <code>UPDATING</code> state.
         *        </p>
         *        </li>
         * @see StreamStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StreamStatus
         */
        Builder streamStatus(String streamStatus);

        /**
         * <p>
         * The current status of the stream being described. The stream status is one of the following states:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The stream is being created. Kinesis Data Streams immediately returns and sets
         * <code>StreamStatus</code> to <code>CREATING</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The stream is being deleted. The specified stream is in the <code>DELETING</code>
         * state until Kinesis Data Streams completes the deletion.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The stream exists and is ready for read and write operations or deletion. You should
         * perform read and write operations only on an <code>ACTIVE</code> stream.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - Shards in the stream are being merged or split. Read and write operations continue to
         * work while the stream is in the <code>UPDATING</code> state.
         * </p>
         * </li>
         * </ul>
         * 
         * @param streamStatus
         *        The current status of the stream being described. The stream status is one of the following
         *        states:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The stream is being created. Kinesis Data Streams immediately returns and sets
         *        <code>StreamStatus</code> to <code>CREATING</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The stream is being deleted. The specified stream is in the
         *        <code>DELETING</code> state until Kinesis Data Streams completes the deletion.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The stream exists and is ready for read and write operations or deletion. You
         *        should perform read and write operations only on an <code>ACTIVE</code> stream.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - Shards in the stream are being merged or split. Read and write operations
         *        continue to work while the stream is in the <code>UPDATING</code> state.
         *        </p>
         *        </li>
         * @see StreamStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StreamStatus
         */
        Builder streamStatus(StreamStatus streamStatus);

        /**
         * <p>
         * The current retention period, in hours.
         * </p>
         * 
         * @param retentionPeriodHours
         *        The current retention period, in hours.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retentionPeriodHours(Integer retentionPeriodHours);

        /**
         * <p>
         * The approximate time that the stream was created.
         * </p>
         * 
         * @param streamCreationTimestamp
         *        The approximate time that the stream was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamCreationTimestamp(Instant streamCreationTimestamp);

        /**
         * <p>
         * Represents the current enhanced monitoring settings of the stream.
         * </p>
         * 
         * @param enhancedMonitoring
         *        Represents the current enhanced monitoring settings of the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enhancedMonitoring(Collection<EnhancedMetrics> enhancedMonitoring);

        /**
         * <p>
         * Represents the current enhanced monitoring settings of the stream.
         * </p>
         * 
         * @param enhancedMonitoring
         *        Represents the current enhanced monitoring settings of the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enhancedMonitoring(EnhancedMetrics... enhancedMonitoring);

        /**
         * <p>
         * Represents the current enhanced monitoring settings of the stream.
         * </p>
         * This is a convenience that creates an instance of the {@link List<EnhancedMetrics>.Builder} avoiding the need
         * to create one manually via {@link List<EnhancedMetrics>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<EnhancedMetrics>.Builder#build()} is called immediately and
         * its result is passed to {@link #enhancedMonitoring(List<EnhancedMetrics>)}.
         * 
         * @param enhancedMonitoring
         *        a consumer that will call methods on {@link List<EnhancedMetrics>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #enhancedMonitoring(List<EnhancedMetrics>)
         */
        Builder enhancedMonitoring(Consumer<EnhancedMetrics.Builder>... enhancedMonitoring);

        /**
         * <p>
         * The encryption type used. This value is one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KMS</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param encryptionType
         *        The encryption type used. This value is one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KMS</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NONE</code>
         *        </p>
         *        </li>
         * @see EncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionType
         */
        Builder encryptionType(String encryptionType);

        /**
         * <p>
         * The encryption type used. This value is one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KMS</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param encryptionType
         *        The encryption type used. This value is one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KMS</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NONE</code>
         *        </p>
         *        </li>
         * @see EncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionType
         */
        Builder encryptionType(EncryptionType encryptionType);

        /**
         * <p>
         * The GUID for the customer-managed AWS KMS key to use for encryption. This value can be a globally unique
         * identifier, a fully specified ARN to either an alias or a key, or an alias name prefixed by "alias/".You can
         * also use a master key owned by Kinesis Data Streams by specifying the alias <code>aws/kinesis</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Key ARN example: <code>arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias ARN example: <code> arn:aws:kms:us-east-1:123456789012:alias/MyAliasName</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Globally unique key ID example: <code>12345678-1234-1234-1234-123456789012</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias name example: <code>alias/MyAliasName</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Master key owned by Kinesis Data Streams: <code>alias/aws/kinesis</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param keyId
         *        The GUID for the customer-managed AWS KMS key to use for encryption. This value can be a globally
         *        unique identifier, a fully specified ARN to either an alias or a key, or an alias name prefixed by
         *        "alias/".You can also use a master key owned by Kinesis Data Streams by specifying the alias
         *        <code>aws/kinesis</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Key ARN example:
         *        <code>arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias ARN example: <code> arn:aws:kms:us-east-1:123456789012:alias/MyAliasName</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Globally unique key ID example: <code>12345678-1234-1234-1234-123456789012</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias name example: <code>alias/MyAliasName</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Master key owned by Kinesis Data Streams: <code>alias/aws/kinesis</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyId(String keyId);

        /**
         * <p>
         * The number of open shards in the stream.
         * </p>
         * 
         * @param openShardCount
         *        The number of open shards in the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder openShardCount(Integer openShardCount);

        /**
         * <p>
         * The number of enhanced fan-out consumers registered with the stream.
         * </p>
         * 
         * @param consumerCount
         *        The number of enhanced fan-out consumers registered with the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder consumerCount(Integer consumerCount);
    }

    static final class BuilderImpl implements Builder {
        private String streamName;

        private String streamARN;

        private String streamStatus;

        private Integer retentionPeriodHours;

        private Instant streamCreationTimestamp;

        private List<EnhancedMetrics> enhancedMonitoring = DefaultSdkAutoConstructList.getInstance();

        private String encryptionType;

        private String keyId;

        private Integer openShardCount;

        private Integer consumerCount;

        private BuilderImpl() {
        }

        private BuilderImpl(StreamDescriptionSummary model) {
            streamName(model.streamName);
            streamARN(model.streamARN);
            streamStatus(model.streamStatus);
            retentionPeriodHours(model.retentionPeriodHours);
            streamCreationTimestamp(model.streamCreationTimestamp);
            enhancedMonitoring(model.enhancedMonitoring);
            encryptionType(model.encryptionType);
            keyId(model.keyId);
            openShardCount(model.openShardCount);
            consumerCount(model.consumerCount);
        }

        public final String getStreamName() {
            return streamName;
        }

        @Override
        public final Builder streamName(String streamName) {
            this.streamName = streamName;
            return this;
        }

        public final void setStreamName(String streamName) {
            this.streamName = streamName;
        }

        public final String getStreamARN() {
            return streamARN;
        }

        @Override
        public final Builder streamARN(String streamARN) {
            this.streamARN = streamARN;
            return this;
        }

        public final void setStreamARN(String streamARN) {
            this.streamARN = streamARN;
        }

        public final String getStreamStatus() {
            return streamStatus;
        }

        @Override
        public final Builder streamStatus(String streamStatus) {
            this.streamStatus = streamStatus;
            return this;
        }

        @Override
        public final Builder streamStatus(StreamStatus streamStatus) {
            this.streamStatus(streamStatus.toString());
            return this;
        }

        public final void setStreamStatus(String streamStatus) {
            this.streamStatus = streamStatus;
        }

        public final Integer getRetentionPeriodHours() {
            return retentionPeriodHours;
        }

        @Override
        public final Builder retentionPeriodHours(Integer retentionPeriodHours) {
            this.retentionPeriodHours = retentionPeriodHours;
            return this;
        }

        public final void setRetentionPeriodHours(Integer retentionPeriodHours) {
            this.retentionPeriodHours = retentionPeriodHours;
        }

        public final Instant getStreamCreationTimestamp() {
            return streamCreationTimestamp;
        }

        @Override
        public final Builder streamCreationTimestamp(Instant streamCreationTimestamp) {
            this.streamCreationTimestamp = streamCreationTimestamp;
            return this;
        }

        public final void setStreamCreationTimestamp(Instant streamCreationTimestamp) {
            this.streamCreationTimestamp = streamCreationTimestamp;
        }

        public final Collection<EnhancedMetrics.Builder> getEnhancedMonitoring() {
            return enhancedMonitoring != null ? enhancedMonitoring.stream().map(EnhancedMetrics::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder enhancedMonitoring(Collection<EnhancedMetrics> enhancedMonitoring) {
            this.enhancedMonitoring = EnhancedMonitoringListCopier.copy(enhancedMonitoring);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder enhancedMonitoring(EnhancedMetrics... enhancedMonitoring) {
            enhancedMonitoring(Arrays.asList(enhancedMonitoring));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder enhancedMonitoring(Consumer<EnhancedMetrics.Builder>... enhancedMonitoring) {
            enhancedMonitoring(Stream.of(enhancedMonitoring).map(c -> EnhancedMetrics.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setEnhancedMonitoring(Collection<EnhancedMetrics.BuilderImpl> enhancedMonitoring) {
            this.enhancedMonitoring = EnhancedMonitoringListCopier.copyFromBuilder(enhancedMonitoring);
        }

        public final String getEncryptionType() {
            return encryptionType;
        }

        @Override
        public final Builder encryptionType(String encryptionType) {
            this.encryptionType = encryptionType;
            return this;
        }

        @Override
        public final Builder encryptionType(EncryptionType encryptionType) {
            this.encryptionType(encryptionType.toString());
            return this;
        }

        public final void setEncryptionType(String encryptionType) {
            this.encryptionType = encryptionType;
        }

        public final String getKeyId() {
            return keyId;
        }

        @Override
        public final Builder keyId(String keyId) {
            this.keyId = keyId;
            return this;
        }

        public final void setKeyId(String keyId) {
            this.keyId = keyId;
        }

        public final Integer getOpenShardCount() {
            return openShardCount;
        }

        @Override
        public final Builder openShardCount(Integer openShardCount) {
            this.openShardCount = openShardCount;
            return this;
        }

        public final void setOpenShardCount(Integer openShardCount) {
            this.openShardCount = openShardCount;
        }

        public final Integer getConsumerCount() {
            return consumerCount;
        }

        @Override
        public final Builder consumerCount(Integer consumerCount) {
            this.consumerCount = consumerCount;
            return this;
        }

        public final void setConsumerCount(Integer consumerCount) {
            this.consumerCount = consumerCount;
        }

        @Override
        public StreamDescriptionSummary build() {
            return new StreamDescriptionSummary(this);
        }
    }
}
