/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediatailor.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * AWS Secrets Manager access token configuration parameters. For information about Secrets Manager access token
 * authentication, see <a
 * href="https://docs.aws.amazon.com/mediatailor/latest/ug/channel-assembly-access-configuration-access-token.html"
 * >Working with AWS Secrets Manager access token authentication</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SecretsManagerAccessTokenConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<SecretsManagerAccessTokenConfiguration.Builder, SecretsManagerAccessTokenConfiguration> {
    private static final SdkField<String> HEADER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HeaderName").getter(getter(SecretsManagerAccessTokenConfiguration::headerName))
            .setter(setter(Builder::headerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HeaderName").build()).build();

    private static final SdkField<String> SECRET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretArn").getter(getter(SecretsManagerAccessTokenConfiguration::secretArn))
            .setter(setter(Builder::secretArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretArn").build()).build();

    private static final SdkField<String> SECRET_STRING_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretStringKey").getter(getter(SecretsManagerAccessTokenConfiguration::secretStringKey))
            .setter(setter(Builder::secretStringKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretStringKey").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(HEADER_NAME_FIELD,
            SECRET_ARN_FIELD, SECRET_STRING_KEY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String headerName;

    private final String secretArn;

    private final String secretStringKey;

    private SecretsManagerAccessTokenConfiguration(BuilderImpl builder) {
        this.headerName = builder.headerName;
        this.secretArn = builder.secretArn;
        this.secretStringKey = builder.secretStringKey;
    }

    /**
     * <p>
     * The name of the HTTP header used to supply the access token in requests to the source location.
     * </p>
     * 
     * @return The name of the HTTP header used to supply the access token in requests to the source location.
     */
    public final String headerName() {
        return headerName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the AWS Secrets Manager secret that contains the access token.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the AWS Secrets Manager secret that contains the access token.
     */
    public final String secretArn() {
        return secretArn;
    }

    /**
     * <p>
     * The AWS Secrets Manager <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_CreateSecret.html#SecretsManager-CreateSecret-request-SecretString.html"
     * >SecretString</a> key associated with the access token. MediaTailor uses the key to look up SecretString key and
     * value pair containing the access token.
     * </p>
     * 
     * @return The AWS Secrets Manager <a href=
     *         "https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_CreateSecret.html#SecretsManager-CreateSecret-request-SecretString.html"
     *         >SecretString</a> key associated with the access token. MediaTailor uses the key to look up SecretString
     *         key and value pair containing the access token.
     */
    public final String secretStringKey() {
        return secretStringKey;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(headerName());
        hashCode = 31 * hashCode + Objects.hashCode(secretArn());
        hashCode = 31 * hashCode + Objects.hashCode(secretStringKey());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SecretsManagerAccessTokenConfiguration)) {
            return false;
        }
        SecretsManagerAccessTokenConfiguration other = (SecretsManagerAccessTokenConfiguration) obj;
        return Objects.equals(headerName(), other.headerName()) && Objects.equals(secretArn(), other.secretArn())
                && Objects.equals(secretStringKey(), other.secretStringKey());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SecretsManagerAccessTokenConfiguration").add("HeaderName", headerName())
                .add("SecretArn", secretArn()).add("SecretStringKey", secretStringKey()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "HeaderName":
            return Optional.ofNullable(clazz.cast(headerName()));
        case "SecretArn":
            return Optional.ofNullable(clazz.cast(secretArn()));
        case "SecretStringKey":
            return Optional.ofNullable(clazz.cast(secretStringKey()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("HeaderName", HEADER_NAME_FIELD);
        map.put("SecretArn", SECRET_ARN_FIELD);
        map.put("SecretStringKey", SECRET_STRING_KEY_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SecretsManagerAccessTokenConfiguration, T> g) {
        return obj -> g.apply((SecretsManagerAccessTokenConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SecretsManagerAccessTokenConfiguration> {
        /**
         * <p>
         * The name of the HTTP header used to supply the access token in requests to the source location.
         * </p>
         * 
         * @param headerName
         *        The name of the HTTP header used to supply the access token in requests to the source location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder headerName(String headerName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the AWS Secrets Manager secret that contains the access token.
         * </p>
         * 
         * @param secretArn
         *        The Amazon Resource Name (ARN) of the AWS Secrets Manager secret that contains the access token.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretArn(String secretArn);

        /**
         * <p>
         * The AWS Secrets Manager <a href=
         * "https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_CreateSecret.html#SecretsManager-CreateSecret-request-SecretString.html"
         * >SecretString</a> key associated with the access token. MediaTailor uses the key to look up SecretString key
         * and value pair containing the access token.
         * </p>
         * 
         * @param secretStringKey
         *        The AWS Secrets Manager <a href=
         *        "https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_CreateSecret.html#SecretsManager-CreateSecret-request-SecretString.html"
         *        >SecretString</a> key associated with the access token. MediaTailor uses the key to look up
         *        SecretString key and value pair containing the access token.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretStringKey(String secretStringKey);
    }

    static final class BuilderImpl implements Builder {
        private String headerName;

        private String secretArn;

        private String secretStringKey;

        private BuilderImpl() {
        }

        private BuilderImpl(SecretsManagerAccessTokenConfiguration model) {
            headerName(model.headerName);
            secretArn(model.secretArn);
            secretStringKey(model.secretStringKey);
        }

        public final String getHeaderName() {
            return headerName;
        }

        public final void setHeaderName(String headerName) {
            this.headerName = headerName;
        }

        @Override
        public final Builder headerName(String headerName) {
            this.headerName = headerName;
            return this;
        }

        public final String getSecretArn() {
            return secretArn;
        }

        public final void setSecretArn(String secretArn) {
            this.secretArn = secretArn;
        }

        @Override
        public final Builder secretArn(String secretArn) {
            this.secretArn = secretArn;
            return this;
        }

        public final String getSecretStringKey() {
            return secretStringKey;
        }

        public final void setSecretStringKey(String secretStringKey) {
            this.secretStringKey = secretStringKey;
        }

        @Override
        public final Builder secretStringKey(String secretStringKey) {
            this.secretStringKey = secretStringKey;
            return this;
        }

        @Override
        public SecretsManagerAccessTokenConfiguration build() {
            return new SecretsManagerAccessTokenConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
