/*
 * $Id: Writer.java 9 2007-02-06 18:20:13Z max $
 * 
 * Copyright (c) 2006-2007 Maximilian Antoni. All rights reserved.
 * 
 * This software is licensed as described in the file LICENSE.txt, which you
 * should have received as part of this distribution. The terms are also
 * available at http://www.maxantoni.de/projects/eva-properties/license.txt.
 */
package com.eva.properties;

/**
 * helper class for writing properties to a string buffer.
 * 
 * @author Max Antoni
 * @version $Revision: 9 $
 */
class Writer {
    private static final String NEWLINE = "\r\n";
    private StringBuffer buffer = new StringBuffer();
    private StringBuffer indentation = new StringBuffer();
    
    /**
     * writes the given object to this writer.
     * 
     * @param inValue the object.
     */
    void write(Object inValue) {
        if(buffer.charAt(buffer.length() - 1) == '\n') {
            appendIndentation();
        }
        if(inValue instanceof Properties) {
            ((Properties) inValue).write(this);
        }
        else if(inValue instanceof Replaceable) {
            ((Replaceable) inValue).write(this);
        }
        else if(inValue instanceof String) {
            append('\"');
            append((String) inValue);
            append("\"\n");
        }
        else {
            append(String.valueOf(inValue));
            append('\n');
        }
    }
    
    /**
     * standard constructor.
     */
    Writer() {
        super();
    }
    
    /* 
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return buffer.toString();
    }
    
    /**
     * appends the given string to this writer.
     * 
     * @param inString the string.
     */
    public void append(String inString) {
        buffer.append(inString);
    }
    
    /**
     * appends the current indentation to this writer.
     */
    public void appendIndentation() {
        buffer.append(indentation);
    }
    
    /**
     * appends a line with the current indentation, the given string and a
     * newline to this writer.
     * 
     * @param inLine the line to write.
     */
    public void appendLine(String inLine) {
        buffer.append(indentation);
        buffer.append(inLine);
        buffer.append(NEWLINE);
    }
    
    /**
     * appends a newline to this writer.
     */
    public void appendNewline() {
        buffer.append(NEWLINE);
    }
    
    /**
     * decreases the indentation level for this writer.
     */
    public void decreaseIndentation() {
        indentation.deleteCharAt(0);
    }

    /**
     * increases the indentation level for this writer.
     */
    public void increaseIndentation() {
        indentation.append('\t');
    }

    /**
     * appends the given char to this writer.
     * 
     * @param inChar the char.
     */
    public void append(char inChar) {
        buffer.append(inChar);
    }
    
}
