/*
 * $Id: ParserTest.java 109 2007-03-24 14:55:03Z max $
 * 
 * Copyright (c) 2006-2007 Maximilian Antoni. All rights reserved.
 * 
 * This software is licensed as described in the file LICENSE.txt, which you
 * should have received as part of this distribution. The terms are also
 * available at http://www.maxantoni.de/projects/eva-properties/license.txt.
 */
package com.eva.properties;

import java.io.IOException;
import java.io.StringReader;
import java.util.Map;

import junit.framework.TestCase;

/**
 * test cases for the parser.
 * 
 * @author Max Antoni
 * @version $Revision: 109 $
 */
public class ParserTest extends TestCase {

    public void testSimple() throws IOException {
        Map properties = new MapProperties(
                "classpath://com/eva/properties/simple.eva");
        assertEquals("bar", properties.get("foo"));
        assertEquals("A1", properties.get("a1"));
        assertEquals("A2", properties.get("a2"));
        assertEquals("B1", properties.get("b1"));
        assertEquals("B2", properties.get("b2"));
        assertEquals("C1", properties.get("c1"));
        assertEquals("C2", properties.get("c2"));
        assertEquals("C3", properties.get("c3"));
        assertTrue(properties.get("map") instanceof MapProperties);
        assertTrue(properties.get("list") instanceof ListProperties);
        
        assertEquals("world", properties.get("map.hello"));
        assertEquals(new Long(1), properties.get("map.one"));
        assertEquals(new Long(2), properties.get("map.two"));
        assertEquals(new Double(3.14), properties.get("map.pi"));
        
        assertEquals(new Integer(6), properties.get("list.length"));
        assertEquals(new Long(1), properties.get("list.0"));
        assertEquals(new Long(2), properties.get("list.1"));
        assertEquals(new Long(3), properties.get("list.2"));
        assertEquals(new String("a"), properties.get("list.3"));
        assertEquals(new String("b"), properties.get("list.4"));
        assertEquals(new String("c"), properties.get("list.5"));
        assertEquals(new String("world"), properties.get("hello"));
        assertEquals(new Long(1), properties.get("one"));
        assertEquals(new String("2/bla/bla"), properties.get("two"));
        assertEquals(new Double(3.14), properties.get("pi"));
    }

    public void testComplex() throws IOException {
        Map properties = new MapProperties(
                "classpath://com/eva/properties/complex.eva");
        assertEquals("base", properties.get("base.hello"));
        assertEquals("child1", properties.get("child1.hello"));
        assertEquals("base", properties.get("child2.hello"));
        assertEquals("child1", properties.get("child3.hello"));
        assertEquals("child4", properties.get("child4.hello"));
        try {
            properties.get("child5.hello");
            fail("IllegalStateException expected");
        }
        catch(RuntimeException e) {
            assertEquals("Cannot resolve \"child5.hello\", "
                    + "Invalid super reference, "
                    + "\"child.not.available\" is null.", e.getMessage());
        }
        assertEquals("world", properties.get("child6.map.hello"));
        assertEquals("child1", properties.get("child7.ref"));
    }
    
    public void testPath() throws IOException {
        Map properties = new MapProperties(
                "classpath://com/eva/properties/path.eva");
        assertEquals("/test/path", properties.get("server.home"));
        assertEquals("/test/path/conf", properties.get("server.conf"));
        assertEquals("/test/path/apps", properties.get("projects.home"));
        assertEquals("myapp", properties.get("project.name"));
        assertEquals("/test/path/apps/myapp", properties.get("project.home"));
    }
    
    public void testMapOrList() throws IOException {
        Object properties = PropertiesParser.read(new DataSource(
                "classpath://com/eva/properties/path.eva"));
        assertTrue(properties instanceof MapProperties);
        properties = PropertiesParser.read(new DataSource(
                "classpath://com/eva/properties/list1.eva"));
        assertTrue(properties instanceof ListProperties);
        ListProperties list1 = (ListProperties) properties;
        assertEquals("hello", list1.get(0));
        assertEquals("from", list1.get(1));
        assertEquals("list", list1.get(2));
        properties = PropertiesParser.read(new DataSource(
                "classpath://com/eva/properties/list2.eva"));
        assertTrue(properties instanceof ListProperties);
        ListProperties list2 = (ListProperties) properties;
        assertEquals(new Double(3.14), list2.get(0));
        assertEquals(new Long(815), list2.get(1));
    }

    public void testToFlatMap() throws IOException {
        Map map = new MapProperties("classpath://com/eva/properties/path.eva");
        assertEquals("/test/path", map.get("server.home"));
        assertEquals("/test/path/conf", map.get("server.conf"));
        assertEquals("/test/path/apps", map.get("projects.home"));
        assertEquals("myapp", map.get("project.name"));
        assertEquals("/test/path/apps/myapp", map.get("project.home"));
    }
    
    public void testJoker() throws IOException {
        Map properties = new MapProperties(
                "classpath://com/eva/properties/joker.eva");
        Map names = (Map) properties.get("names");
        assertEquals("Map 1", names.get("map1"));
        assertEquals("Map 2", names.get("map2"));
        assertEquals(2, names.size());
    }
    
    public void testNestedJoker() throws IOException {
        Map properties = new MapProperties(
                "classpath://com/eva/properties/joker.eva");
        Map names = (Map) properties.get("nestedNames");
        assertEquals("Red", names.get("colors.red"));
        assertEquals("Green", names.get("colors.green"));
        assertEquals("HelloWorld", names.get("hello.world"));
        assertEquals(2, names.size());
        
        Map values = (Map) properties.get("nestedValues");
        assertEquals("#ff0000", values.get("colors.red"));
        assertEquals("#00ff00", values.get("colors.green"));
        assertEquals("Hello world!", values.get("hello.world"));
        assertEquals(2, values.size());
    }
    
    public void testSwitch() throws IOException {
        Map properties = new MapProperties(
                "classpath://com/eva/properties/switch.eva");
        assertEquals("bar", properties.get("foo"));
        assertEquals("world", properties.get("hello"));
        assertEquals("yes", properties.get("partial"));
    }
    
    public void testIllegalKey() {
        try {
            eval("foo.bar:{}");
            fail("Illegal key \"foo.bar\" didn't throw an exception.");
        }
        catch(RuntimeException e) {
            assertEquals(": or = expected, . line 1", e.getMessage());
        }
    }
    
    public void testUnescapedValueWithDot() {
        try {
            eval("foo: foo.bar");
            fail("Illegal value \"foo.bar\" didn't throw an exception.");
        }
        catch(RuntimeException e) {
            assertEquals("EOF expected, . line 1", e.getMessage());
        }
    }
    
    public void testUnescapedValueWithColon() {
        try {
            eval("foo: foo:bar");
            fail("Illegal value \"foo:bar\" didn't throw an exception.");
        }
        catch(RuntimeException e) {
            assertEquals("EOF expected, : line 1", e.getMessage());
        }
    }
    
    public void testString() {
        Object object = eval("foo: \"c\"").get("foo");
        assertTrue(object instanceof String);
        assertEquals("c", object);
        object = eval("foo: \"abc\"").get("foo");
        assertTrue(object instanceof String);
        assertEquals("abc", object);
    }
    
    public void testChar() {
        Object object = eval("foo: 'abc'").get("foo");
        assertTrue(object instanceof char[]);
        char[] chars = (char[]) object;
        assertEquals(3, chars.length);
        assertEquals('a', chars[0]);
        assertEquals('b', chars[1]);
        assertEquals('c', chars[2]);
    }
    
    public void testCharArray() {
        Object object = eval("foo: 'c").get("foo");
        assertTrue(object instanceof Character);
        assertEquals('c', ((Character) object).charValue());
    }
    
    public void testParent() throws IOException {
        Map properties = new MapProperties(
                "classpath://com/eva/properties/parent.eva");
        assertEquals("Outer", properties.get("thing"));
        assertEquals("Inner", properties.get("map.thing"));
        assertEquals("Outer", properties.get("map.foo"));
    }
    
    public void testParseObject() {
        assertEquals("foo", PropertiesParser.readObject("\"foo\""));
        String escapedPath = "\"/path/to/file.txt\"";
        assertEquals("/path/to/file.txt", PropertiesParser
                .readObject(escapedPath));
    }

    private Map eval(String inString) {
        return new MapProperties(new StringReader(inString));
    }
    
}
