/*
 * Decompiled with CFR 0.152.
 */
package com.jn.langx.util.io.file;

import com.jn.langx.annotation.NonNull;
import com.jn.langx.annotation.Nullable;
import com.jn.langx.exception.FileExistsException;
import com.jn.langx.security.Securitys;
import com.jn.langx.security.privileged.CommonPrivilegedAction;
import com.jn.langx.text.StringTemplates;
import com.jn.langx.util.Maths;
import com.jn.langx.util.Objs;
import com.jn.langx.util.Preconditions;
import com.jn.langx.util.Strings;
import com.jn.langx.util.Throwables;
import com.jn.langx.util.collection.Collects;
import com.jn.langx.util.function.Functions;
import com.jn.langx.util.function.Predicate2;
import com.jn.langx.util.function.Supplier0;
import com.jn.langx.util.io.Charsets;
import com.jn.langx.util.io.IOs;
import com.jn.langx.util.io.LineDelimiter;
import com.jn.langx.util.io.file.FileFilter;
import com.jn.langx.util.io.file.FileIOMode;
import com.jn.langx.util.io.file.FileSystems;
import com.jn.langx.util.io.file.Filenames;
import com.jn.langx.util.io.file.filter.BooleanFileFilter;
import com.jn.langx.util.io.file.filter.IsDirectoryFileFilter;
import com.jn.langx.util.io.file.filter.IsFileFilter;
import com.jn.langx.util.logging.Loggers;
import com.jn.langx.util.net.URLs;
import com.jn.langx.util.os.Platform;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.RandomAccessFile;
import java.io.Writer;
import java.math.BigInteger;
import java.net.URI;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import org.slf4j.Logger;

public class Files {
    private static final long FILE_COPY_BUFFER_SIZE = 0x1E00000L;
    private static final int MAX_BUFFER_SIZE = 0x7FFFFFF7;
    private static final int BUFFER_SIZE = 8192;
    private static final Logger logger = Loggers.getLogger(Files.class);
    private static final File tmpdir = CommonPrivilegedAction.doPrivileged(new Supplier0<File>(){

        @Override
        public File get() {
            return Platform.getTempDirectory();
        }
    });
    private static final SecureRandom random = Securitys.getSecureRandom();

    public static File newFile(@Nullable File directory, String ... names) {
        File file = directory;
        for (String name : names) {
            name = Filenames.normalize(name, true);
            file = file == null ? new File(name) : new File(file, name);
        }
        return file;
    }

    public static File newFile(String ... names) {
        return Files.newFile(null, names);
    }

    public static RandomAccessFile newRandomAccessFile(File file, FileIOMode mode) throws FileNotFoundException {
        return new RandomAccessFile(file, mode.getIdentifier());
    }

    public static RandomAccessFile newRandomAccessFile(String filepath, FileIOMode mode) throws FileNotFoundException {
        Preconditions.checkArgument(Filenames.checkFilePath(filepath), "filepath");
        return new RandomAccessFile(filepath, mode.getIdentifier());
    }

    public static String getSuffix(File file) {
        return Filenames.getSuffix(Files.getCanonicalPath(file));
    }

    public static boolean makeDirs(String dirPath) {
        return Files.makeDirs(new File(dirPath));
    }

    public static boolean makeDirs(File dir) {
        return dir.exists() || dir.mkdirs();
    }

    public static boolean makeFile(File file) throws IOException {
        if (!file.exists()) {
            File dir = file.getParentFile();
            Files.makeDirs(dir);
            if (dir.exists()) {
                return file.createNewFile();
            }
            throw new IOException(StringTemplates.formatWithPlaceholder("Can't create directory: {}", Files.getCanonicalPath(dir)));
        }
        return true;
    }

    public static boolean makeFile(String filePath) throws IOException {
        return Files.makeFile(new File(filePath));
    }

    public static void forceMkdir(File directory) throws IOException {
        if (directory.exists()) {
            if (!directory.isDirectory()) {
                String message = "File " + directory + " exists and is not a directory. Unable to create directory.";
                throw new IOException(message);
            }
        } else if (!directory.mkdirs() && !directory.isDirectory()) {
            String message = "Unable to create directory " + directory;
            throw new IOException(message);
        }
    }

    public static FileInputStream openInputStream(File file) {
        try {
            if (file.exists()) {
                if (file.isDirectory()) {
                    throw new IOException(StringTemplates.formatWithPlaceholder("File '{}' exists but is a directory", file));
                }
                if (!file.canRead()) {
                    throw new IOException(StringTemplates.formatWithPlaceholder("File '{}' cannot be read", file));
                }
            } else {
                throw new FileNotFoundException(StringTemplates.formatWithPlaceholder("File '{}' does not exist", file));
            }
            File f = URLs.getFile(URLs.toURL(file.toURI()));
            if (f != null) {
                return new FileInputStream(f);
            }
            throw new IOException(StringTemplates.formatWithPlaceholder("file not found: {}", file.getPath()));
        }
        catch (IOException e) {
            logger.error(e.getMessage(), (Throwable)e);
            throw Throwables.wrapAsRuntimeIOException(e);
        }
    }

    public static FileOutputStream openOutputStream(File file) throws IOException {
        return Files.openOutputStream(file, false);
    }

    public static FileOutputStream openOutputStream(File file, boolean append) throws IOException {
        if (file.exists()) {
            if (file.isDirectory()) {
                throw new IOException("File '" + file + "' exists but is a directory");
            }
            if (!file.canWrite()) {
                throw new IOException("File '" + file + "' cannot be written to");
            }
        } else if (!Files.makeFile(file)) {
            throw new IOException("File '" + file + "' could not be created");
        }
        return new FileOutputStream(file, append);
    }

    public static void touch(File file) throws IOException {
        boolean success;
        if (!file.exists()) {
            Files.openOutputStream(file).close();
        }
        if (!(success = file.setLastModified(System.currentTimeMillis()))) {
            throw new IOException("Unable to set the last modification time for " + file);
        }
    }

    public static File toFile(URI uri) {
        try {
            return Files.toFile(uri.toURL());
        }
        catch (Throwable ex) {
            throw Throwables.wrapAsRuntimeException(ex);
        }
    }

    public static File toFile(URL url) {
        if (url == null || !"file".equalsIgnoreCase(url.getProtocol())) {
            return null;
        }
        String filename = url.getFile().replace('/', File.separatorChar);
        filename = Files.decodeUrl(filename);
        return new File(filename);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static String decodeUrl(String url) {
        String decoded = url;
        if (url != null && url.indexOf(37) >= 0) {
            int n = url.length();
            StringBuilder buffer = new StringBuilder();
            ByteBuffer bytes = ByteBuffer.allocate(n);
            int i = 0;
            while (i < n) {
                if (url.charAt(i) == '%') {
                    try {
                        do {
                            byte octet = (byte)Integer.parseInt(url.substring(i + 1, i + 3), 16);
                            bytes.put(octet);
                        } while ((i += 3) < n && url.charAt(i) == '%');
                    }
                    catch (RuntimeException runtimeException) {
                    }
                    finally {
                        if (bytes.position() > 0) {
                            bytes.flip();
                            buffer.append(Charsets.UTF_8.decode(bytes).toString());
                            bytes.clear();
                        }
                    }
                }
                buffer.append(url.charAt(i++));
            }
            decoded = buffer.toString();
        }
        return decoded;
    }

    public static void copyFileToDirectory(File srcFile, File destDir) throws IOException {
        Files.copyFileToDirectory(srcFile, destDir, null);
    }

    public static void copyFileToDirectory(File srcFile, File destDir, FileFilter filter) throws IOException {
        Files.copyFileToDirectory(srcFile, destDir, true, filter);
    }

    public static void copyFileToDirectory(File srcFile, File destDir, boolean preserveFileDate, FileFilter filter) throws IOException {
        if (destDir == null) {
            throw new NullPointerException("Destination must not be null");
        }
        if (destDir.exists() && !destDir.isDirectory()) {
            throw new IllegalArgumentException("Destination '" + destDir + "' is not a directory");
        }
        File destFile = new File(destDir, srcFile.getName());
        Files.copyFile(srcFile, destFile, preserveFileDate, filter);
    }

    public static void copyFile(File srcFile, File destFile) throws IOException {
        Files.copyFile(srcFile, destFile, true);
    }

    public static void copyFile(File srcFile, File destFile, boolean preserveFileDate) throws IOException {
        Files.copyFile(srcFile, destFile, preserveFileDate, null);
    }

    public static void copyFile(File srcFile, File destFile, boolean preserveFileDate, FileFilter filter) throws IOException {
        Files.checkFileRequirements(srcFile, destFile);
        if (srcFile.isDirectory()) {
            throw new IOException("Source '" + srcFile + "' exists but is a directory");
        }
        if (srcFile.getCanonicalPath().equals(destFile.getCanonicalPath())) {
            throw new IOException("Source '" + srcFile + "' and destination '" + destFile + "' are the same");
        }
        File parentFile = destFile.getParentFile();
        if (parentFile != null && !parentFile.mkdirs() && !parentFile.isDirectory()) {
            throw new IOException("Destination '" + parentFile + "' directory cannot be created");
        }
        if (destFile.exists() && !destFile.canWrite()) {
            throw new IOException("Destination '" + destFile + "' exists but is read-only");
        }
        if (filter != null && !filter.accept(srcFile)) {
            return;
        }
        Files.doCopyFile(srcFile, destFile, preserveFileDate);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static long copyFile(File input, OutputStream output) throws IOException {
        long l;
        FileInputStream fis = null;
        try {
            fis = new FileInputStream(input);
            l = IOs.copyLarge(fis, output);
        }
        catch (Throwable throwable) {
            IOs.close(fis);
            throw throwable;
        }
        IOs.close(fis);
        return l;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void doCopyFile(File srcFile, File destFile, boolean preserveFileDate) throws IOException {
        boolean actionResult;
        if (destFile.exists() && destFile.isDirectory()) {
            throw new IOException("Destination '" + destFile + "' exists but is a directory");
        }
        FileInputStream fis = null;
        FileChannel input = null;
        FileOutputStream fos = null;
        FileChannel output = null;
        try {
            long remain;
            long count;
            long bytesCopied;
            fis = new FileInputStream(srcFile);
            input = fis.getChannel();
            fos = new FileOutputStream(destFile);
            output = fos.getChannel();
            long size = input.size();
            for (long pos = 0L; pos < size && (bytesCopied = output.transferFrom(input, pos, count = Math.min(remain = size - pos, 0x1E00000L))) != 0L; pos += bytesCopied) {
            }
        }
        catch (Throwable throwable) {
            IOs.close(output);
            IOs.close(input);
            IOs.close(fis);
            IOs.close(fos);
            throw throwable;
        }
        IOs.close(output);
        IOs.close(input);
        IOs.close(fis);
        IOs.close(fos);
        long srcLen = srcFile.length();
        long dstLen = destFile.length();
        if (srcLen != dstLen) {
            throw new IOException("Failed to copy full contents from '" + srcFile + "' to '" + destFile + "' Expected length: " + srcLen + " Actual: " + dstLen);
        }
        if (preserveFileDate && !(actionResult = destFile.setLastModified(srcFile.lastModified()))) {
            logger.error("touch file failed, filepath: {}", (Object)srcFile.getPath());
        }
    }

    public static void copyDirectoryToDirectory(File srcDir, File destDir, FileFilter filter) throws IOException {
        if (srcDir == null) {
            throw new NullPointerException("Source must not be null");
        }
        if (srcDir.exists() && !srcDir.isDirectory()) {
            throw new IllegalArgumentException("Source '" + destDir + "' is not a directory");
        }
        if (destDir == null) {
            throw new NullPointerException("Destination must not be null");
        }
        if (destDir.exists() && !destDir.isDirectory()) {
            throw new IllegalArgumentException("Destination '" + destDir + "' is not a directory");
        }
        Files.copyDirectory(srcDir, new File(destDir, srcDir.getName()), filter, true);
    }

    public static void copyDirectory(File srcDir, File destDir) throws IOException {
        Files.copyDirectory(srcDir, destDir, true);
    }

    public static void copyDirectory(File srcDir, File destDir, boolean preserveFileDate) throws IOException {
        Files.copyDirectory(srcDir, destDir, null, preserveFileDate);
    }

    public static void copyDirectory(File srcDir, File destDir, FileFilter filter) throws IOException {
        Files.copyDirectory(srcDir, destDir, filter, true);
    }

    public static void copyDirectory(File srcDir, File destDir, FileFilter filter, boolean preserveFileDate) throws IOException {
        Files.checkFileRequirements(srcDir, destDir);
        if (!srcDir.isDirectory()) {
            throw new IOException("Source '" + srcDir + "' exists but is not a directory");
        }
        if (srcDir.getCanonicalPath().equals(destDir.getCanonicalPath())) {
            throw new IOException("Source '" + srcDir + "' and destination '" + destDir + "' are the same");
        }
        ArrayList<String> exclusionList = null;
        if (destDir.getCanonicalPath().startsWith(srcDir.getCanonicalPath())) {
            File[] srcFiles;
            File[] fileArray = srcFiles = filter == null ? srcDir.listFiles() : srcDir.listFiles(filter);
            if (Objs.isNotEmpty(srcFiles)) {
                exclusionList = new ArrayList<String>(srcFiles.length);
                for (File srcFile : srcFiles) {
                    File copiedFile = new File(destDir, srcFile.getName());
                    exclusionList.add(copiedFile.getCanonicalPath());
                }
            }
        }
        Files.doCopyDirectory(srcDir, destDir, filter, preserveFileDate, exclusionList);
    }

    private static void checkFileRequirements(File src, File dest) throws FileNotFoundException {
        if (src == null) {
            throw new NullPointerException("Source must not be null");
        }
        if (dest == null) {
            throw new NullPointerException("Destination must not be null");
        }
        if (!src.exists()) {
            throw new FileNotFoundException("Source '" + src + "' does not exist");
        }
    }

    private static void doCopyDirectory(File srcDir, File destDir, FileFilter filter, boolean preserveFileDate, List<String> exclusionList) throws IOException {
        boolean actionResult;
        File[] srcFiles;
        File[] fileArray = srcFiles = filter == null ? srcDir.listFiles() : srcDir.listFiles(filter);
        if (srcFiles == null) {
            throw new IOException("Failed to list contents of " + srcDir);
        }
        if (destDir.exists()) {
            if (!destDir.isDirectory()) {
                throw new IOException("Destination '" + destDir + "' exists but is not a directory");
            }
        } else if (!destDir.mkdirs() && !destDir.isDirectory()) {
            throw new IOException("Destination '" + destDir + "' directory cannot be created");
        }
        if (!destDir.canWrite()) {
            throw new IOException("Destination '" + destDir + "' cannot be written to");
        }
        for (File srcFile : srcFiles) {
            File dstFile = new File(destDir, srcFile.getName());
            if (exclusionList != null && exclusionList.contains(srcFile.getCanonicalPath())) continue;
            if (srcFile.isDirectory()) {
                Files.doCopyDirectory(srcFile, dstFile, filter, preserveFileDate, exclusionList);
                continue;
            }
            Files.doCopyFile(srcFile, dstFile, preserveFileDate);
        }
        if (preserveFileDate && !(actionResult = destDir.setLastModified(srcDir.lastModified()))) {
            logger.error("touch file failed, the filepath: {}", (Object)destDir.getPath());
        }
    }

    public static void copyURLToFile(URL source, File destination) throws IOException {
        FileInputStream fis = null;
        try {
            fis = new FileInputStream(Files.toFile(source));
            Files.copyInputStreamToFile(fis, destination);
        }
        catch (Throwable throwable) {
            IOs.close(fis);
            throw throwable;
        }
        IOs.close(fis);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void copyURLToFile(URL source, File destination, int connectionTimeout, int readTimeout) throws IOException {
        FileInputStream fis = null;
        try {
            fis = new FileInputStream(Files.toFile(source));
            Files.copyInputStreamToFile(fis, destination);
        }
        catch (Throwable throwable) {
            IOs.close(fis);
            throw throwable;
        }
        IOs.close(fis);
    }

    public static void copyInputStreamToFile(InputStream source, File destination) throws IOException {
        Files.copyToFile(source, destination);
    }

    public static void copyToFile(InputStream source, File destination) throws IOException {
        FileOutputStream out = null;
        try {
            out = Files.openOutputStream(destination);
            IOs.copy(source, (OutputStream)out);
        }
        finally {
            IOs.close(out);
        }
    }

    public static void copyToDirectory(File src, File destDir) throws IOException {
        Files.copyToDirectory(src, destDir, null);
    }

    public static void copyToDirectory(File src, File destDir, FileFilter filter) throws IOException {
        if (src == null) {
            throw new NullPointerException("Source must not be null");
        }
        if (src.isFile()) {
            Files.copyFileToDirectory(src, destDir, filter);
        } else if (src.isDirectory()) {
            Files.copyDirectoryToDirectory(src, destDir, filter);
        } else {
            throw new IOException("The source " + src + " does not exist");
        }
    }

    public static void copyToDirectory(Iterable<File> srcs, File destDir) throws IOException {
        Files.copyToDirectory(srcs, destDir, null);
    }

    public static void copyToDirectory(Iterable<File> srcs, File destDir, FileFilter filter) throws IOException {
        if (srcs == null) {
            throw new NullPointerException("Sources must not be null");
        }
        for (File src : srcs) {
            Files.copyToDirectory(src, destDir, filter);
        }
    }

    public static void deleteDirectory(File directory) throws IOException {
        if (!directory.exists()) {
            return;
        }
        if (FileSystems.isNotSymlink(directory)) {
            Files.cleanDirectory(directory);
            boolean actionResult = directory.delete();
            if (!actionResult) {
                logger.warn("delete directory failed, directory path is: {}", (Object)directory.getPath());
            }
        }
        if (directory.exists() && directory.isDirectory()) {
            Files.deleteDirectory(directory);
        }
    }

    public static boolean deleteQuietly(File file) {
        if (file == null) {
            return false;
        }
        try {
            if (file.isDirectory()) {
                Files.cleanDirectory(file);
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        try {
            return file.delete();
        }
        catch (Exception ignored) {
            return false;
        }
    }

    public static void cleanDirectory(File directory) throws IOException {
        if (directory.exists() && directory.isDirectory()) {
            boolean actionResult;
            File[] children = directory.listFiles(new IsFileFilter());
            if (children != null) {
                for (File child : children) {
                    actionResult = child.delete();
                    if (actionResult) continue;
                    logger.warn("delete file failed, file path: {}", (Object)child.getPath());
                }
            }
            if ((children = directory.listFiles(new IsDirectoryFileFilter())) != null) {
                for (File child : children) {
                    Files.cleanDirectory(child);
                    actionResult = child.delete();
                    if (actionResult) continue;
                    logger.warn("delete directory failed, directory path: {}", (Object)child.getPath());
                }
            }
        }
    }

    private static File[] verifiedListFiles(File directory) throws IOException {
        if (!directory.exists()) {
            return new File[0];
        }
        if (!directory.isDirectory()) {
            String message = directory + " is not a directory";
            throw new IllegalArgumentException(message);
        }
        File[] files = directory.listFiles(new IsFileFilter());
        if (files == null) {
            return new File[0];
        }
        return files;
    }

    public static void forceDelete(File file) throws IOException {
        if (file.isDirectory()) {
            Files.deleteDirectory(file);
        } else {
            boolean filePresent = file.exists();
            if (filePresent && !file.delete() && file.exists()) {
                String message = "Unable to delete file: " + file;
                throw new IOException(message);
            }
        }
    }

    public static void forceDeleteOnExit(File file) throws IOException {
        if (file.isDirectory()) {
            Files.deleteDirectoryOnExit(file);
        } else {
            file.deleteOnExit();
        }
    }

    private static void deleteDirectoryOnExit(File directory) throws IOException {
        if (!directory.exists()) {
            return;
        }
        directory.deleteOnExit();
        if (FileSystems.isNotSymlink(directory)) {
            Files.cleanDirectoryOnExit(directory);
        }
    }

    private static void cleanDirectoryOnExit(File directory) throws IOException {
        File[] files = Files.verifiedListFiles(directory);
        IOException exception = null;
        for (File file : files) {
            try {
                Files.forceDeleteOnExit(file);
            }
            catch (IOException ioe) {
                exception = ioe;
            }
        }
        if (null != exception) {
            throw exception;
        }
    }

    public static void forceMakeDirParent(File file) throws IOException {
        File parent = file.getParentFile();
        if (parent == null) {
            return;
        }
        Files.forceMkdir(parent);
    }

    public static long sizeOf(File file) {
        if (!file.exists()) {
            String message = file + " does not exist";
            throw new IllegalArgumentException(message);
        }
        if (file.isDirectory()) {
            return Files.sizeOfDirectory0(file);
        }
        return file.length();
    }

    public static BigInteger sizeOfAsBigInteger(File file) {
        if (!file.exists()) {
            String message = file + " does not exist";
            throw new IllegalArgumentException(message);
        }
        if (file.isDirectory()) {
            return Files.sizeOfDirectoryBig0(file);
        }
        return BigInteger.valueOf(file.length());
    }

    public static long sizeOfDirectory(File directory) {
        Files.checkDirectory(directory);
        return Files.sizeOfDirectory0(directory);
    }

    private static long sizeOfDirectory0(File directory) {
        File[] files = directory.listFiles();
        if (files == null) {
            return 0L;
        }
        long size = 0L;
        for (File file : files) {
            try {
                if (!FileSystems.isNotSymlink(file) || (size += Files.sizeOf0(file)) >= 0L) continue;
                break;
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        return size;
    }

    private static long sizeOf0(File file) {
        if (file.isDirectory()) {
            return Files.sizeOfDirectory0(file);
        }
        return file.length();
    }

    public static BigInteger sizeOfDirectoryAsBigInteger(File directory) {
        Files.checkDirectory(directory);
        return Files.sizeOfDirectoryBig0(directory);
    }

    private static BigInteger sizeOfDirectoryBig0(File directory) {
        File[] files = directory.listFiles();
        if (files == null) {
            return BigInteger.ZERO;
        }
        BigInteger size = BigInteger.ZERO;
        for (File file : files) {
            try {
                if (!FileSystems.isNotSymlink(file)) continue;
                size = size.add(Files.sizeOfBig0(file));
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        return size;
    }

    private static BigInteger sizeOfBig0(File fileOrDir) {
        if (fileOrDir.isDirectory()) {
            return Files.sizeOfDirectoryBig0(fileOrDir);
        }
        return BigInteger.valueOf(fileOrDir.length());
    }

    private static void checkDirectory(File directory) {
        if (!directory.exists()) {
            throw new IllegalArgumentException(directory + " does not exist");
        }
        if (!directory.isDirectory()) {
            throw new IllegalArgumentException(directory + " is not a directory");
        }
    }

    public static boolean isFileNewer(File file, File reference) {
        if (reference == null) {
            throw new IllegalArgumentException("No specified reference file");
        }
        if (!reference.exists()) {
            throw new IllegalArgumentException("The reference file '" + reference + "' doesn't exist");
        }
        return Files.isFileNewer(file, reference.lastModified());
    }

    public static boolean isFileNewer(File file, Date date) {
        if (date == null) {
            throw new IllegalArgumentException("No specified date");
        }
        return Files.isFileNewer(file, date.getTime());
    }

    public static boolean isFileNewer(File file, long timeMillis) {
        if (file == null) {
            throw new IllegalArgumentException("No specified file");
        }
        return file.exists() && file.lastModified() > timeMillis;
    }

    public static boolean isFileOlder(File file, Date date) {
        if (date == null) {
            throw new IllegalArgumentException("No specified date");
        }
        return Files.isFileOlder(file, date.getTime());
    }

    public static boolean isFileOlder(File file, long timeMillis) {
        if (file == null) {
            throw new IllegalArgumentException("No specified file");
        }
        return file.exists() && file.lastModified() < timeMillis;
    }

    public static void moveDirectory(File srcDir, File destDir) throws IOException {
        if (srcDir == null) {
            throw new NullPointerException("Source must not be null");
        }
        if (destDir == null) {
            throw new NullPointerException("Destination must not be null");
        }
        if (!srcDir.exists()) {
            throw new FileNotFoundException("Source '" + srcDir + "' does not exist");
        }
        if (!srcDir.isDirectory()) {
            throw new IOException("Source '" + srcDir + "' is not a directory");
        }
        if (destDir.exists()) {
            throw new FileExistsException("Destination '" + destDir + "' already exists");
        }
        boolean rename = srcDir.renameTo(destDir);
        if (!rename) {
            if (destDir.getCanonicalPath().startsWith(srcDir.getCanonicalPath() + File.separator)) {
                throw new IOException("Cannot move directory: " + srcDir + " to a subdirectory of itself: " + destDir);
            }
            Files.copyDirectory(srcDir, destDir);
            Files.deleteDirectory(srcDir);
            if (srcDir.exists()) {
                throw new IOException("Failed to delete original directory '" + srcDir + "' after copy to '" + destDir + "'");
            }
        }
    }

    public static void moveDirectoryToDirectory(File src, File destDir, boolean createDestDir) throws IOException {
        if (src == null) {
            throw new NullPointerException("Source must not be null");
        }
        if (destDir == null) {
            throw new NullPointerException("Destination directory must not be null");
        }
        if (!destDir.exists() && createDestDir) {
            Files.makeDirs(destDir);
        }
        if (!destDir.exists()) {
            throw new FileNotFoundException(StringTemplates.formatWithPlaceholder("Destination directory '{}' does not exist [createDestDir={}]", destDir, createDestDir));
        }
        if (!destDir.isDirectory()) {
            throw new IOException("Destination '" + destDir + "' is not a directory");
        }
        Files.moveDirectory(src, new File(destDir, src.getName()));
    }

    public static void moveFile(File srcFile, File destFile) throws IOException {
        if (srcFile == null) {
            throw new NullPointerException("Source must not be null");
        }
        if (destFile == null) {
            throw new NullPointerException("Destination must not be null");
        }
        if (!srcFile.exists()) {
            throw new FileNotFoundException("Source '" + srcFile + "' does not exist");
        }
        if (srcFile.isDirectory()) {
            throw new IOException("Source '" + srcFile + "' is a directory");
        }
        if (destFile.exists()) {
            throw new FileExistsException("Destination '" + destFile + "' already exists");
        }
        if (destFile.isDirectory()) {
            throw new IOException("Destination '" + destFile + "' is a directory");
        }
        boolean rename = srcFile.renameTo(destFile);
        if (!rename) {
            Files.copyFile(srcFile, destFile);
            if (!srcFile.delete()) {
                Files.deleteQuietly(destFile);
                throw new IOException(StringTemplates.formatWithPlaceholder("Failed to delete original file '{}' after copy to '{}'", srcFile, destFile));
            }
        }
    }

    public static void moveFileToDirectory(File srcFile, File destDir, boolean createDestDir) throws IOException {
        if (srcFile == null) {
            throw new NullPointerException("Source must not be null");
        }
        if (destDir == null) {
            throw new NullPointerException("Destination directory must not be null");
        }
        if (!destDir.exists() && createDestDir) {
            Files.makeDirs(destDir);
        }
        if (!destDir.exists()) {
            throw new FileNotFoundException("Destination directory '" + destDir + "' does not exist [createDestDir=" + createDestDir + "]");
        }
        if (!destDir.isDirectory()) {
            throw new IOException("Destination '" + destDir + "' is not a directory");
        }
        Files.moveFile(srcFile, new File(destDir, srcFile.getName()));
    }

    public static void moveToDirectory(File src, File destDir, boolean createDestDir) throws IOException {
        if (src == null) {
            throw new NullPointerException("Source must not be null");
        }
        if (destDir == null) {
            throw new NullPointerException("Destination must not be null");
        }
        if (!src.exists()) {
            throw new FileNotFoundException("Source '" + src + "' does not exist");
        }
        if (src.isDirectory()) {
            Files.moveDirectoryToDirectory(src, destDir, createDestDir);
        } else {
            Files.moveFileToDirectory(src, destDir, createDestDir);
        }
    }

    public static boolean exists(File file) {
        return file != null && file.exists();
    }

    public static boolean isReadable(File file) {
        return file != null && file.canRead();
    }

    public static boolean isWritable(File file) {
        return file != null && file.canWrite();
    }

    public static String readAsText(File file) throws IOException {
        return Files.readAsText(file, Charsets.UTF_8);
    }

    public static String readAsText(File file, Charset charset) throws IOException {
        byte[] bytes = Files.readAllBytes(file);
        String str = Strings.newString(bytes, charset == null ? Charsets.UTF_8 : charset);
        return str;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static byte[] readAllBytes(File file) throws IOException {
        byte[] byArray;
        FileInputStream inputStream = null;
        try {
            inputStream = new FileInputStream(file);
            long filesize = file.length();
            if (filesize > 0x7FFFFFF7L) {
                throw new OutOfMemoryError("Required array size too large");
            }
            byArray = Files.read(inputStream, (int)filesize);
        }
        catch (Throwable throwable) {
            IOs.close(inputStream);
            throw throwable;
        }
        IOs.close(inputStream);
        return byArray;
    }

    private static byte[] read(InputStream source, int initialSize) throws IOException {
        int capacity = initialSize;
        byte[] buf = new byte[capacity];
        int nread = 0;
        while (true) {
            int n;
            if ((n = source.read(buf, nread, capacity - nread)) > 0) {
                nread += n;
                continue;
            }
            if (n < 0 || (n = source.read()) < 0) break;
            if (capacity <= 0x7FFFFFF7 - capacity) {
                capacity = Math.max(capacity << 1, 8192);
            } else {
                if (capacity == 0x7FFFFFF7) {
                    throw new OutOfMemoryError("Required array size too large");
                }
                capacity = 0x7FFFFFF7;
            }
            buf = Arrays.copyOf(buf, capacity);
            buf[nread++] = (byte)n;
        }
        return capacity == nread ? buf : Arrays.copyOf(buf, nread);
    }

    public static void write(String string, File file) throws IOException {
        Files.write(string, Charsets.getDefault(), file, false);
    }

    public static void write(String string, File file, boolean append) throws IOException {
        Files.write(string, Charsets.getDefault(), file, append);
    }

    public static void write(String string, Charset charset, File file) throws IOException {
        Files.write(string, charset, file, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void write(String string, Charset charset, File file, boolean append) throws IOException {
        BufferedWriter fileWriter = null;
        try {
            fileWriter = new BufferedWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(file, append), charset));
            IOs.write(string, (Writer)fileWriter);
        }
        catch (Throwable throwable) {
            IOs.close(fileWriter);
            throw throwable;
        }
        IOs.close(fileWriter);
    }

    public static void appendLine(File file, String line) throws IOException {
        Files.write(line + LineDelimiter.DEFAULT.getValue(), Charsets.UTF_8, file, true);
    }

    public static void appendLine(File file, String line, Charset charset) throws IOException {
        Files.write(line + LineDelimiter.DEFAULT.getValue(), charset, file, true);
    }

    public static List<String> lines(File file) throws IOException {
        return IOs.readLines(file);
    }

    public static List<String> lines(File file, Charset charset) throws IOException {
        return IOs.readLines(file, charset);
    }

    public static List<File> find(File directory, @Nullable FileFilter childrenFilter, FileFilter filter, Predicate2<List<File>, File> breakPredicate) {
        return Files.find(directory, 1, childrenFilter, filter, breakPredicate);
    }

    public static List<File> find(File directory, int maxDepth, @Nullable FileFilter childrenFilter, FileFilter filter, Predicate2<List<File>, File> breakPredicate) {
        List<File> out = Collects.emptyArrayList();
        Files.find(directory, out, maxDepth, childrenFilter, filter, breakPredicate);
        return out;
    }

    public static void find(@NonNull File directory, @NonNull List<File> out, int maxDepth, @Nullable FileFilter childrenFilter, @Nullable FileFilter filter, @NonNull Predicate2<List<File>, File> breakPredicate) {
        File[] children;
        if (directory == null || out == null || maxDepth < 1) {
            return;
        }
        maxDepth = Maths.min(maxDepth, 100);
        if (filter == null) {
            filter = BooleanFileFilter.TRUE_FILTER;
        }
        if (breakPredicate == null) {
            breakPredicate = Functions.booleanPredicate2(false);
        }
        File[] fileArray = children = childrenFilter == null ? directory.listFiles() : directory.listFiles(childrenFilter);
        if (children != null) {
            for (File child : children) {
                if (breakPredicate.test(out, child)) break;
                if (filter.test(child)) {
                    out.add(child);
                }
                if (!child.isDirectory()) continue;
                Files.find(child, out, maxDepth - 1, childrenFilter, filter, breakPredicate);
            }
        }
    }

    public static String getCanonicalPath(File file) {
        if (file == null) {
            return null;
        }
        try {
            return file.getCanonicalPath();
        }
        catch (IOException e) {
            throw Throwables.wrapAsRuntimeIOException(e);
        }
    }

    public static File getCanonicalFile(File file) {
        try {
            return file.getCanonicalFile();
        }
        catch (IOException e) {
            throw Throwables.wrapAsRuntimeIOException(e);
        }
    }

    public static void setLastModified(File file) {
        Files.setLastModified(file, System.currentTimeMillis());
    }

    public static void setLastModified(File file, long time) {
        boolean actionResult = file.setLastModified(time);
        if (!actionResult) {
            Loggers.getLogger(Files.class).warn("touch file failed, file path: {}", (Object)file.getPath());
        }
    }

    public static long getLastModified(File file) {
        return file.lastModified();
    }

    private Files() {
    }

    private static File generateTempPath(@Nullable String prefix, @Nullable String suffix, @Nullable File dir) {
        long n = random.nextLong();
        n = n == Long.MIN_VALUE ? 0L : Math.abs(n);
        File f = Files.newFile(dir, prefix + "" + n + suffix);
        return f;
    }

    private static File createTemp(File dir, String prefix, String suffix, boolean createDirectory) throws IOException {
        if (prefix == null) {
            prefix = "";
        }
        if (suffix == null) {
            String string = suffix = createDirectory ? "" : ".tmp";
        }
        if (dir == null) {
            dir = tmpdir;
        }
        File f = Files.generateTempPath(prefix, suffix, dir);
        while (f.exists()) {
            f = Files.generateTempPath(prefix, suffix, dir);
        }
        if (createDirectory) {
            Files.forceMkdir(f);
            return f;
        }
        Files.makeFile(f);
        return f;
    }

    public static File createTempFile(@Nullable File dir, @Nullable String prefix, @Nullable String suffix) throws IOException {
        return Files.createTemp(dir, prefix, suffix, false);
    }

    public static File createTempDirectory(@Nullable File dir, @Nullable String prefix, @Nullable String suffix) throws IOException {
        return Files.createTemp(dir, prefix, suffix, true);
    }
}

