/*
 * Decompiled with CFR 0.152.
 */
package io.github.bonigarcia.wdm.docker;

import com.github.dockerjava.api.DockerClient;
import com.github.dockerjava.api.async.ResultCallback;
import com.github.dockerjava.api.command.AsyncDockerCmd;
import com.github.dockerjava.api.command.CreateContainerCmd;
import com.github.dockerjava.api.command.CreateContainerResponse;
import com.github.dockerjava.api.command.CreateNetworkCmd;
import com.github.dockerjava.api.command.CreateNetworkResponse;
import com.github.dockerjava.api.command.ExecCreateCmdResponse;
import com.github.dockerjava.api.command.ExecStartCmd;
import com.github.dockerjava.api.command.LogContainerCmd;
import com.github.dockerjava.api.exception.DockerException;
import com.github.dockerjava.api.model.Bind;
import com.github.dockerjava.api.model.Capability;
import com.github.dockerjava.api.model.ContainerNetwork;
import com.github.dockerjava.api.model.ExposedPort;
import com.github.dockerjava.api.model.Frame;
import com.github.dockerjava.api.model.HostConfig;
import com.github.dockerjava.api.model.Mount;
import com.github.dockerjava.api.model.PortBinding;
import com.github.dockerjava.api.model.Ports;
import com.github.dockerjava.api.model.PullResponseItem;
import com.github.dockerjava.api.model.TmpfsOptions;
import com.github.dockerjava.core.DefaultDockerClientConfig;
import com.github.dockerjava.core.DockerClientBuilder;
import com.github.dockerjava.core.DockerClientConfig;
import com.github.dockerjava.httpclient5.ApacheDockerHttpClient;
import com.github.dockerjava.transport.DockerHttpClient;
import io.github.bonigarcia.wdm.WebDriverManager;
import io.github.bonigarcia.wdm.cache.ResolutionCache;
import io.github.bonigarcia.wdm.config.Architecture;
import io.github.bonigarcia.wdm.config.Config;
import io.github.bonigarcia.wdm.config.DriverManagerType;
import io.github.bonigarcia.wdm.config.WebDriverManagerException;
import io.github.bonigarcia.wdm.docker.DockerContainer;
import io.github.bonigarcia.wdm.docker.DockerHost;
import io.github.bonigarcia.wdm.versions.Shell;
import io.github.bonigarcia.wdm.versions.VersionComparator;
import io.github.bonigarcia.wdm.versions.VersionDetector;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DockerService {
    final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    public static final String NETWORK_HOST = "host";
    public static final String NETWORK_DRIVER = "bridge";
    public static final String CACHE_KEY_LABEL = "-container-";
    public static final String CACHE_KEY_CUSTOM = "custom";
    private static final String SELENIUM_IMAGE_LABEL = "selenium";
    private static final String SELENIARM_IMAGE_LABEL = "seleniarm";
    private static final String DEFAULT_GATEWAY = "172.17.0.1";
    private static final String BETA = "beta";
    private static final String DEV = "dev";
    private static final String LATEST_MINUS = "latest-";
    private static final String RECORDING_EXT = ".mp4";
    private static final String SEPARATOR = "_";
    private static final String DATE_FORMAT = "yyyy.MM.dd_HH.mm.ss.SSS";
    private static final int POLL_TIME_MSEC = 500;
    private Config config;
    private DockerClient dockerClient;
    private ResolutionCache resolutionCache;
    private URI dockerHostUri;

    public DockerService(Config config, ResolutionCache resolutionCache) {
        this.config = config;
        this.resolutionCache = resolutionCache;
        if (config.isDockerLocalFallback() && !this.isRunningInsideDocker() && !WebDriverManager.isDockerAvailable()) {
            this.log.warn("Docker is not available in your machine... local browsers are used instead");
        } else {
            this.dockerClient = this.createDockerClient();
        }
    }

    private DockerClient createDockerClient() {
        String dockerDaemonUrl = this.config.getDockerDaemonUrl();
        String dockerHost = Config.isNullOrEmpty(dockerDaemonUrl) ? DockerHost.fromEnv().endpoint() : dockerDaemonUrl;
        return this.getDockerClient(dockerHost);
    }

    private DockerClient getDockerClient(String dockerHost) {
        DefaultDockerClientConfig.Builder dockerClientConfigBuilder = DefaultDockerClientConfig.createDefaultConfigBuilder();
        if (!Config.isNullOrEmpty(dockerHost)) {
            dockerClientConfigBuilder.withDockerHost(dockerHost);
        }
        DefaultDockerClientConfig dockerClientConfig = dockerClientConfigBuilder.build();
        this.dockerHostUri = dockerClientConfig.getDockerHost();
        ApacheDockerHttpClient dockerHttpClient = new ApacheDockerHttpClient.Builder().dockerHost(this.dockerHostUri).build();
        return DockerClientBuilder.getInstance((DockerClientConfig)dockerClientConfig).withDockerHttpClient((DockerHttpClient)dockerHttpClient).build();
    }

    public String getHost(String containerId, String network) {
        String host = this.getDefaultHost();
        if (SystemUtils.IS_OS_LINUX && this.isRunningInsideDocker()) {
            host = this.getGateway(containerId, network);
            this.log.debug("WebDriverManager running inside a Docker container. Using gateway address: {}", (Object)host);
        }
        return host;
    }

    private boolean isCommandResultPresent(String command) {
        String[] commandArray = new String[]{"bash", "-c", command};
        String commandOutput = Shell.runAndWait(false, commandArray);
        return !Config.isNullOrEmpty(commandOutput);
    }

    public boolean isRunningInsideDocker() {
        return this.isCommandResultPresent("cat /proc/self/cgroup | grep docker") || this.isCommandResultPresent("cat /proc/self/mountinfo | grep docker/containers");
    }

    public String getDefaultHost() {
        return Optional.ofNullable(this.dockerHostUri.getHost()).orElse(DockerHost.defaultAddress());
    }

    public String getGateway(String containerId, String network) {
        String gateway = ((ContainerNetwork)this.dockerClient.inspectContainerCmd(containerId).exec().getNetworkSettings().getNetworks().get(network)).getGateway();
        if (Config.isNullOrEmpty(gateway)) {
            return DEFAULT_GATEWAY;
        }
        return gateway;
    }

    public String getAddress(String containerId, String network) {
        return ((ContainerNetwork)this.dockerClient.inspectContainerCmd(containerId).exec().getNetworkSettings().getNetworks().get(network)).getIpAddress();
    }

    public synchronized String startContainer(DockerContainer dockerContainer) throws DockerException {
        String containerId;
        String imageId = dockerContainer.getImageId();
        this.log.info("Starting Docker container {}", (Object)imageId);
        HostConfig hostConfigBuilder = new HostConfig();
        try (CreateContainerCmd containerConfigBuilder = this.dockerClient.createContainerCmd(imageId);){
            Optional<List<String>> entryPoint;
            Optional<List<String>> cmd;
            Optional<List<String>> envs;
            Optional<List<Mount>> mounts;
            Optional<List<Bind>> binds;
            List<String> exposedPorts;
            Optional<String> network;
            Optional<Long> shmSize;
            boolean privileged;
            Optional<String> containerName = dockerContainer.getContainerName();
            if (containerName.isPresent()) {
                this.log.trace("Using container name: {}", (Object)containerName.get());
                containerConfigBuilder.withName(containerName.get());
            }
            if (privileged = dockerContainer.isPrivileged()) {
                this.log.trace("Using privileged mode");
                hostConfigBuilder.withPrivileged(Boolean.valueOf(true));
            }
            if (dockerContainer.isSysadmin()) {
                this.log.trace("Adding sysadmin capability");
                hostConfigBuilder.withCapAdd(new Capability[]{Capability.SYS_ADMIN});
            }
            if ((shmSize = dockerContainer.getShmSize()).isPresent()) {
                this.log.trace("Using shm size: {}", (Object)shmSize.get());
                hostConfigBuilder.withShmSize(shmSize.get());
            }
            if ((network = dockerContainer.getNetwork()).isPresent()) {
                String dockerNetwork = network.get();
                this.createDockerNetworkIfNotExists(dockerNetwork);
                this.log.trace("Using network: {}", (Object)dockerNetwork);
                hostConfigBuilder.withNetworkMode(dockerNetwork);
            }
            if (!(exposedPorts = dockerContainer.getExposedPorts()).isEmpty()) {
                this.log.trace("Using exposed ports: {}", exposedPorts);
                containerConfigBuilder.withExposedPorts(exposedPorts.stream().map(ExposedPort::parse).collect(Collectors.toList()));
                hostConfigBuilder.withPortBindings(exposedPorts.stream().map(PortBinding::parse).collect(Collectors.toList()));
                hostConfigBuilder.withPublishAllPorts(Boolean.valueOf(true));
            }
            if ((binds = dockerContainer.getBinds()).isPresent()) {
                this.log.trace("Using binds: {}", binds.get());
                hostConfigBuilder.withBinds(binds.get());
            }
            if ((mounts = dockerContainer.getMounts()).isPresent()) {
                this.log.trace("Using mounts: {}", mounts.get());
                hostConfigBuilder.withMounts(mounts.get());
            }
            if ((envs = dockerContainer.getEnvs()).isPresent()) {
                this.log.trace("Using envs: {}", envs.get());
                containerConfigBuilder.withEnv(envs.get().toArray(new String[0]));
            }
            if ((cmd = dockerContainer.getCmd()).isPresent()) {
                this.log.trace("Using cmd: {}", cmd.get());
                containerConfigBuilder.withCmd(cmd.get().toArray(new String[0]));
            }
            if ((entryPoint = dockerContainer.getEntryPoint()).isPresent()) {
                this.log.trace("Using entryPoint: {}", entryPoint.get());
                containerConfigBuilder.withEntrypoint(entryPoint.get().toArray(new String[0]));
            }
            hostConfigBuilder.withExtraHosts(dockerContainer.getExtraHosts());
            containerId = containerConfigBuilder.withHostConfig(hostConfigBuilder).exec().getId();
            this.dockerClient.startContainerCmd(containerId).exec();
        }
        return containerId;
    }

    public void createDockerNetworkIfNotExists(String networkName) {
        List networks = (List)this.dockerClient.listNetworksCmd().exec();
        boolean networkExists = networks.stream().anyMatch(network -> network.getName().equals(networkName));
        if (networkExists) {
            this.log.trace("Docker network {} already exits", (Object)networkName);
        } else {
            try (CreateNetworkCmd networkCmd = this.dockerClient.createNetworkCmd();){
                CreateNetworkResponse networkResponse = (CreateNetworkResponse)networkCmd.withName(networkName).withDriver(NETWORK_DRIVER).exec();
                this.log.trace("Docker network {} created with id {}", (Object)networkName, (Object)networkResponse.getId());
            }
        }
    }

    public String execCommandInContainer(String containerId, String ... command) {
        String commandStr = Arrays.toString(command);
        this.log.trace("Running command {} in container {}", (Object)commandStr, (Object)containerId);
        String execId = ((ExecCreateCmdResponse)this.dockerClient.execCreateCmd(containerId).withCmd(command).withAttachStdout(Boolean.valueOf(true)).withAttachStderr(Boolean.valueOf(true)).exec()).getId();
        ExecStartCmd execStartCmd = this.dockerClient.execStartCmd(execId);
        return this.getOutputFromCmd((AsyncDockerCmd<?, Frame>)execStartCmd);
    }

    public String getOutputFromCmd(AsyncDockerCmd<?, Frame> execStartCmd) {
        final StringBuilder output = new StringBuilder();
        try {
            (execStartCmd.exec((ResultCallback)new ResultCallback.Adapter<Frame>(){

                public void onNext(Frame object) {
                    output.append(new String(object.getPayload(), StandardCharsets.UTF_8));
                    super.onNext((Object)object);
                }
            })).awaitCompletion();
        }
        catch (InterruptedException e) {
            this.log.error("Exception executing command on container {}", (Object)e.getMessage());
            Thread.currentThread().interrupt();
        }
        return output.toString().trim();
    }

    public String getBindPort(String containerId, String exposed) throws DockerException {
        String bindPort = null;
        int waitTimeoutSec = this.config.getTimeout();
        long timeoutMs = System.currentTimeMillis() + Duration.ofSeconds(waitTimeoutSec).toMillis();
        do {
            Ports ports = this.dockerClient.inspectContainerCmd(containerId).exec().getNetworkSettings().getPorts();
            Ports.Binding[] exposedPort = (Ports.Binding[])ports.getBindings().get(ExposedPort.parse((String)exposed));
            this.log.trace("Port list {} -- Exposed port {} = {}", new Object[]{ports, exposed, exposedPort});
            if (ports.getBindings() == null || exposedPort == null || ports.getBindings().isEmpty() || exposedPort.length == 0) {
                String dockerImage = this.dockerClient.inspectContainerCmd(containerId).exec().getConfig().getImage();
                if (System.currentTimeMillis() > timeoutMs) {
                    throw new WebDriverManagerException("Timeout of " + waitTimeoutSec + " getting bind port in container " + dockerImage);
                }
                try {
                    this.log.trace("Port {} is not bindable in container {}", (Object)exposed, (Object)dockerImage);
                    Thread.sleep(500L);
                }
                catch (InterruptedException e) {
                    this.log.warn("Interrupted exception getting bind port", (Throwable)e);
                    Thread.currentThread().interrupt();
                }
                continue;
            }
            bindPort = exposedPort[0].getHostPortSpec();
        } while (bindPort == null);
        return bindPort;
    }

    public void pullImageIfNecessary(String cacheKey, String imageId, String imageVersion) throws DockerException {
        if (!this.config.getDockerAvoidPulling() && !this.resolutionCache.checkKeyInResolutionCache(cacheKey)) {
            try {
                this.log.info("Pulling Docker image {} (this might take some time, but only the first time)", (Object)imageId);
                (this.dockerClient.pullImageCmd(imageId).exec((ResultCallback)new ResultCallback.Adapter<PullResponseItem>(){})).awaitCompletion();
                this.log.trace("Docker image {} pulled", (Object)imageId);
                if (!this.config.isAvoidResolutionCache()) {
                    this.resolutionCache.putValueInResolutionCacheIfEmpty(cacheKey, imageVersion, this.config.getTtlForBrowsers());
                }
            }
            catch (Exception e) {
                this.log.warn("Exception pulling image {}: {}", (Object)imageId, (Object)e.getMessage());
            }
        }
    }

    public synchronized void stopAndRemoveContainer(DockerContainer dockerContainer) {
        String containerId = dockerContainer.getContainerId();
        String imageId = dockerContainer.getImageId();
        this.log.info("Stopping Docker container {}", (Object)imageId);
        try {
            this.stopContainer(containerId);
            this.removeContainer(containerId);
        }
        catch (Exception e) {
            this.log.warn("Exception stopping container {}", (Object)imageId, (Object)e);
        }
    }

    public synchronized void stopContainer(String containerId) throws DockerException {
        int stopTimeoutSec = this.config.getDockerStopTimeoutSec();
        if (stopTimeoutSec == 0) {
            this.log.trace("Killing container {}", (Object)containerId);
            this.dockerClient.killContainerCmd(containerId).exec();
        } else {
            this.log.trace("Stopping container {} (timeout {} seconds)", (Object)containerId, (Object)stopTimeoutSec);
            this.dockerClient.stopContainerCmd(containerId).withTimeout(Integer.valueOf(stopTimeoutSec)).exec();
        }
    }

    public synchronized void removeContainer(String containerId) throws DockerException {
        this.log.trace("Removing container {}", (Object)containerId);
        int stopTimeoutSec = this.config.getDockerStopTimeoutSec();
        if (stopTimeoutSec == 0) {
            this.dockerClient.removeContainerCmd(containerId).withForce(Boolean.valueOf(true)).exec();
        } else {
            this.dockerClient.removeContainerCmd(containerId).exec();
        }
    }

    public void close() throws IOException {
        this.dockerClient.close();
    }

    public void updateDockerClient(String dockerHost) {
        this.log.debug("Updating Docker client using {}", (Object)dockerHost);
        this.dockerClient = this.getDockerClient(dockerHost);
    }

    public String getDockerImageVersion(DriverManagerType driverManagerType, String cacheKey, String browserName, String browserVersion) {
        String latestVersion = "latest";
        int minusIndex = this.getMinusIndex(browserVersion);
        if (minusIndex != 0) {
            if (!this.resolutionCache.checkKeyInResolutionCache(cacheKey, false)) {
                String dockerImage = this.getDockerImage(browserName, latestVersion);
                String cacheKeyLatest = DockerService.getCacheKey(browserName, latestVersion);
                String browserVersionFromContainer = this.getBrowserVersionFromContainer(driverManagerType, cacheKeyLatest, latestVersion, dockerImage);
                int majorBrowserVersion = Integer.parseInt(VersionDetector.getMajorVersion(browserVersionFromContainer));
                latestVersion = String.valueOf(majorBrowserVersion - minusIndex) + ".0";
                if (!this.resolutionCache.checkKeyInResolutionCache(cacheKey, false)) {
                    dockerImage = this.getDockerImage(browserName, latestVersion);
                    this.pullImageIfNecessary(cacheKey, dockerImage, latestVersion);
                }
            } else {
                latestVersion = this.resolutionCache.getValueFromResolutionCache(cacheKey);
            }
        }
        return latestVersion;
    }

    public static String getCacheKey(String browserName, String browserVersion) {
        return browserName + CACHE_KEY_LABEL + browserVersion;
    }

    public String getBrowserVersionFromContainer(DriverManagerType driverManagerType, String cacheKey, String browserVersion, String dockerImage) {
        String browserVersionFromContainer = "latest";
        try {
            this.pullImageIfNecessary(cacheKey, dockerImage, browserVersion);
            ArrayList<String> cmd = new ArrayList<String>();
            switch (driverManagerType) {
                case CHROME: {
                    cmd.add("google-chrome");
                    break;
                }
                case FIREFOX: {
                    cmd.add("firefox");
                    break;
                }
                case EDGE: {
                    cmd.add("microsoft-edge");
                    break;
                }
                default: {
                    throw new WebDriverManagerException(driverManagerType.getBrowserName() + " is not available as Docker container");
                }
            }
            cmd.add("--version");
            try (CreateContainerCmd containerCmd = this.dockerClient.createContainerCmd(dockerImage);){
                CreateContainerResponse container = containerCmd.withCmd(cmd).withHostConfig(HostConfig.newHostConfig().withAutoRemove(Boolean.valueOf(true))).exec();
                this.dockerClient.startContainerCmd(container.getId()).exec();
                LogContainerCmd logContainerCmd = this.dockerClient.logContainerCmd(container.getId()).withStdOut(Boolean.valueOf(true)).withFollowStream(Boolean.valueOf(true));
                browserVersionFromContainer = VersionDetector.parseVersion(this.getOutputFromCmd((AsyncDockerCmd<?, Frame>)logContainerCmd));
            }
        }
        catch (Exception e) {
            this.log.warn("Exception discovering browser version from container: {}", (Object)e.getMessage());
        }
        return browserVersionFromContainer;
    }

    public int getMinusIndex(String browserVersion) {
        int minusIndex = 0;
        if (this.isBrowserVersionLatestMinus(browserVersion)) {
            minusIndex = Integer.parseInt(browserVersion.substring(browserVersion.indexOf(LATEST_MINUS) + LATEST_MINUS.length()));
        }
        return minusIndex;
    }

    public String getDockerImage(String browserName, String browserVersion) {
        String dockerImageFormat = this.getDockerImageFormat();
        String imageLabel = this.config.getArchitecture() == Architecture.ARM64 ? SELENIARM_IMAGE_LABEL : SELENIUM_IMAGE_LABEL;
        String dockerImage = String.format(dockerImageFormat, imageLabel, browserName, browserVersion);
        this.log.trace("Docker image: {}", (Object)dockerImage);
        return dockerImage;
    }

    public String getDockerImageFormat() {
        return this.config.getDockerBrowserImageFormat();
    }

    public boolean isBrowserVersionWildCard(String browserVersion) {
        return this.isBrowserVersionBetaOrDev(browserVersion) || this.isBrowserVersionLatestMinus(browserVersion);
    }

    public boolean isBrowserVersionBetaOrDev(String browserVersion) {
        return browserVersion.equalsIgnoreCase(BETA) || browserVersion.equalsIgnoreCase(DEV);
    }

    public boolean isBrowserVersionLatestMinus(String browserVersion) {
        return browserVersion.toLowerCase(Locale.ROOT).contains(LATEST_MINUS);
    }

    private String getPrefixedDockerImage(String dockerImage) {
        String newDockerImage = dockerImage;
        String prefix = this.config.getDockerPrivateEndpoint();
        if (StringUtils.isNotBlank((CharSequence)prefix)) {
            newDockerImage = String.format("%s/%s", prefix, dockerImage);
        }
        return newDockerImage;
    }

    public DockerContainer startNoVncContainer(String dockerImage, String cacheKey, String browserVersion, DockerContainer browserContainer) {
        dockerImage = this.getPrefixedDockerImage(dockerImage);
        this.pullImageIfNecessary(cacheKey, dockerImage, browserVersion);
        ArrayList<String> exposedPorts = new ArrayList<String>();
        String dockerNoVncPort = String.valueOf(this.config.getDockerNoVncPort());
        exposedPorts.add(dockerNoVncPort);
        ArrayList<String> envs = new ArrayList<String>();
        envs.add("AUTOCONNECT=true");
        envs.add("VIEW_ONLY=" + this.config.isDockerViewOnly());
        envs.add("VNC_PASSWORD=" + this.config.getDockerVncPassword());
        String vncAddress = browserContainer.getAddress();
        String vncPort = String.valueOf(this.config.getDockerVncPort());
        envs.add("VNC_SERVER=" + vncAddress + ":" + vncPort);
        String network = this.config.getDockerNetwork();
        List<String> extraHosts = this.config.getDockerExtraHosts();
        DockerContainer noVncContainer = DockerContainer.dockerBuilder(dockerImage).exposedPorts(exposedPorts).network(network).extraHosts(extraHosts).envs(envs).build();
        String containerId = this.startContainer(noVncContainer);
        noVncContainer.setContainerId(containerId);
        String noVncHost = this.getDefaultHost();
        String noVncPort = this.isHost(network) ? dockerNoVncPort : this.getBindPort(containerId, dockerNoVncPort + "/tcp");
        String noVncUrlFormat = "http://%s:%s/";
        String noVncUrl = String.format(noVncUrlFormat, noVncHost, noVncPort);
        noVncContainer.setContainerUrl(noVncUrl);
        return noVncContainer;
    }

    public DockerContainer startBrowserContainer(String dockerImage, String cacheKey, String browserVersion) {
        dockerImage = this.getPrefixedDockerImage(dockerImage);
        this.pullImageIfNecessary(cacheKey, dockerImage, browserVersion);
        ArrayList<String> exposedPorts = new ArrayList<String>();
        String dockerBrowserPort = String.valueOf(this.config.getDockerBrowserPort());
        exposedPorts.add(dockerBrowserPort);
        long shmSize = this.config.getDockerMemSizeBytes(this.config.getDockerShmSize());
        ArrayList<Mount> mounts = new ArrayList<Mount>();
        Mount tmpfsMount = new Mount().withTmpfsOptions(new TmpfsOptions().withSizeBytes(Long.valueOf(this.config.getDockerMemSizeBytes(this.config.getDockerTmpfsSize())))).withTarget(this.config.getDockerTmpfsMount());
        mounts.add(tmpfsMount);
        ArrayList<String> binds = new ArrayList<String>();
        String dockerVolumes = this.config.getDockerVolumes();
        if (!Config.isNullOrEmpty(dockerVolumes)) {
            List<String> volumeList = Arrays.asList(dockerVolumes.split(","));
            this.log.trace("Using custom volumes: {}", volumeList);
            binds.addAll(volumeList);
        }
        ArrayList<String> envs = new ArrayList<String>();
        envs.add("TZ=" + this.config.getDockerTimezone());
        envs.add("LANGUAGE=" + this.config.getDockerLang());
        envs.add("SCREEN_RESOLUTION=" + this.config.getDockerScreenResolution());
        envs.addAll(this.config.getDockerEnvVariables());
        String dockerVncPort = String.valueOf(this.config.getDockerVncPort());
        if (this.config.isDockerEnabledVnc()) {
            envs.add("ENABLE_VNC=true");
            exposedPorts.add(dockerVncPort);
        }
        if (this.isChromeAllowedOrigins(dockerImage, browserVersion)) {
            envs.add("DRIVER_ARGS=--whitelisted-ips= --allowed-origins=*");
        }
        String network = this.config.getDockerNetwork();
        List<String> extraHosts = this.config.getDockerExtraHosts();
        String containerName = "selenium_" + UUID.randomUUID().toString().substring(0, 6);
        DockerContainer.DockerBuilder dockerBuilder = DockerContainer.dockerBuilder(dockerImage).exposedPorts(exposedPorts).network(network).mounts(mounts).binds(binds).shmSize(shmSize).envs(envs).extraHosts(extraHosts).sysadmin().containerName(containerName);
        DockerContainer browserContainer = dockerBuilder.build();
        String containerId = this.startContainer(browserContainer);
        browserContainer.setContainerId(containerId);
        browserContainer.setContainerName(Optional.of(containerName));
        String gateway = this.getGateway(containerId, network);
        browserContainer.setGateway(gateway);
        String browserHost = this.getHost(containerId, network);
        String browserPort = this.isHost(network) ? dockerBrowserPort : this.getBindPort(containerId, dockerBrowserPort + "/tcp");
        String browserUrlFormat = "http://%s:%s/";
        if (dockerImage.contains("firefox")) {
            browserUrlFormat = browserUrlFormat + "wd/hub";
        }
        String browserUrl = String.format(browserUrlFormat, browserHost, browserPort);
        browserContainer.setContainerUrl(browserUrl);
        String address = this.getAddress(containerId, network);
        browserContainer.setAddress(address);
        this.log.trace("Browser remote URL {}", (Object)browserUrl);
        if (this.config.isDockerEnabledVnc()) {
            String vncPort = this.isHost(network) ? dockerVncPort : this.getBindPort(containerId, dockerVncPort + "/tcp");
            browserContainer.setVncPort(vncPort);
            String vncAddress = String.format("vnc://%s:%s/", this.getDefaultHost(), vncPort);
            this.log.debug("VNC server URL: {}", (Object)vncAddress);
            browserContainer.setVncAddress(vncAddress);
        }
        return browserContainer;
    }

    private boolean isHost(String dockerNetwork) {
        return dockerNetwork.equalsIgnoreCase(NETWORK_HOST);
    }

    private boolean isChromeAllowedOrigins(String dockerImage, String browserVersion) {
        if (dockerImage.contains("chrome")) {
            String parsedVersion = VersionDetector.parseVersion(browserVersion);
            return new VersionComparator().compare(parsedVersion, "95") >= 0;
        }
        return false;
    }

    public DockerContainer startRecorderContainer(String dockerImage, String cacheKey, String recorderVersion, DockerContainer browserContainer) {
        dockerImage = this.getPrefixedDockerImage(dockerImage);
        this.pullImageIfNecessary(cacheKey, dockerImage, recorderVersion);
        String network = this.config.getDockerNetwork();
        ArrayList<String> envs = new ArrayList<String>();
        Optional<String> containerName = browserContainer.getContainerName();
        if (containerName.isPresent()) {
            envs.add("BROWSER_CONTAINER_NAME=" + containerName.get());
        }
        Path recordingPath = this.getRecordingPath(browserContainer);
        envs.add("FILE_NAME=" + recordingPath.getFileName().toString());
        envs.add("VIDEO_SIZE=" + this.config.getDockerVideoSize());
        envs.add("FRAME_RATE=" + this.config.getDockerRecordingFrameRate());
        List<String> extraHosts = this.config.getDockerExtraHosts();
        ArrayList<String> binds = new ArrayList<String>();
        binds.add(recordingPath.toAbsolutePath().getParent().toString() + ":/data");
        String dockerVolumes = this.config.getDockerVolumes();
        if (!Config.isNullOrEmpty(dockerVolumes)) {
            List<String> volumeList = Arrays.asList(dockerVolumes.split(","));
            this.log.trace("Using custom volumes: {}", volumeList);
            binds.addAll(volumeList);
        }
        DockerContainer recorderContainer = DockerContainer.dockerBuilder(dockerImage).network(network).envs(envs).binds(binds).extraHosts(extraHosts).sysadmin().build();
        String containerId = this.startContainer(recorderContainer);
        recorderContainer.setContainerId(containerId);
        recorderContainer.setRecordingPath(recordingPath);
        return recorderContainer;
    }

    public Path getRecordingPath(DockerContainer browserContainer) {
        Path recordingPath;
        Path dockerRecordingPath = this.config.getDockerRecordingOutput();
        if (dockerRecordingPath.toString().toLowerCase(Locale.ROOT).endsWith(RECORDING_EXT)) {
            recordingPath = dockerRecordingPath;
        } else {
            String sessionId = browserContainer.getSessionId();
            Date now = new Date();
            SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_FORMAT);
            String recordingFileName = browserContainer.getBrowserName() + SEPARATOR + dateFormat.format(now) + SEPARATOR + sessionId + RECORDING_EXT;
            String prefix = this.config.getDockerRecordingPrefix();
            if (!Config.isNullOrEmpty(prefix)) {
                recordingFileName = prefix + recordingFileName;
            }
            recordingPath = Paths.get(dockerRecordingPath.toString(), recordingFileName);
        }
        return recordingPath;
    }

    public String getVersionFromImage(String dockerImage) {
        return dockerImage.substring(dockerImage.indexOf(":") + 1);
    }

    public DockerClient getDockerClient() {
        return this.dockerClient;
    }
}

