/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.vertx.http.runtime.cors;

import io.quarkus.vertx.http.runtime.cors.CORSConfig;
import io.vertx.core.Handler;
import io.vertx.core.http.HttpHeaders;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.http.HttpServerRequest;
import io.vertx.core.http.HttpServerResponse;
import io.vertx.ext.web.RoutingContext;
import java.net.URI;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;
import org.jboss.logging.Logger;

public class CORSFilter
implements Handler<RoutingContext> {
    private static final Logger LOG = Logger.getLogger(CORSFilter.class);
    final CORSConfig corsConfig;
    private final boolean wildcardOrigin;
    private final boolean wildcardMethod;
    private final List<Pattern> allowedOriginsRegex;
    private final Set<HttpMethod> configuredHttpMethods;
    private final String exposedHeaders;
    private final String allowedHeaders;
    private final String allowedMethods;

    public CORSFilter(CORSConfig corsConfig) {
        this.corsConfig = corsConfig;
        this.wildcardOrigin = CORSFilter.isOriginConfiguredWithWildcard(this.corsConfig.origins);
        this.wildcardMethod = CORSFilter.isConfiguredWithWildcard(corsConfig.methods);
        this.allowedOriginsRegex = this.wildcardOrigin ? List.of() : CORSFilter.parseAllowedOriginsRegex(this.corsConfig.origins);
        this.configuredHttpMethods = this.createConfiguredHttpMethods(this.corsConfig.methods);
        this.exposedHeaders = this.createHeaderString(this.corsConfig.exposedHeaders);
        this.allowedHeaders = this.createHeaderString(this.corsConfig.headers);
        this.allowedMethods = this.createHeaderString(this.corsConfig.methods);
    }

    private String createHeaderString(Optional<List<String>> headers) {
        if (headers.isEmpty()) {
            return null;
        }
        if (headers.get().isEmpty()) {
            return null;
        }
        if (headers.get().size() == 1 && headers.get().get(0).equals("*")) {
            return null;
        }
        return String.join((CharSequence)",", (Iterable<? extends CharSequence>)headers.get());
    }

    private Set<HttpMethod> createConfiguredHttpMethods(Optional<List<String>> methods) {
        if (methods.isEmpty()) {
            return Set.of();
        }
        List<String> corsConfigMethods = methods.get();
        LinkedHashSet<HttpMethod> result = new LinkedHashSet<HttpMethod>(corsConfigMethods.size());
        for (String value : corsConfigMethods) {
            result.add(HttpMethod.valueOf((String)value));
        }
        return result;
    }

    public static boolean isConfiguredWithWildcard(Optional<List<String>> optionalList) {
        if (optionalList == null || !optionalList.isPresent()) {
            return true;
        }
        List<String> list = optionalList.get();
        return list.isEmpty() || list.size() == 1 && "*".equals(list.get(0));
    }

    private static boolean isOriginConfiguredWithWildcard(Optional<List<String>> origins) {
        if (origins.isEmpty() || origins.get().size() != 1) {
            return false;
        }
        String origin = origins.get().get(0);
        return "*".equals(origin) || "/.*/".equals(origin);
    }

    public static List<Pattern> parseAllowedOriginsRegex(Optional<List<String>> allowedOrigins) {
        if (allowedOrigins == null || !allowedOrigins.isPresent()) {
            return List.of();
        }
        ArrayList<Pattern> allowOriginsRegex = new ArrayList<Pattern>();
        for (String o : allowedOrigins.get()) {
            if (o == null || !o.startsWith("/") || !o.endsWith("/")) continue;
            allowOriginsRegex.add(Pattern.compile(o.substring(1, o.length() - 1)));
        }
        return allowOriginsRegex;
    }

    public static boolean isOriginAllowedByRegex(List<Pattern> allowOriginsRegex, String origin) {
        if (allowOriginsRegex == null) {
            return false;
        }
        for (Pattern pattern : allowOriginsRegex) {
            if (!pattern.matcher(origin).matches()) continue;
            return true;
        }
        return false;
    }

    public void handle(RoutingContext event) {
        Objects.requireNonNull(this.corsConfig, "CORS config is not set");
        HttpServerRequest request = event.request();
        HttpServerResponse response = event.response();
        String origin = request.getHeader(HttpHeaders.ORIGIN);
        if (origin == null) {
            event.next();
        } else {
            boolean originMatches;
            boolean allowsOrigin = this.wildcardOrigin;
            boolean bl = originMatches = !this.wildcardOrigin && this.corsConfig.origins.isPresent() && (this.corsConfig.origins.get().contains(origin) || CORSFilter.isOriginAllowedByRegex(this.allowedOriginsRegex, origin));
            if (!allowsOrigin) {
                allowsOrigin = this.corsConfig.origins.isPresent() ? originMatches || CORSFilter.isSameOrigin(request, origin) : CORSFilter.isSameOrigin(request, origin);
            }
            if (!allowsOrigin) {
                LOG.debugf("Invalid origin %s", (Object)origin);
                response.setStatusCode(403);
                response.setStatusMessage("CORS Rejected - Invalid origin");
            } else {
                boolean allowCredentials = this.corsConfig.accessControlAllowCredentials.orElse(originMatches);
                response.headers().set(HttpHeaders.ACCESS_CONTROL_ALLOW_CREDENTIALS, (CharSequence)String.valueOf(allowCredentials));
                response.headers().set(HttpHeaders.ACCESS_CONTROL_ALLOW_ORIGIN, (CharSequence)origin);
            }
            if (request.method().equals((Object)HttpMethod.OPTIONS)) {
                String requestedMethods = request.getHeader(HttpHeaders.ACCESS_CONTROL_REQUEST_METHOD);
                String requestedHeaders = request.getHeader(HttpHeaders.ACCESS_CONTROL_REQUEST_HEADERS);
                if (requestedHeaders != null || requestedMethods != null) {
                    this.handlePreflightRequest(event, requestedHeaders, requestedMethods, origin, allowsOrigin);
                    response.end();
                    return;
                }
            }
            if (this.allowedHeaders != null) {
                response.headers().add(HttpHeaders.ACCESS_CONTROL_ALLOW_HEADERS, (CharSequence)this.allowedHeaders);
            }
            if (this.allowedMethods != null) {
                response.headers().add(HttpHeaders.ACCESS_CONTROL_ALLOW_METHODS, (CharSequence)this.allowedMethods);
            }
            if (this.exposedHeaders != null) {
                response.headers().add(HttpHeaders.ACCESS_CONTROL_EXPOSE_HEADERS, (CharSequence)this.exposedHeaders);
            }
            if (!this.isMethodAllowed(request.method())) {
                LOG.debugf("Method %s is not allowed", (Object)request.method());
                response.setStatusCode(403);
                response.setStatusMessage("CORS Rejected - Invalid method");
                response.end();
                return;
            }
            if (!allowsOrigin) {
                response.end();
                return;
            }
            event.next();
        }
    }

    private void handlePreflightRequest(RoutingContext event, String requestedHeaders, String requestedMethods, String origin, boolean allowsOrigin) {
        if (this.corsConfig.accessControlMaxAge.isPresent()) {
            event.response().putHeader(HttpHeaders.ACCESS_CONTROL_MAX_AGE, (CharSequence)String.valueOf(this.corsConfig.accessControlMaxAge.get().getSeconds()));
        }
        HttpServerResponse response = event.response();
        if (requestedMethods != null) {
            this.processPreFlightMethods(response, requestedMethods);
        }
        if (requestedHeaders != null) {
            this.processPreFlightRequestedHeaders(response, requestedHeaders);
        }
        if (this.exposedHeaders != null) {
            response.headers().add(HttpHeaders.ACCESS_CONTROL_EXPOSE_HEADERS, (CharSequence)this.exposedHeaders);
        }
    }

    static boolean isSameOrigin(HttpServerRequest request, String origin) {
        if (origin.startsWith(request.scheme())) {
            if (!CORSFilter.substringMatch(origin, request.scheme().length(), "://", false)) {
                LOG.debugf("Same origin check has failed, the origin is not a substring of the request URI. Request URI: %s, origin: %s", (Object)request.absoluteURI(), (Object)origin);
                return false;
            }
            if (CORSFilter.substringMatch(origin, request.scheme().length() + 3, request.host(), true)) {
                return true;
            }
            return CORSFilter.isSameOriginSlowPath(request, origin);
        }
        return false;
    }

    static boolean isSameOriginSlowPath(HttpServerRequest request, String origin) {
        String absUriString = request.absoluteURI();
        URI baseUri = URI.create(absUriString);
        URI originUri = URI.create(origin);
        if (!originUri.getPath().isEmpty()) {
            LOG.debugf("Same origin check has failed as the origin contains a path component. Request URI: %s, origin: %s", (Object)request.absoluteURI(), (Object)origin);
            return false;
        }
        if (!baseUri.getHost().equals(originUri.getHost())) {
            LOG.debugf("Same origin check has failed, the host values do not match. Request URI: %s, origin: %s", (Object)request.absoluteURI(), (Object)origin);
            return false;
        }
        if (baseUri.getPort() == originUri.getPort()) {
            return true;
        }
        if (baseUri.getPort() != -1 && originUri.getPort() != -1) {
            return false;
        }
        if (baseUri.getScheme().equals("http") ? !(baseUri.getPort() != 80 && baseUri.getPort() != -1 || originUri.getPort() != 80 && originUri.getPort() != -1) : !(!baseUri.getScheme().equals("https") || baseUri.getPort() != 443 && baseUri.getPort() != -1 || originUri.getPort() != 443 && originUri.getPort() != -1)) {
            return true;
        }
        LOG.debugf("Same origin check has failed. Request URI: %s, origin: %s", (Object)request.absoluteURI(), (Object)origin);
        return false;
    }

    static boolean substringMatch(String str, int pos, String substring, boolean requireFull) {
        int length = str.length();
        int subLength = substring.length();
        int strPos = pos;
        int subPos = 0;
        if (pos + subLength > length) {
            return false;
        }
        while (subPos != subLength) {
            if (str.charAt(strPos) != substring.charAt(subPos)) {
                return false;
            }
            ++strPos;
            ++subPos;
        }
        return !requireFull || strPos == length;
    }

    private void processPreFlightRequestedHeaders(HttpServerResponse response, String allowHeadersValue) {
        if (CORSFilter.isConfiguredWithWildcard(this.corsConfig.headers)) {
            response.headers().set(HttpHeaders.ACCESS_CONTROL_ALLOW_HEADERS, (CharSequence)allowHeadersValue);
        } else {
            response.headers().set(HttpHeaders.ACCESS_CONTROL_ALLOW_HEADERS, (CharSequence)this.allowedHeaders);
        }
    }

    private void processPreFlightMethods(HttpServerResponse response, String allowMethodsValue) {
        if (this.wildcardMethod) {
            response.headers().set(HttpHeaders.ACCESS_CONTROL_ALLOW_METHODS, (CharSequence)allowMethodsValue);
        } else {
            StringBuilder allowMethods = new StringBuilder();
            boolean isFirst = true;
            for (HttpMethod configMethod : this.configuredHttpMethods) {
                if (isFirst) {
                    isFirst = false;
                } else {
                    allowMethods.append(",");
                }
                allowMethods.append(configMethod.name());
            }
            response.headers().set(HttpHeaders.ACCESS_CONTROL_ALLOW_METHODS, (CharSequence)allowMethods.toString());
        }
    }

    private boolean isMethodAllowed(HttpMethod method) {
        if (this.wildcardMethod) {
            return true;
        }
        return this.configuredHttpMethods.contains(method);
    }
}

