///////////////////////////////////////////////////////////////////////////////
//
// JTOpen (IBM Toolbox for Java - OSS version)
//
// Filename:  AspListItem.java
//
// The source code contained herein is licensed under the IBM Public License
// Version 1.0, which has been approved by the Open Source Initiative.
// Copyright (C) 2018-2019 International Business Machines Corporation and
// others.  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////


package com.ibm.as400.access.list;

import java.util.Calendar;
import java.util.Date;

import com.ibm.as400.access.AS400Calendar;

/**
Contains ASPs information that was generated by {@link com.ibm.as400.access.list.AspOpenList AspOpenList}.
<p>Some attributes will not be available, depending on the {@link #getFormat format} that was used to generate this item.  
The javadoc for each attribute getter indicates which formats will generate valid data for the given attribute. 
 If an attribute getter that is only valid for a format other than what was used to generate this AspListItem, 
 the data returned by that getter is not valid.
**/
public class AspListItem {
	
	
	//YASP0100
	private int aspNumber_;
	private String aspResourceName_;
	private String aspDevDescriptionName_;
	private int aspVersion_;
	private int aspUsage_;
	private int aspStatus_;
	private String aspDatabaseName_;
	private String primaryASPResourceName_;
	
	//YASP0200
	private int diskUnitNumber_;
	private int aspCapacity_;
	private int aspCapacityAvailable_;
	private int aspCapacityProtected_;
	private int aspCapacityAvaProtected_;
	private int aspCapacityUnprotected_;
	private int aspCapacityAvaUnprotected_;
	private int aspSystemStorage_;
	private int overflowStorage_;
	private int space2ErrorLog_;
	private int space2MachineLog_;
	private int space2MachineTreac_;
	private int space4MainStoragedump_;
	private int space2Microcode_;
	private int storageThresholdPer_;
	private String aspType_;
	private String overflowRecovery_;
	private String endImmeControl_;
	private String compressionRecoveryPolicy_;
	private String compressedDiskUnitInASP_;
	private String balanceStatus_;
	private String balanceType_;
	private String balanceDateTime_;
	private int balanceDataMoved_;
	private int balanceDataRemain_;
	private int traceDuration_;
	private String traceStatus_;
	private String traceDateTime_;
	private String changesWritten2Disk_;
	private int multiConnDiskUnit_;
	private int geographicMirrorRole_;
	private int geographicMirrorCpStat_;
	private int geographicMirrorCpDataStat_;
	private int geographicMirrorPerfMode_;
	private int geographicMirrorResumePriority_;
	private int geographicMirrorSuspendTimeout_;
	private int diskUnitpresence_;
	
	//YASP0300
	private String diskType_;
	private String diskModel_;
	private String diskSerialNumber_;
	private String diskResourceName_;
	private int diskCapacity_;
	private int diskStorageAvailable_;
	private int diskStorageReserved_;
	private String mirroredUnitProtected_;
	private String mirroredUnitReported_;
	private String mirroredUnitStatus_;
	private String RAIDType_;
	private int unitControl_;
	private int blockTransfer2MainStorage_;
	private int blockTransferFromMainStorage_;
	private int reqDataTransfer2MainStorage_;
	private int reqDataTransferFromMainStorage_;
	private int permanentBlockTransferFromMainStorage_;
	private int reqPermanentBlockTransferFromMainStorage_;
	private int sampleCount_;
	private int notBusyCount_;
	private String compressedStatus_;
	private String diskProtectionType_;
	private String compressedUnit_;
	private String stroageAllocationRestrictUnit_;
	private String avaParitySetUnit_;
	private String multiConnectionUnit_;
	
	//YASP0400
	private String errorCode_;
	private String mirrorUnitIdentifier_;
	
	//YASP0500
	private String tranfitionTarget_;
	private String function_;
	private int currentCount_;
	private int totalCount_;
	private int currentItemCount_;
	private int totalItemCount_;
	private String elapsedTime_;
	
	//YASP0600
	private int useIdentification_;
	private String jobName_;
	private String jobUserName_;
	private String jobNumber_;
	private String threadIdentifier_;
	private String threadStatus_;
	
	private String format_;
	
	//YASP0100
	AspListItem(int aspNumber, String aspResourceName, String aspDevDescriptionName, int aspVersion, int aspUsage, int aspStatus, String aspDatabaseName, String primaryASPResourceName) {
		aspNumber_ = aspNumber;
		aspResourceName_ = aspResourceName;
		aspDevDescriptionName_ = aspDevDescriptionName;
		aspVersion_ = aspVersion;
		aspUsage_ = aspUsage;
		aspStatus_ = aspStatus;
		aspDatabaseName_ = aspDatabaseName;
		primaryASPResourceName_ = primaryASPResourceName;
		format_ = AspOpenList.FORMAT_0100;
	}
	
	//YASP0200
	AspListItem(int aspNumber, int diskUnitNumber,int aspCapacity,int aspCapacityAvailable,int aspCapacityProtected,int aspCapacityAvaProtected,int aspCapacityUnprotected,int aspCapacityAvaUnprotected,int aspSystemStorage,int overflowStorage,int space2ErrorLog,int space2MachineLog,int space2MachineTreac,int space4MainStoragedump,int space2Microcode,int storageThresholdPer ,String aspType,String overflowRecovery,String endImmeControl,String compressionRecoveryPolicy,String compressedDiskUnitInASP,String balanceStatus,String balanceType,String balanceDateTime,int balanceDataMoved,int balanceDataRemain,int traceDuration,String traceStatus,String traceDateTime,String changesWritten2Disk,int multiConnDiskUnit,int geographicMirrorRole,int geographicMirrorCpStat,int geographicMirrorCpDataStat,int geographicMirrorPerfMode,int geographicMirrorResumePriority,int geographicMirrorSuspendTimeout,int diskUnitpresence) {
		aspNumber_ = aspNumber;
		diskUnitNumber_ = diskUnitNumber;
		aspCapacity_ = aspCapacity;
		aspCapacityAvailable_ = aspCapacityAvailable;
		aspCapacityProtected_ = aspCapacityProtected;
		aspCapacityAvaProtected_ = aspCapacityAvaProtected;
		aspCapacityUnprotected_ = aspCapacityUnprotected;
		aspCapacityAvaUnprotected_ = aspCapacityAvaUnprotected;
		aspSystemStorage_ = aspSystemStorage;
		overflowStorage_ = overflowStorage;
		space2ErrorLog_ = space2ErrorLog;
		space2MachineLog_ = space2MachineLog;
		space2MachineTreac_ = space2MachineTreac;
		space4MainStoragedump_ = space4MainStoragedump;
		space2Microcode_ = space2Microcode;
		storageThresholdPer_ = storageThresholdPer;
		aspType_ = aspType;
		overflowRecovery_ = overflowRecovery;
		endImmeControl_ = endImmeControl;
		compressionRecoveryPolicy_ = compressionRecoveryPolicy;
		compressedDiskUnitInASP_ = compressedDiskUnitInASP;
		balanceStatus_ = balanceStatus;
		balanceType_ = balanceType;
		balanceDateTime_ = balanceDateTime;
		//Reserved BINARY(3)
		balanceDataMoved_ = balanceDataMoved;
		balanceDataRemain_ = balanceDataRemain;
		traceDuration_ = traceDuration;
		traceStatus_ = traceStatus;
		traceDateTime_ = traceDateTime;
		changesWritten2Disk_ = changesWritten2Disk;
		//Reserved BINARY(1)
		multiConnDiskUnit_ = multiConnDiskUnit;
		geographicMirrorRole_ = geographicMirrorRole;
		geographicMirrorCpStat_ = geographicMirrorCpStat;
		geographicMirrorCpDataStat_ = geographicMirrorCpDataStat;
		geographicMirrorPerfMode_ = geographicMirrorPerfMode;
		geographicMirrorResumePriority_ = geographicMirrorResumePriority;
		geographicMirrorSuspendTimeout_ = geographicMirrorSuspendTimeout;
		diskUnitpresence_ = diskUnitpresence;
		format_ = AspOpenList.FORMAT_0200;
	}
	
	//YASP0300
	AspListItem(int aspNumber, String diskType, String diskModel, String diskSerialNumber, String diskResourceName, int diskUnitNumber, int diskCapacity, int diskStorageAvailable, int diskStorageReserved, String mirroredUnitProtected, String mirroredUnitReported, String mirroredUnitStatus, String RAIDType, int unitControl, int blockTransfer2MainStorage, int blockTransferFromMainStorage,int reqDataTransfer2MainStorage, int reqDataTransferFromMainStorage, int permanentBlockTransferFromMainStorage, int reqPermanentBlockTransferFromMainStorage, int sampleCount, int notBusyCount, String compressedStatus, String diskProtectionType, String compressedUnit, String stroageAllocationRestrictUnit, String avaParitySetUnit, String multiConnectionUnit) 
	{
		aspNumber_ = aspNumber;
		diskType_ = diskType;
		diskModel_ = diskModel;
		diskSerialNumber_ = diskSerialNumber;
		diskResourceName_ = diskResourceName;
		diskUnitNumber_ = diskUnitNumber;
		diskCapacity_ = diskCapacity;
		diskStorageAvailable_ = diskStorageAvailable;
		diskStorageReserved_ = diskStorageReserved;
		mirroredUnitProtected_ = mirroredUnitProtected;
		mirroredUnitReported_ = mirroredUnitReported;
		mirroredUnitStatus_ = mirroredUnitStatus;
		RAIDType_ = RAIDType;
		unitControl_ = unitControl;
		blockTransfer2MainStorage_ = blockTransfer2MainStorage;
		blockTransferFromMainStorage_ = blockTransferFromMainStorage;
		reqDataTransfer2MainStorage_ = reqDataTransfer2MainStorage;
		reqDataTransferFromMainStorage_ = reqDataTransferFromMainStorage;
		permanentBlockTransferFromMainStorage_ = permanentBlockTransferFromMainStorage;
		reqPermanentBlockTransferFromMainStorage_ = reqPermanentBlockTransferFromMainStorage;
		sampleCount_ = sampleCount;
		notBusyCount_ = notBusyCount;
		compressedStatus_ = compressedStatus;
		diskProtectionType_ = diskProtectionType;
		compressedUnit_ = compressedUnit;
		stroageAllocationRestrictUnit_ = stroageAllocationRestrictUnit;
		avaParitySetUnit_ = avaParitySetUnit;
		multiConnectionUnit_ = multiConnectionUnit;
		format_ = AspOpenList.FORMAT_0300;
	}
	
	//YASP0400
	AspListItem(int aspNumber, String errorCode, int diskUnitNumber, String mirrorUnitIdentifier) {
		aspNumber_ = aspNumber;
		errorCode_ = errorCode;
		diskUnitNumber_ = diskUnitNumber;
		mirrorUnitIdentifier_ = mirrorUnitIdentifier;
		format_ = AspOpenList.FORMAT_0400;
	} 
	
	//YASP0500
	AspListItem(int aspNumber, String tranfitionTarget, String function, int currentCount, int totalCount, int currentItemCount, int totalItemCount, String elapsedTime) {
		aspNumber_ = aspNumber;
		tranfitionTarget_ = tranfitionTarget;
		function_ = function;
		currentCount_ = currentCount;
		totalCount_=totalCount;
		currentItemCount_=currentItemCount;
		totalItemCount_=totalItemCount;
		elapsedTime_=elapsedTime;
		format_ = AspOpenList.FORMAT_0500;
	} 
	
    AspListItem(int aspNumber, int useIdentification, String jobName, String jobUserName, String jobNumber, String threadIdentifier, String threadStatus) {
    	aspNumber_ = aspNumber;
    	useIdentification_=useIdentification;
    	jobName_=jobName;
    	jobUserName_=jobUserName;
    	jobNumber_=jobNumber;
    	threadIdentifier_=threadIdentifier;
    	threadStatus_=threadStatus;
		format_ = AspOpenList.FORMAT_0600;
	}

	/**
    Returns the auxiliary storage pool (ASP) number.  Possible values are:
    <ul>
    <li>1 - The system auxiliary storage pool (*SYSTEM).
    <li>2-255 - The number of the auxiliary storage pool.
    </ul>
    <p>Available in all format.
    @return  The auxiliary storage pool number.
    **/
    public int getASPNumber() {
    	return aspNumber_;
    }
    
    /**
    Returns the auxiliary storage pool (ASP) resource name.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0100 FORMAT_0100}, {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return  The auxiliary storage pool resource name.
    **/
    public String getResourceName() {
    	return aspResourceName_;
    }
    
    /**
    Returns the auxiliary storage pool (ASP) device description name.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0100 FORMAT_0100}
    @return  The auxiliary storage pool device description name.
    **/
    public String getDevDescriptionName() {
    	return aspDevDescriptionName_;
    }
    
    /**
    Returns the version of objects in an independent ASP.  Possible values are:
    <ul>
    <li>0 - Objects are usable by a system at release level V5R1M0.
    <li>1 - Objects are usable by a system at release level V5R2M0.
    <li>2 - Objects are usable by a system at release level V5R3M0.
    <li>3 - Objects are usable by a system at release level V5R4M0.
    <li>4 - Objects are usable by a system at release level V6R1M0.
    <li>5 - Objects are usable by a system at release level V7R1M0.
    <li>6 - Objects are usable by a system at release level V7R2M0.
    <li>7 - Objects are usable by a system at release level V7R3M0.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0100 FORMAT_0100}
    @return  The version of objects in an independent ASP.
    **/
    public int getVersion() {
    	return aspVersion_;
    }
    
    /**
    Returns the use that is assigned to the auxiliary storage pool.  Possible values are:
    <ul>
    <li>0 - The usage attribute does not apply or is not known. This value is used for the system ASP, any basic user ASPs and any independent ASPs whose use cannot be determined (eg, one whose disk drives are not powered on).
    <li>1 - The ASP is a user-defined file system ASP.
    <li>2 - The ASP is a primary ASP.
    <li>3 - The ASP is a secondary ASP.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0100 FORMAT_0100}
    @return  The use that is assigned to the auxiliary storage pool.
    **/
    public int getASPUsage() {
    	return aspUsage_;
    }
    
    /**
    Returns the device configuration status of an auxiliary storage pool.  Possible values are:
    <ul>
    <li>0 - There is no status. This value is used for the system ASP and any basic user ASPs.
    <li>1 - The status of the ASP is varyoff.
    <li>2 - The status of the ASP is varyon.
    <li>3 - The status of the ASP is active.
    <li>4 - The status of the ASP is available.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0100 FORMAT_0100}
    @return  The device configuration status of an auxiliary storage pool.
    **/
    public int getASPStatus() {
    	return aspStatus_;
    }
    
    /**
    Returns the name that is assigned to the database that this ASP defines. If the ASP is neither a primary or secondary ASP, this field consists of blanks.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0100 FORMAT_0100}
    @return  The name that is assigned to the database that this ASP defines. If the ASP is neither a primary or secondary ASP, this field consists of blanks.
    **/
    public String getDatabaseName() {
    	return aspDatabaseName_;
    }
    
    /**
    Returns  the resource name of the primary ASP for a secondary ASP. Only a secondary ASP contains this value.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0100 FORMAT_0100}
    @return   The resource name of the primary ASP for a secondary ASP. Only a secondary ASP contains this value.
    **/
    public String getPrimaryASPResourceName() {
    	return primaryASPResourceName_;
    }
    
    //Format AspOpenList.FORMAT_0200 YASP0200
    /**
    Returns  the total number of disk units in the ASP. If mirroring is active for disk units within the ASP, the mirrored pair of units is counted as one.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200},{@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300},{@link com.ibm.as400.access.list.AspOpenList#FORMAT_0400 FORMAT_0400}
    @return   The total number of disk units in the ASP. If mirroring is active for disk units within the ASP, the mirrored pair of units is counted as one.
    **/
    public int getDiskUnitNumber() {
    	return diskUnitNumber_;
    }
    
    /**
    Returns  the total number of used and unused megabytes in the ASP. A special value of -2 is returned if the size of this field is exceeded.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The total number of used and unused megabytes in the ASP. A special value of -2 is returned if the size of this field is exceeded.
    **/
    public int getaspCapacity() {
    	return aspCapacity_;
    }
    
    /**
    Returns  The total number of unused megabytes in the ASP. A special value of -2 is returned if the size of this field is exceeded.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The total number of unused megabytes in the ASP. A special value of -2 is returned if the size of this field is exceeded.
    **/
    public int getaspCapacityAvailable() {
    	return aspCapacityAvailable_;
    }
    
    /**
    Returns  The total number of used and unused megabytes in the ASP that are protected by mirroring or device parity. A special value of -2 is returned if the size of this field is exceeded
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The total number of used and unused megabytes in the ASP that are protected by mirroring or device parity. A special value of -2 is returned if the size of this field is exceeded
    **/
    public int getaspCapacityProtected() {
    	return aspCapacityProtected_;
    }
    
    /**
    Returns  The number of unused megabytes in the ASP that are protected by mirroring or device parity. A special value of -2 is returned if the size of this field is exceeded.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The number of unused megabytes in the ASP that are protected by mirroring or device parity. A special value of -2 is returned if the size of this field is exceeded.
    **/
    public int getaspCapacityAvaProtected() {
    	return aspCapacityAvaProtected_;
    }
    
    /**
    Returns  The total number of used and unused megabytes in the ASP that are not protected by mirroring or device parity. A special value of -2 is returned if the size of this field is exceeded.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The total number of used and unused megabytes in the ASP that are not protected by mirroring or device parity. A special value of -2 is returned if the size of this field is exceeded.
    **/
    public int getaspCapacityUnprotected() {
    	return aspCapacityUnprotected_;
    }
    
    /**
    Returns  The number of unused megabytes in the ASP that are not protected by mirroring or device parity. A special value of -2 is returned if the size of this field is exceeded.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The number of unused megabytes in the ASP that are not protected by mirroring or device parity. A special value of -2 is returned if the size of this field is exceeded.
    **/
    public int getaspCapacityAvaUnprotected() {
    	return aspCapacityAvaUnprotected_;
    }
    
    /**
    Returns  The amount of storage in megabytes currently allocated in the ASP for system use.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The amount of storage in megabytes currently allocated in the ASP for system use.
    **/
    public int getaspSystemStorage() {
    	return aspSystemStorage_;
    }
    
    /**
    Returns  The number of megabytes of storage that has overflowed from the user ASP into the system ASP. A special value of -2 is returned if the size of this field is exceeded.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The number of megabytes of storage that has overflowed from the user ASP into the system ASP. A special value of -2 is returned if the size of this field is exceeded.
    **/
    public int getOverflowStorage() {
    	return overflowStorage_;
    }
    
    /**
    Returns  The number of megabytes of auxiliary storage allocated to the error log.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The number of megabytes of auxiliary storage allocated to the error log.
    **/
    public int getspace2ErrorLog() {
    	return space2ErrorLog_;
    }
    
    /**
    Returns  The number of megabytes of auxiliary storage allocated to the machine log.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The number of megabytes of auxiliary storage allocated to the machine log.
    **/
    public int getspace2MachineLog() {
    	return space2MachineLog_;
    }
    
    /**
    Returns  The number of megabytes of auxiliary storage allocated to the machine trace.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The number of megabytes of auxiliary storage allocated to the machine trace.
    **/
    public int getspace2MachineTreac() {
    	return space2MachineTreac_;
    }
    
    /**
    Returns  The number of megabytes of auxiliary storage allocated to the main storage dump space.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The number of megabytes of auxiliary storage allocated to the main storage dump space.
    **/
    public int getspace4MainStoragedump() {
    	return space4MainStoragedump_;
    }
    
    /**
    Returns  The number of megabytes of auxiliary storage allocated to the microcode and space used by the microcode.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The number of megabytes of auxiliary storage allocated to the microcode and space used by the microcode.
    **/
    public int getspace2Microcode() {
    	return space2Microcode_;
    }
    
    /**
    Returns  When the storage in the ASP reaches this percentage, a warning message is sent to the QSYSOPR message queue. When this percentage is reached for the system ASP (ASP 1), message CPF0907 is sent. When this percentage is reached for one of the user ASPs, message CPI0953 is sent.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   When the storage in the ASP reaches this percentage, a warning message is sent to the QSYSOPR message queue. When this percentage is reached for the system ASP (ASP 1), message CPF0907 is sent. When this percentage is reached for one of the user ASPs, message CPI0953 is sent.
    **/
    public int getstorageThresholdPer() {
    	return storageThresholdPer_;
    }
    
    /**
    Returns  The type of ASP. The possible ASP types are:
    <ul>
    <li>00 - The ASP is the system ASP.
    <li>10 - The ASP is a user ASP that does not contain libraries.
    <li>20 - The ASP is a user ASP that does contain libraries.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The type of ASP.
    **/
    public String getaspType() {
    	return aspType_;
    }
    
    /**
    Returns  An indicator of the result of the ASP overflow recovery operation, which is performed during IPL at the user's request. When this operation is requested, the machine attempts to recover the user ASP from an overflow condition by moving overflowed auxiliary storage from the system ASP back to the user ASP during the storage management recovery step of an IPL.
    <ul>
    <li>0 - All overflowed storage was successfully moved.
    <li>1 - ASP overflow recovery failed due to insufficient space in the user ASP.
    <li>2 - ASP overflow recovery was cancelled prior to completion.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   An indicator of the result of the ASP overflow recovery operation, which is performed during IPL at the user's request. When this operation is requested, the machine attempts to recover the user ASP from an overflow condition by moving overflowed auxiliary storage from the system ASP back to the user ASP during the storage management recovery step of an IPL.
    **/
    public String getoverflowRecovery() {
    	return overflowRecovery_;
    }
    
    /**
    Returns  This field has meaning in the system ASP (ASP 1) only. The possible ASP types are:
    <ul>
    <li>0 - If a request for space in the system ASP cannot be satisfied because there is not enough storage, the system will be allowed to continue running.
    <li>1 - If a request for space in the system ASP cannot be satisfied because there is not enough storage, the system will be ended immediately.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   This field has meaning in the system ASP (ASP 1) only.
    **/
    public String getendImmeControl() {
    	return endImmeControl_;
    }
    
    /**
    The compression recovery policy for the ASP. If the ASP has compressed drives as part of its configuration, this value controls how overflow situations are handled for this ASP. The following policies allow the user to control what is done when the ASP appears full. For more information about the compression recovery policy, see the Recovering your system topic collection. The following values are returned:
    <ul>
    <li>1 - When the system detects that the ASP capacity is about to be exceeded, the system posts system reference code (SRC) A6xx 0277 in the system control panel and waits for a limited time for space to become available. If space becomes available before the limited time ends, the SRC is removed from the system control panel and normal operations resume. If space does not become available before the limited time ends, data overflows into the system ASP.
    <li>2 - When the system detects that the ASP capacity is about to be exceeded, the data immediately overflows into the system ASP.
    <li>3 - When the system detects that the ASP capacity is about to be exceeded, the system posts SRC A6xx 0277 in the system control panel and waits indefinitely for space to become available. The user must take action before normal operation resumes. Possible actions include deleting objects from the ASP or changing the compression recovery policy to a value that allows the ASP to overflow.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The compression recovery policy for the ASP. If the ASP has compressed drives as part of its configuration, this value controls how overflow situations are handled for this ASP. The following policies allow the user to control what is done when the ASP appears full. For more information about the compression recovery policy, see the Recovering your system topic collection.
    **/
    public String getcompressionRecoveryPolicy() {
    	return compressionRecoveryPolicy_;
    }
    
    /**
    Returns  Whether there are compressed disk units in the ASP. The following special values are returned:
    <ul>
    <li>0 - If a request for space in the system ASP cannot be satisfied because there is not enough storage, the system will be allowed to continue running.
    <li>1 - If a request for space in the system ASP cannot be satisfied because there is not enough storage, the system will be ended immediately.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   Whether there are compressed disk units in the ASP.
    **/
    public String getcompressedDiskUnitInASP() {
    	return compressedDiskUnitInASP_;
    }
    
    /**
    Returns  The current status of the balance function for this ASP. This field is valid for an IASP if its status is either ACTIVE or AVAILABLE, otherwise hexadecimal zeros are returned. The following special values are returned:
    <ul>
    <li>0 - No balance activity has occurred for this ASP.
    <li>1 - The ASP balance function is currently running for this ASP.
    <li>2 - The ASP balance function is currently in the process of ending. Either the time limit has run out or the End ASP Balance (ENDASPBAL) command was issued for this ASP.
    <li>3 - The ASP balance function has run, but was ended before the ASP was completely balanced. The Start ASP Balance (STRASPBAL) command can be used to restart the balance function.
    <li>4 - The ASP balance function has completed running. The ASP is completely balanced.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The current status of the balance function for this ASP. This field is valid for an IASP if its status is either ACTIVE or AVAILABLE, otherwise hexadecimal zeros are returned.
    **/
    public String getbalanceStatus() {
    	return balanceStatus_;
    }
    
    /**
    Returns  The type of balance activity that is currently running or was done last. This field is valid for an IASP if its status is either ACTIVE or AVAILABLE, otherwise hexadecimal zeros are returned. The possible types of ASP balancing are:
    <ul>
    <li>0 - No balance activity has occurred for this ASP.
    <li>1 - Capacity balancing. Capacity balancing redistributes data so that the percentage of disk space used is the same on all disk units within the ASP.
    <li>2 - Usage balancing. Usage balancing redistributes data so that the percentage of disk activity is the same on all disk units within the ASP. High-use and low-use data is identified by running the Trace ASP Balance (TRCASPBAL) command. Usage balancing moves data among the disk units, guided by the trace results, in an attempt to equalize the utilizations.
    <li>3 - Heirarchical Storage Management (HSM) balancing. HSM balancing can be run only on an ASP that contains a mixture of high-performance and low-performance disk units. An example of low-performance disk units is compressed disk units. The HSM balance function moves high-use data to high-performance units and moves low-use data to low-performance units. The high-use and low-use data is identified by running the Trace ASP Balance (TRCASPBAL) command.
    <li>4 - Clear collection data. Clear collection data removes the trace data created by running the Trace ASP Balance (TRCASPBAL) command.
    <li>5 - Move data. Move data is used to reduce the down time associated with removing a disk unit. The Check ASP Balance (CHKASPBAL) command can be used to determine which units are currently marked to no longer receive new allocations and to have their existing allocations moved to other disk units.
    <li>6 - Media Preference (MP) balancing. MP balancing can be run only on an ASP that contains a mixture of Solid State Disk (SSD) units and Hard Disk Drive (HDD) units. The goal of the MP balance function is to have high-use data on SSD units and low-use data on HDD units. The high-use and low-use data is identified by running the Trace ASP Balance (TRCASPBAL) command.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The type of balance activity that is currently running or was done last. This field is valid for an IASP if its status is either ACTIVE or AVAILABLE, otherwise hexadecimal zeros are returned.
    **/
    public String getbalanceType() {
    	return balanceType_;
    }
    
    /**
    Returns  The date and time of the last status change for the balance function. This field is valid for an IASP if its status is either ACTIVE or AVAILABLE, otherwise hexadecimal zeros are returned. The 13 characters are: 
    <ul>
    <li>1     Century, where 0 indicates years 19xx and 1 indicates years 20xx.
    <li>2-7   The date, in YYMMDD (year, month, day) format.
    <li>8-13  The time of day, in HHMMSS (hours, minutes, seconds) format.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The date and time of the last status change for the balance function. This field is valid for an IASP if its status is either ACTIVE or AVAILABLE, otherwise hexadecimal zeros are returned.
    **/
    public Date getbalanceDateTime() {
    	//return balanceDateTime_;
    	if (balanceDateTime_ == null || balanceDateTime_.length() == 0) return null;
        Calendar c = AS400Calendar.getGregorianInstance();
        c.clear();
        c.set(Integer.parseInt(balanceDateTime_.substring(0, 3)) + 1900, // Year.
              Integer.parseInt(balanceDateTime_.substring(3, 5)) - 1, // Month is zero based.
              Integer.parseInt(balanceDateTime_.substring(5, 7)), // Day.
              Integer.parseInt(balanceDateTime_.substring(7, 9)), // Hour.
              Integer.parseInt(balanceDateTime_.substring(9, 11)), // Minute.
              Integer.parseInt(balanceDateTime_.substring(11, 13))); // Second.
        return c.getTime();  
    }
    
    /**
    Returns  The number of megabytes that have been moved by the balance function. A special value of -2 is returned if the size of this field is exceeded. If the ASP status is neither ACTIVE nor AVAILABLE, 0 is returned.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The number of megabytes that have been moved by the balance function. A special value of -2 is returned if the size of this field is exceeded. If the ASP status is neither ACTIVE nor AVAILABLE, 0 is returned.
    **/
    public int getbalanceDataMoved() {
    	return balanceDataMoved_;
    }
    
    /**
    Returns  The number of megabytes that remain to be moved by the balance function before the move is considered complete. A special value of -2 is returned if the size of this field is exceeded. If the ASP status is neither ACTIVE nor AVAILABLE, 0 is returned.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The number of megabytes that remain to be moved by the balance function before the move is considered complete. A special value of -2 is returned if the size of this field is exceeded. If the ASP status is neither ACTIVE nor AVAILABLE, 0 is returned.
    **/
    public int getbalanceDataRemain() {
    	return balanceDataRemain_;
    }
    
    /**
    Returns  The number of minutes that the trace function has run collecting data for this ASP. The trace can be run multiple times for an ASP. If the ASP status is neither ACTIVE nor AVAILABLE, zero is returned.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The number of minutes that the trace function has run collecting data for this ASP. The trace can be run multiple times for an ASP. If the ASP status is neither ACTIVE nor AVAILABLE, zero is returned.
    **/
    public int gettraceDuration() {
    	return traceDuration_;
    }
    
    /**
    Returns  The current status of the trace function. The trace gathers statistics about the data on the disk units within the ASP. This data is used by the balance functions. This field is valid for an IASP if its status is either ACTIVE or AVAILABLE, otherwise hexadecimal zeros are returned. The following special values are returned:
    <ul>
    <li>0 - There is no current trace data for this ASP.
    <li>1 - The trace function is currently running for this ASP.
    <li>2 - The trace function is currently in the process of ending. Either the time limit has run out or the trace was stopped through use of the Trace ASP Balance (TRCASPBAL) command.
    <li>3 - The trace function has completed running. The statistics for the ASP have been gathered and are ready for the balance function to start.
    <li>4 - The trace data for this ASP is being cleared.
    <li>5 - The trace function has completed and the statistics for the ASP have been gathered. The ASP is ready for further collection or for the balance function to start.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The current status of the trace function. The trace gathers statistics about the data on the disk units within the ASP. This data is used by the balance functions. This field is valid for an IASP if its status is either ACTIVE or AVAILABLE, otherwise hexadecimal zeros are returned.
    **/
    public String gettraceStatus() {
    	return traceStatus_;
    }
    
    /**
    Returns  The date and time of the last status change for the trace function. The 13 characters are: 
    <ul>
    <li>1     Century, where 0 indicates years 19xx and 1 indicates years 20xx.
    <li>2-7   The date, in YYMMDD (year, month, day) format.
    <li>8-13  The time of day, in HHMMSS (hours, minutes, seconds) format.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The date and time of the last status change for the trace function.
    **/
    public Date gettraceDateTime() {
    	//return traceDateTime_;
    	if (traceDateTime_ == null || traceDateTime_.length() == 0) return null;
        Calendar c = AS400Calendar.getGregorianInstance();
        c.clear();
        c.set(Integer.parseInt(traceDateTime_.substring(0, 3)) + 1900, // Year.
              Integer.parseInt(traceDateTime_.substring(3, 5)) - 1, // Month is zero based.
              Integer.parseInt(traceDateTime_.substring(5, 7)), // Day.
              Integer.parseInt(traceDateTime_.substring(7, 9)), // Hour.
              Integer.parseInt(traceDateTime_.substring(9, 11)), // Minute.
              Integer.parseInt(traceDateTime_.substring(11, 13))); // Second.
        return c.getTime(); 
    }
    
    /**
    Returns  An indicator of whether all changes made the previous time the IASP was online were written to disk. Varyoff processing attempts to write changed IASP storage but, in some failures, it may not be successful.
    <ul>
    <li>0 - Not all changes were written.
    <li>1 - All changes were written.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   An indicator of whether all changes made the previous time the IASP was online were written to disk. Varyoff processing attempts to write changed IASP storage but, in some failures, it may not be successful.
    **/
    public String getchangesWritten2Disk() {
    	return changesWritten2Disk_;
    }
    
    
    public int getmultiConnDiskUnit() {
    	return multiConnDiskUnit_;
    }
    
    /**
    Returns  The current role of a physical IASP.
    <ul>
    <li>0 - Geographic mirroring is not configured.
    <li>1 - System does not own a physical IASP copy.
    <li>2 - Geographic mirror role is unknown.
    <li>196 - System owns a detached mirror copy.
    <li>212 - System owns the mirror copy.
    <li>215 - System owns the production copy.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The current role of a physical IASP.
    **/
    public int getgeographicMirrorRole() {
    	return geographicMirrorRole_;
    }
    
    /**
    Returns  The mirror state of the mirror copy.
    <ul>
    <li>0 - Geographic mirroring is not configured.
    <li>1 - The system attempts to perform geographic mirroring when the IASP is online.
    <li>2 - The geographic mirroring role is resuming.
    <li>3 - The system is resuming and the IASP is online and performing synchronization.
    <li>4 - The geographic mirroring mirror copy is suspended and geographic mirroring is not being performed.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return   The mirror state of the mirror copy.
    **/
    public int getgeographicMirrorCpStat() {
    	return geographicMirrorCpStat_;
    }
    
    /**
    Returns   The condition of the data on the target.
    <ul>
    <li>0 - Geographic mirroring is not configured.
    <li>1 - The remote copy is absolutely in sync with the production copy.
    <li>2 - The remote copy contains usable data. A detached mirror copy always has this state.
    <li>3 - There is incoherent data in the mirror copy and the data cannot be used.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return    The condition of the data on the target.
    **/
    public int getgeographicMirrorCpDataStat() {
    	return geographicMirrorCpDataStat_;
    }
    
    /**
    Returns   The number of seconds the production node waits for a response from the mirror copy node before suspending geographic mirroring.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return    The number of seconds the production node waits for a response from the mirror copy node before suspending geographic mirroring.
    **/
    public int getgeographicMirrorSuspendTimeout() {
    	return geographicMirrorSuspendTimeout_;
    }
    
    /**
    Returns   The performance mode of the mirroring.
    <ul>
    <li>-1 - The performance mode is unknown.
    <li>0 - There is no performance mode.
    <li>1 - The performance mode is synchronous.
    <li>2 - The performance mode is asynchronous.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return    The performance mode of the mirroring.
    **/
    public int getgeographicMirrorPerfMode() {
    	return geographicMirrorPerfMode_;
    }
    
    /**
    Returns   Indicates whether disk units in the ASP were found on the system.
    <ul>
    <li>0 - No disk units were found.
    <li>1 - The disk unit that is used to provide the identity of the ASP was found but some other disk units were not found.
    <li>2 - All disk units were found.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return    Indicates whether disk units in the ASP were found on the system.
    **/
    public int getdiskUnitpresence() {
    	return diskUnitpresence_;
    }
    
    /**
    Returns   The resume priority following a suspend.
    <ul>
    <li>-1 - The priority is unknown.
    <li>0 - There is no priority.
    <li>1 - The priority is low.
    <li>2 - The priority is medium.
    <li>3 - The priority is high.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0200 FORMAT_0200}
    @return    The resume priority following a suspend.
    **/
    public int getgeographicMirrorResumePriority() {
    	return geographicMirrorResumePriority_;
    }
    
    //YASP0300
    /**
    Returns   The type of disk unit
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The type of disk unit
    **/
    public String getdiskType() { 
    	return diskType_;
    }
    
    /**
    Returns   The model of the disk unit.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The model of the disk unit.
    **/
    public String getdiskModel() { 
    	return diskModel_;
    }
    
    /**
    Returns   The serial number of the disk unit.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The serial number of the disk unit.
    **/
    public String getdiskSerialNumber() { 
    	return diskSerialNumber_;
    } 
    
    /**
    Returns    The unique system-assigned name of the disk unit.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return     The unique system-assigned name of the disk unit.
    **/
    public String getdiskResourceName() { 
    	return diskResourceName_;
    }
    
    /**
    Returns   The total size of the disk unit in megabytes.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The total size of the disk unit in megabytes.
    **/
    public int getdiskCapacity() { 
    	return diskCapacity_;
    } 
    
    /**
    Returns   The number of megabytes of space not currently assigned.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The number of megabytes of space not currently assigned.
    **/
    public int getdiskStorageAvailable() { 
    	return diskStorageAvailable_;
    } 
    
    /**
    Returns   The number of megabytes of auxiliary storage on the disk unit reserved for use by the system. This storage contains system data related to hardware configuration, hardware error processing, and so forth.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The number of megabytes of auxiliary storage on the disk unit reserved for use by the system. This storage contains system data related to hardware configuration, hardware error processing, and so forth.
    **/
    public int getdiskStorageReserved() { 
    	return diskStorageReserved_;
    } 
    
    /**
    Returns   The mirrored status of the mirrored pair of which this unit is a member.
    <ul>
    <li>0 - One mirrored unit of a mirrored pair is not active.
    <li>1 - Both mirrored units of a mirrored pair are active.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The mirrored status of the mirrored pair of which this unit is a member.
    **/
    public String getmirroredUnitProtected() { 
    	return mirroredUnitProtected_;
    }
    
    /**
    Returns   This mirrored unit reported present. Information concerning this unit may or may not be current.
    <ul>
    <li>0 - The mirrored unit is missing. Information concerning the unit may not be current.
    <li>1 - The mirrored unit reported at the time this information was gathered. The information is current to that point in time.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    This mirrored unit reported present. Information concerning this unit may or may not be current.
    **/
    public String getmirroredUnitReported() { 
    	return mirroredUnitReported_;
    } 
    
    /**
    Returns   The mirrored status of the unit.
    <ul>
    <li>0 - This mirrored unit of a mirrored pair is active (that is, online with current data).
    <li>1 - This mirrored unit is being synchronized.
    <li>2 - This mirrored unit is suspended.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The mirrored status of the unit.
    **/
    public String getmirroredUnitStatus() { 
    	return mirroredUnitStatus_;
    } 
    
    /**
    Returns   The type of RAID protection that has been assigned to this disk unit. The following values are returned:
    <ul>
    <li>0 - No storage protection has been set up for this disk unit.
    <li>5 - This disk unit has been set up with RAID 5 protection.
    <li>6 - This disk unit has been set up with RAID 6 protection.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The type of RAID protection that has been assigned to this disk unit.
    **/
    public String getRAIDType() { 
    	return RAIDType_;
    } 
    
    /**
    Returns   The disk status for the disk unit. The possible values for the field are:
    <ul>
    <li>0 -	There is no unit control value.
    <li>1 -	The disk unit is active.
    <li>2 -	The disk unit has failed.
    <li>3 -	Some other disk unit in the disk subsystem has failed.
    <li>4 -	There is a hardware failure within the disk subsystem that affects performance, but does not affect the function of the disk unit.
    <li>5 -	There is a hardware failure within the disk subsystem that does not affect the function or performance of the disk unit.
    <li>6 -	The disk unit's parity protection is being rebuilt.
    <li>7 -	The disk unit is not ready.
    <li>8 -	The disk unit is write protected.
    <li>9 -	The disk unit is busy.
    <li>10 - The disk unit is not operational.
    <li>11 - The disk unit has returned a status that is not recognizable by the system.
    <li>12 - The disk unit cannot be accessed.
    <li>13 - The disk unit is read/write protected.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The disk status for the disk unit.
    **/
    public int getunitControl() { 
    	return unitControl_;
    } 
    
    /**
    Returns   The number of 512-byte blocks transferred from the disk unit since the last IPL. This value wraps around to zero and continues counting in the case of an overflow of the field with no indication of the overflow having occurred.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The number of 512-byte blocks transferred from the disk unit since the last IPL. This value wraps around to zero and continues counting in the case of an overflow of the field with no indication of the overflow having occurred.
    **/
    public int getblockTransfer2MainStorage() { 
    	return blockTransfer2MainStorage_;
    } 
    
    /**
    Returns   The number of 512-byte blocks transferred to the disk unit since the last IPL. This value wraps around to zero and continues counting in the case of an overflow of the field with no indication of the overflow having occurred.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The number of 512-byte blocks transferred to the disk unit since the last IPL. This value wraps around to zero and continues counting in the case of an overflow of the field with no indication of the overflow having occurred.
    **/
    public int getblockTransferFromMainStorage() { 
    	return blockTransferFromMainStorage_;
    } 
    
    /**
    Returns   The number of input data transfer requests processed for the disk unit since the last IPL. If the field overflows, this value wraps around to zero and continues counting, with no indication that the overflow occurred. This value is not directly related to the number of blocks transferred for the disk unit because the number of blocks to be transferred for a given transfer request can vary greatly.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The number of input data transfer requests processed for the disk unit since the last IPL. If the field overflows, this value wraps around to zero and continues counting, with no indication that the overflow occurred. This value is not directly related to the number of blocks transferred for the disk unit because the number of blocks to be transferred for a given transfer request can vary greatly.
    **/
    public int getreqDataTransfer2MainStorage() { 
    	return reqDataTransfer2MainStorage_;
    } 
    
    /**
    Returns   The number of output data transfer requests processed for the disk unit since the last IPL. If the field overflows, this value wraps around to zero and continues counting, with no indication that the overflow occurred. This value is not directly related to the number of blocks transferred for the disk unit because the number of blocks to be transferred for a given transfer request can vary greatly.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The number of output data transfer requests processed for the disk unit since the last IPL. If the field overflows, this value wraps around to zero and continues counting, with no indication that the overflow occurred. This value is not directly related to the number of blocks transferred for the disk unit because the number of blocks to be transferred for a given transfer request can vary greatly.
    **/
    public int getreqDataTransferFromMainStorage() { 
    	return reqDataTransferFromMainStorage_;
    } 
    
    /**
    Returns   The number of 512-byte blocks of permanent storage transferred to the disk unit since the last IPL. If the field overflows, this value wraps around to zero and continues counting, with no indication that the overflow condition occurred.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The number of 512-byte blocks of permanent storage transferred to the disk unit since the last IPL. If the field overflows, this value wraps around to zero and continues counting, with no indication that the overflow condition occurred.
    **/
    public int getpermanentBlockTransferFromMainStorage() { 
    	return permanentBlockTransferFromMainStorage_;
    } 
    
    /**
    Returns    The number of output permanent data transfer requests processed for the disk unit since the last IPL. If the field overflows, this value wraps around to zero and continues counting, with no indication that the overflow occurred. This value is not directly related to the permanent blocks transferred from main storage for the disk unit because the number of blocks transferred for a given transfer request can vary greatly.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return     The number of output permanent data transfer requests processed for the disk unit since the last IPL. If the field overflows, this value wraps around to zero and continues counting, with no indication that the overflow occurred. This value is not directly related to the permanent blocks transferred from main storage for the disk unit because the number of blocks transferred for a given transfer request can vary greatly.
    **/
    public int getreqPermanentBlockTransferFromMainStorage() { 
    	return reqPermanentBlockTransferFromMainStorage_;
    } 
    
    /**
    Returns   The number of times the disk queue was checked to determine whether or not the queue is empty.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The number of times the disk queue was checked to determine whether or not the queue is empty.
    **/
    public int getsampleCount() { 
    	return sampleCount_;
    } 
    
    /**
    Returns   The number of times the disk queue was empty during the same time period that the sample count was taken.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The number of times the disk queue was empty during the same time period that the sample count was taken.
    **/
    public int getnotBusyCount() { 
    	return notBusyCount_;
    } 
    
    /**
    Returns   The current status of compression for this disk unit. The following values are returned:
    <ul>
    <li>0 - Compression for this disk unit is not active.
    <li>1 - Compression for this disk unit is active.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The current status of compression for this disk unit.
    **/
    public String getcompressedStatus() { 
    	return compressedStatus_;
    } 
    
    /**
    Returns   The type of protection that has been assigned to this disk unit. The following values are returned:
    <ul>
    <li>0 - No storage protection has been set up for this disk unit.
    <li>1 - This disk unit has been set up with mirrored protection.
    <li>2 - This disk unit is part of a parity protection array.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The type of protection that has been assigned to this disk unit.
    **/
    public String getdiskProtectionType() { 
    	return diskProtectionType_;
    } 
    
    /**
    Returns   The compression attribute of the disk unit.
    <ul>
    <li>0 - Data on the disk unit is not compressed.
    <li>1 - Data on the disk unit is compressed.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    The compression attribute of the disk unit.
    **/
    public String getcompressedUnit() { 
    	return compressedUnit_;
    } 
    
    /**
    Returns   An indicator of whether new storage allocations are allowed on the disk unit.
    <ul>
    <li>0 - The disk unit does not allow new storage allocations.
    <li>1 - The disk unit allows new storage allocations.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    An indicator of whether new storage allocations are allowed on the disk unit.
    **/
    public String getstroageAllocationRestrictUnit() { 
    	return stroageAllocationRestrictUnit_;
    } 
    
    /**
    Returns   A disk unit may have multiple resource names. Each resource name represents a unique connection to the disk unit. All active connections are used to communicate with the disk unit. This attribute indicates whether the disk unit has more than one connection.
    <ul>
    <li>0 - The disk unit has only one connection.
    <li>1 - The disk unit has more than one connection.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    A disk unit may have multiple resource names. Each resource name represents a unique connection to the disk unit. All active connections are used to communicate with the disk unit. This attribute indicates whether the disk unit has more than one connection.
    **/
    public String getmultiConnectionUnit() {
    	return multiConnectionUnit_;
    }
    
    /**
    Returns   An indicator of whether a disk unit is in an availability parity set.
    <ul>
    <li>0 - The disk unit is not in an availability parity set.
    <li>1 - The disk unit is in an availability parity set.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0300 FORMAT_0300}
    @return    An indicator of whether a disk unit is in an availability parity set.
    **/
    public String getavaParitySetUnit() { 
    	return avaParitySetUnit_;
    } 
    
    //YASP0400
    /**
    Returns   An error code that identifies a hardware problem related to the use of an independent ASP (error codes are not available for other ASP types). An error code of less than 1000 is an error. An error code of 1000 or greater is a warning. The possible error code values and their descriptions and recovery actions can be found in message file QSYS/QCPFMSG in these messages: CPDB8E0, CPDB8E1, CPDB8E2 and CPDB8E3.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0400 FORMAT_0400}
    @return    An error code that identifies a hardware problem related to the use of an independent ASP (error codes are not available for other ASP types). An error code of less than 1000 is an error. An error code of 1000 or greater is a warning. The possible error code values and their descriptions and recovery actions can be found in message file QSYS/QCPFMSG in these messages: CPDB8E0, CPDB8E1, CPDB8E2 and CPDB8E3.
    **/
    public String geterrorCode() {
    	return errorCode_;
    }
    
    /**
    Returns   One of a pair of mirrored units. The possible values are:
    <ul>
    <li>0 - The disk unit is not mirrored or the error does not relate to a specific disk unit.
    <li>1 - The first half of a mirrored pair.
    <li>2 - The second half of a mirrored pair.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0400 FORMAT_0400}
    @return    One of a pair of mirrored units. The possible values are:
    **/
    public String getmirrorUnitIdentifier() { 
    	return mirrorUnitIdentifier_;
    } 

    //YASP0500
    /**
    Returns   The state transition target of the ASP. This field is valid for an IASP if its status is either ACTIVE or AVAILABLE, otherwise hexadecimal zeros are returned. The possible values are:
    <ul>
    <li>0 - The ASP is not in a state transition.
    <li>1 - The disk unit has more than one connection.
    </ul>
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0500 FORMAT_0500}
    @return    The state transition target of the ASP. This field is valid for an IASP if its status is either ACTIVE or AVAILABLE, otherwise hexadecimal zeros are returned.
    **/
    public String gettranfitionTarget() {
    	 return tranfitionTarget_;
    }
    
    /**
    Returns   The function that is currently running as part of a state transition for an ASP.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0500 FORMAT_0500}
    @return    The function that is currently running as part of a state transition for an ASP.
    **/
    public String getfunction() {
    	 return function_;
    }
    
    /**
    Returns   The number of entities that have been processed by the function. The definition of an entity depends on the function; it may be an object, an address, a data structure, and so on.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0500 FORMAT_0500}
    @return    The number of entities that have been processed by the function. The definition of an entity depends on the function; it may be an object, an address, a data structure, and so on.
    **/
    public int getcurrentCount() {
    	 return currentCount_;
    }
    
    /**
    Returns   The total number of entities to be processed by the function. The definition of an entity depends on the function; it may be an object, an address, a data structure, and so forth.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0500 FORMAT_0500}
    @return    The total number of entities to be processed by the function. The definition of an entity depends on the function; it may be an object, an address, a data structure, and so forth.
    **/
    public int gettotalCount() {
    	 return totalCount_;
    }
    
    /**
    Returns   The number of items that have been processed by the function. The definition of an item depends on the function.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0500 FORMAT_0500}
    @return    The number of items that have been processed by the function. The definition of an item depends on the function.
    **/
    public int getcurrentItemCount() {
    	 return currentItemCount_;
    }
    
    /**
    Returns   The total number of items that are to be processed by the function. An item is a part of an object, but has no precise definition; that is, items differ by object type if an object type has them at all.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0500 FORMAT_0500}
    @return    The total number of items that are to be processed by the function. An item is a part of an object, but has no precise definition; that is, items differ by object type if an object type has them at all.
    **/
    public int gettotalItemCount() {
    	 return totalItemCount_;
    }

    /**
    Returns   The time that has elapsed since the start of the function. This value is in the format HHMMSS where HH is the hour, MM is the minute, and SS is the second.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0500 FORMAT_0500}
    @return    The time that has elapsed since the start of the function. This value is in the format HHMMSS where HH is the hour, MM is the minute, and SS is the second.
    **/
    public String getelapsedTime() {
    	 return elapsedTime_;
    }
    
    //YASP0600
    /**
    Returns   An identification of how an ASP is being used
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0600 FORMAT_0600}
    @return    An identification of how an ASP is being used
    **/
    public int getuseIdentification() { 
        return useIdentification_;
   }
    
    /**
    Returns   The simple job name of the job that is using an ASP.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0600 FORMAT_0600}
    @return    The simple job name of the job that is using an ASP.
    **/
   	public String getjobName() { 
        return jobName_;
   }
   	
   	/**
    Returns   The user name under which the job that is using an ASP was started.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0600 FORMAT_0600}
    @return    The user name under which the job that is using an ASP was started.
    **/
   	public String getjobUserName() { 
        return jobUserName_;
   }
   	
   	/**
    Returns   The system-assigned job number of the job that is using an ASP.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0600 FORMAT_0600}
    @return    The system-assigned job number of the job that is using an ASP.
    **/
   	public String getjobNumber() { 
        return jobNumber_;
   }
   	
   	/**
    Returns   A value that addresses a particular thread within a job. While the thread identifier uniquely identifies the thread within the job, the thread handle can improve performance when referencing the thread. 
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0600 FORMAT_0600}
    @return    A value that addresses a particular thread within a job. While the thread identifier uniquely identifies the thread within the job, the thread handle can improve performance when referencing the thread. 
    **/
   	public String getthreadIdentifier() { 
        return threadIdentifier_;
   }
   	
   	/**
    Returns   The current status of the thread. Refer to the description of Thread status in Work Management API Attribute Descriptions for the possible values that can be returned.
    <p>Available in format: {@link com.ibm.as400.access.list.AspOpenList#FORMAT_0600 FORMAT_0600}
    @return    The current status of the thread. Refer to the description of Thread status in Work Management API Attribute Descriptions for the possible values that can be returned.
    **/
   	public String getthreadStatus() { 
        return threadStatus_;
   }
   	
   	/**
    Returns the format that was used by AspOpenList to generate this item.
    @return  The format.
    @see  com.ibm.as400.access.list.AspListItem#getFormat
    **/
   public String getFormat()
   {
       return format_;
   }
}

