/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.extensions.gcp.storage;

import com.google.api.client.util.DateTime;
import com.google.api.services.storage.model.Objects;
import com.google.api.services.storage.model.StorageObject;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import org.apache.beam.sdk.extensions.gcp.options.GcsOptions;
import org.apache.beam.sdk.extensions.gcp.storage.GcsCreateOptions;
import org.apache.beam.sdk.extensions.gcp.storage.GcsResourceId;
import org.apache.beam.sdk.extensions.gcp.util.GcsUtil;
import org.apache.beam.sdk.extensions.gcp.util.gcsfs.GcsPath;
import org.apache.beam.sdk.io.FileSystem;
import org.apache.beam.sdk.io.fs.CreateOptions;
import org.apache.beam.sdk.io.fs.MatchResult;
import org.apache.beam.sdk.metrics.Counter;
import org.apache.beam.sdk.metrics.Metrics;
import org.apache.beam.vendor.guava.v20_0.com.google.common.annotations.VisibleForTesting;
import org.apache.beam.vendor.guava.v20_0.com.google.common.base.MoreObjects;
import org.apache.beam.vendor.guava.v20_0.com.google.common.base.Preconditions;
import org.apache.beam.vendor.guava.v20_0.com.google.common.base.Stopwatch;
import org.apache.beam.vendor.guava.v20_0.com.google.common.collect.FluentIterable;
import org.apache.beam.vendor.guava.v20_0.com.google.common.collect.ImmutableList;
import org.apache.beam.vendor.guava.v20_0.com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class GcsFileSystem
extends FileSystem<GcsResourceId> {
    private static final Logger LOG = LoggerFactory.getLogger(GcsFileSystem.class);
    private final GcsOptions options;
    private Counter numCopies;
    private Counter copyTimeMsec;

    GcsFileSystem(GcsOptions options) {
        this.options = Preconditions.checkNotNull(options, "options");
        if (options.getGcsPerformanceMetrics().booleanValue()) {
            this.numCopies = Metrics.counter(GcsFileSystem.class, "num_copies");
            this.copyTimeMsec = Metrics.counter(GcsFileSystem.class, "copy_time_msec");
        }
    }

    @Override
    protected List<MatchResult> match(List<String> specs) throws IOException {
        List<GcsPath> gcsPaths = this.toGcsPaths(specs);
        ArrayList<GcsPath> globs = Lists.newArrayList();
        ArrayList<GcsPath> nonGlobs = Lists.newArrayList();
        ArrayList<Boolean> isGlobBooleans = Lists.newArrayList();
        for (GcsPath path : gcsPaths) {
            if (GcsUtil.isWildcard(path)) {
                globs.add(path);
                isGlobBooleans.add(true);
                continue;
            }
            nonGlobs.add(path);
            isGlobBooleans.add(false);
        }
        Iterator<MatchResult> globsMatchResults = this.matchGlobs(globs).iterator();
        Iterator<MatchResult> nonGlobsMatchResults = this.matchNonGlobs(nonGlobs).iterator();
        ImmutableList.Builder ret = ImmutableList.builder();
        for (Boolean isGlob : isGlobBooleans) {
            if (isGlob.booleanValue()) {
                Preconditions.checkState(globsMatchResults.hasNext(), "Expect globsMatchResults has next: %s", globs);
                ret.add(globsMatchResults.next());
                continue;
            }
            Preconditions.checkState(nonGlobsMatchResults.hasNext(), "Expect nonGlobsMatchResults has next: %s", nonGlobs);
            ret.add(nonGlobsMatchResults.next());
        }
        Preconditions.checkState(!globsMatchResults.hasNext(), "Expect no more elements in globsMatchResults.");
        Preconditions.checkState(!nonGlobsMatchResults.hasNext(), "Expect no more elements in nonGlobsMatchResults.");
        return ret.build();
    }

    @Override
    protected WritableByteChannel create(GcsResourceId resourceId, CreateOptions createOptions) throws IOException {
        if (createOptions instanceof GcsCreateOptions) {
            return this.options.getGcsUtil().create(resourceId.getGcsPath(), createOptions.mimeType(), ((GcsCreateOptions)createOptions).gcsUploadBufferSizeBytes());
        }
        return this.options.getGcsUtil().create(resourceId.getGcsPath(), createOptions.mimeType());
    }

    @Override
    protected ReadableByteChannel open(GcsResourceId resourceId) throws IOException {
        return this.options.getGcsUtil().open(resourceId.getGcsPath());
    }

    @Override
    protected void rename(List<GcsResourceId> srcResourceIds, List<GcsResourceId> destResourceIds) throws IOException {
        this.copy(srcResourceIds, destResourceIds);
        this.delete((Collection<GcsResourceId>)srcResourceIds);
    }

    @Override
    protected void delete(Collection<GcsResourceId> resourceIds) throws IOException {
        this.options.getGcsUtil().remove(this.toFilenames(resourceIds));
    }

    @Override
    protected GcsResourceId matchNewResource(String singleResourceSpec, boolean isDirectory) {
        if (isDirectory) {
            if (!singleResourceSpec.endsWith("/")) {
                singleResourceSpec = singleResourceSpec + "/";
            }
        } else {
            Preconditions.checkArgument(!singleResourceSpec.endsWith("/"), "Expected a file path, but [%s], ends with '/'. This is unsupported in GcsFileSystem.", (Object)singleResourceSpec);
        }
        GcsPath path = GcsPath.fromUri(singleResourceSpec);
        return GcsResourceId.fromGcsPath(path);
    }

    @Override
    protected void copy(List<GcsResourceId> srcResourceIds, List<GcsResourceId> destResourceIds) throws IOException {
        Stopwatch stopwatch = Stopwatch.createStarted();
        this.options.getGcsUtil().copy(this.toFilenames(srcResourceIds), this.toFilenames(destResourceIds));
        stopwatch.stop();
        if (this.options.getGcsPerformanceMetrics().booleanValue()) {
            this.numCopies.inc(srcResourceIds.size());
            this.copyTimeMsec.inc(stopwatch.elapsed(TimeUnit.MILLISECONDS));
        }
    }

    @Override
    protected String getScheme() {
        return "gs";
    }

    private List<MatchResult> matchGlobs(List<GcsPath> globs) {
        return FluentIterable.from(globs).transform(gcsPath -> {
            try {
                return this.expand((GcsPath)gcsPath);
            }
            catch (IOException e) {
                return MatchResult.create(MatchResult.Status.ERROR, e);
            }
        }).toList();
    }

    @VisibleForTesting
    MatchResult expand(GcsPath gcsPattern) throws IOException {
        Objects objects;
        String prefix = GcsUtil.getNonWildcardPrefix(gcsPattern.getObject());
        Pattern p = Pattern.compile(GcsUtil.wildcardToRegexp(gcsPattern.getObject()));
        LOG.debug("matching files in bucket {}, prefix {} against pattern {}", gcsPattern.getBucket(), prefix, p.toString());
        String pageToken = null;
        ArrayList<MatchResult.Metadata> results = new ArrayList<MatchResult.Metadata>();
        while ((objects = this.options.getGcsUtil().listObjects(gcsPattern.getBucket(), prefix, pageToken)).getItems() != null) {
            for (StorageObject o : objects.getItems()) {
                String name = o.getName();
                if (!p.matcher(name).matches() || name.endsWith("/")) continue;
                LOG.debug("Matched object: {}", (Object)name);
                results.add(this.toMetadata(o));
            }
            pageToken = objects.getNextPageToken();
            if (pageToken != null) continue;
        }
        return MatchResult.create(MatchResult.Status.OK, results);
    }

    @VisibleForTesting
    List<MatchResult> matchNonGlobs(List<GcsPath> gcsPaths) throws IOException {
        List<GcsUtil.StorageObjectOrIOException> results = this.options.getGcsUtil().getObjects(gcsPaths);
        ImmutableList.Builder ret = ImmutableList.builder();
        for (GcsUtil.StorageObjectOrIOException result : results) {
            ret.add(this.toMatchResult(result));
        }
        return ret.build();
    }

    private MatchResult toMatchResult(GcsUtil.StorageObjectOrIOException objectOrException) {
        IOException exception = objectOrException.ioException();
        if (exception instanceof FileNotFoundException) {
            return MatchResult.create(MatchResult.Status.NOT_FOUND, exception);
        }
        if (exception != null) {
            return MatchResult.create(MatchResult.Status.ERROR, exception);
        }
        StorageObject object = objectOrException.storageObject();
        assert (object != null);
        return MatchResult.create(MatchResult.Status.OK, ImmutableList.of(this.toMetadata(object)));
    }

    private MatchResult.Metadata toMetadata(StorageObject storageObject) {
        MatchResult.Metadata.Builder ret = MatchResult.Metadata.builder().setIsReadSeekEfficient(true).setResourceId(GcsResourceId.fromGcsPath(GcsPath.fromObject(storageObject)));
        BigInteger size = MoreObjects.firstNonNull(storageObject.getSize(), BigInteger.ZERO);
        ret.setSizeBytes(size.longValue());
        DateTime lastModified = MoreObjects.firstNonNull(storageObject.getUpdated(), new DateTime(0L));
        ret.setLastModifiedMillis(lastModified.getValue());
        return ret.build();
    }

    private List<String> toFilenames(Collection<GcsResourceId> resources) {
        return FluentIterable.from(resources).transform(resource -> resource.getGcsPath().toString()).toList();
    }

    private List<GcsPath> toGcsPaths(Collection<String> specs) {
        return FluentIterable.from(specs).transform(GcsPath::fromUri).toList();
    }
}

