/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.extensions.gcp.util;

import com.google.api.client.googleapis.batch.BatchRequest;
import com.google.api.client.googleapis.batch.json.JsonBatchCallback;
import com.google.api.client.googleapis.json.GoogleJsonError;
import com.google.api.client.googleapis.json.GoogleJsonResponseException;
import com.google.api.client.http.HttpHeaders;
import com.google.api.client.http.HttpRequestInitializer;
import com.google.api.client.util.BackOff;
import com.google.api.client.util.Sleeper;
import com.google.api.services.storage.Storage;
import com.google.api.services.storage.model.Bucket;
import com.google.api.services.storage.model.Objects;
import com.google.api.services.storage.model.RewriteResponse;
import com.google.api.services.storage.model.StorageObject;
import com.google.auto.value.AutoValue;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageReadChannel;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageWriteChannel;
import com.google.cloud.hadoop.gcsio.ObjectWriteConditions;
import com.google.cloud.hadoop.util.ApiErrorExtractor;
import com.google.cloud.hadoop.util.AsyncWriteChannelOptions;
import com.google.cloud.hadoop.util.ClientRequestHelper;
import com.google.cloud.hadoop.util.ResilientOperation;
import com.google.cloud.hadoop.util.RetryDeterminer;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.channels.SeekableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.AccessDeniedException;
import java.nio.file.FileAlreadyExistsException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nullable;
import org.apache.beam.sdk.extensions.gcp.options.GcsOptions;
import org.apache.beam.sdk.extensions.gcp.util.AutoValue_GcsUtil_StorageObjectOrIOException;
import org.apache.beam.sdk.extensions.gcp.util.BackOffAdapter;
import org.apache.beam.sdk.extensions.gcp.util.Transport;
import org.apache.beam.sdk.extensions.gcp.util.gcsfs.GcsPath;
import org.apache.beam.sdk.options.DefaultValueFactory;
import org.apache.beam.sdk.options.PipelineOptions;
import org.apache.beam.sdk.util.FluentBackoff;
import org.apache.beam.sdk.util.MoreFutures;
import org.apache.beam.vendor.guava.v20_0.com.google.common.annotations.VisibleForTesting;
import org.apache.beam.vendor.guava.v20_0.com.google.common.base.Preconditions;
import org.apache.beam.vendor.guava.v20_0.com.google.common.collect.ImmutableList;
import org.apache.beam.vendor.guava.v20_0.com.google.common.collect.Lists;
import org.apache.beam.vendor.guava.v20_0.com.google.common.util.concurrent.ListeningExecutorService;
import org.apache.beam.vendor.guava.v20_0.com.google.common.util.concurrent.MoreExecutors;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GcsUtil {
    private static final Logger LOG = LoggerFactory.getLogger(GcsUtil.class);
    private static final long MAX_LIST_ITEMS_PER_CALL = 1024L;
    private static final Pattern GLOB_PREFIX = Pattern.compile("(?<PREFIX>[^\\[*?]*)[\\[*?].*");
    private static final int MAX_REQUESTS_PER_BATCH = 100;
    private static final int MAX_CONCURRENT_BATCHES = 256;
    private static final FluentBackoff BACKOFF_FACTORY = FluentBackoff.DEFAULT.withMaxRetries(10).withInitialBackoff(Duration.standardSeconds(1L));
    private Storage storageClient;
    private final HttpRequestInitializer httpRequestInitializer;
    @Nullable
    private final Integer uploadBufferSizeBytes;
    private final ApiErrorExtractor errorExtractor = new ApiErrorExtractor();
    final ExecutorService executorService;
    @Nullable
    @VisibleForTesting
    Long maxBytesRewrittenPerCall;
    @Nullable
    @VisibleForTesting
    AtomicInteger numRewriteTokensUsed;

    public static String getNonWildcardPrefix(String globExp) {
        Matcher m = GLOB_PREFIX.matcher(globExp);
        Preconditions.checkArgument(m.matches(), String.format("Glob expression: [%s] is not expandable.", globExp));
        return m.group("PREFIX");
    }

    public static String wildcardToRegexp(String globExp) {
        StringBuilder dst = new StringBuilder();
        char[] src = globExp.replace("**/*", "**").toCharArray();
        int i = 0;
        block6: while (i < src.length) {
            char c = src[i++];
            switch (c) {
                case '*': {
                    if (i < src.length && src[i] == '*') {
                        dst.append(".*");
                        ++i;
                        continue block6;
                    }
                    dst.append("[^/]*");
                    continue block6;
                }
                case '?': {
                    dst.append("[^/]");
                    continue block6;
                }
                case '$': 
                case '(': 
                case ')': 
                case '+': 
                case '.': 
                case '^': 
                case '{': 
                case '|': 
                case '}': {
                    dst.append('\\').append(c);
                    continue block6;
                }
                case '\\': {
                    i = GcsUtil.doubleSlashes(dst, src, i);
                    continue block6;
                }
            }
            dst.append(c);
        }
        return dst.toString();
    }

    public static boolean isWildcard(GcsPath spec) {
        return GLOB_PREFIX.matcher(spec.getObject()).matches();
    }

    private GcsUtil(Storage storageClient, HttpRequestInitializer httpRequestInitializer, ExecutorService executorService, @Nullable Integer uploadBufferSizeBytes) {
        this.storageClient = storageClient;
        this.httpRequestInitializer = httpRequestInitializer;
        this.uploadBufferSizeBytes = uploadBufferSizeBytes;
        this.executorService = executorService;
        this.maxBytesRewrittenPerCall = null;
        this.numRewriteTokensUsed = null;
    }

    protected void setStorageClient(Storage storageClient) {
        this.storageClient = storageClient;
    }

    public List<GcsPath> expand(GcsPath gcsPattern) throws IOException {
        Objects objects;
        Pattern p = null;
        String prefix = null;
        if (!GcsUtil.isWildcard(gcsPattern)) {
            try {
                this.getObject(gcsPattern);
                return ImmutableList.of(gcsPattern);
            }
            catch (FileNotFoundException e) {
                return ImmutableList.of();
            }
        }
        prefix = GcsUtil.getNonWildcardPrefix(gcsPattern.getObject());
        p = Pattern.compile(GcsUtil.wildcardToRegexp(gcsPattern.getObject()));
        LOG.debug("matching files in bucket {}, prefix {} against pattern {}", gcsPattern.getBucket(), prefix, p.toString());
        String pageToken = null;
        ArrayList<GcsPath> results = new ArrayList<GcsPath>();
        while ((objects = this.listObjects(gcsPattern.getBucket(), prefix, pageToken)).getItems() != null) {
            for (StorageObject o : objects.getItems()) {
                String name = o.getName();
                if (!p.matcher(name).matches() || name.endsWith("/")) continue;
                LOG.debug("Matched object: {}", (Object)name);
                results.add(GcsPath.fromObject(o));
            }
            pageToken = objects.getNextPageToken();
            if (pageToken != null) continue;
        }
        return results;
    }

    @Nullable
    @VisibleForTesting
    Integer getUploadBufferSizeBytes() {
        return this.uploadBufferSizeBytes;
    }

    private static BackOff createBackOff() {
        return BackOffAdapter.toGcpBackOff(BACKOFF_FACTORY.backoff());
    }

    public long fileSize(GcsPath path) throws IOException {
        return this.getObject(path).getSize().longValue();
    }

    public StorageObject getObject(GcsPath gcsPath) throws IOException {
        return this.getObject(gcsPath, GcsUtil.createBackOff(), Sleeper.DEFAULT);
    }

    @VisibleForTesting
    StorageObject getObject(GcsPath gcsPath, BackOff backoff, Sleeper sleeper) throws IOException {
        Storage.Objects.Get getObject = this.storageClient.objects().get(gcsPath.getBucket(), gcsPath.getObject());
        try {
            return ResilientOperation.retry(ResilientOperation.getGoogleRequestCallable(getObject), backoff, RetryDeterminer.SOCKET_ERRORS, IOException.class, sleeper);
        }
        catch (IOException | InterruptedException e) {
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            if (e instanceof IOException && this.errorExtractor.itemNotFound((IOException)e)) {
                throw new FileNotFoundException(gcsPath.toString());
            }
            throw new IOException(String.format("Unable to get the file object for path %s.", gcsPath), e);
        }
    }

    public List<StorageObjectOrIOException> getObjects(List<GcsPath> gcsPaths) throws IOException {
        ArrayList<StorageObjectOrIOException[]> results = new ArrayList<StorageObjectOrIOException[]>();
        GcsUtil.executeBatches(this.makeGetBatches(gcsPaths, results));
        ImmutableList.Builder ret = ImmutableList.builder();
        for (StorageObjectOrIOException[] result : results) {
            ret.add(result[0]);
        }
        return ret.build();
    }

    public Objects listObjects(String bucket, String prefix, @Nullable String pageToken) throws IOException {
        Storage.Objects.List listObject = this.storageClient.objects().list(bucket);
        listObject.setMaxResults(1024L);
        listObject.setPrefix(prefix);
        if (pageToken != null) {
            listObject.setPageToken(pageToken);
        }
        try {
            return ResilientOperation.retry(ResilientOperation.getGoogleRequestCallable(listObject), GcsUtil.createBackOff(), RetryDeterminer.SOCKET_ERRORS, IOException.class);
        }
        catch (Exception e) {
            throw new IOException(String.format("Unable to match files in bucket %s, prefix %s.", bucket, prefix), e);
        }
    }

    @VisibleForTesting
    List<Long> fileSizes(List<GcsPath> paths) throws IOException {
        List<StorageObjectOrIOException> results = this.getObjects(paths);
        ImmutableList.Builder ret = ImmutableList.builder();
        for (StorageObjectOrIOException result : results) {
            ret.add(this.toFileSize(result));
        }
        return ret.build();
    }

    private Long toFileSize(StorageObjectOrIOException storageObjectOrIOException) throws IOException {
        if (storageObjectOrIOException.ioException() != null) {
            throw storageObjectOrIOException.ioException();
        }
        return storageObjectOrIOException.storageObject().getSize().longValue();
    }

    public SeekableByteChannel open(GcsPath path) throws IOException {
        return new GoogleCloudStorageReadChannel(this.storageClient, path.getBucket(), path.getObject(), this.errorExtractor, new ClientRequestHelper<StorageObject>());
    }

    public WritableByteChannel create(GcsPath path, String type) throws IOException {
        return this.create(path, type, this.uploadBufferSizeBytes);
    }

    public WritableByteChannel create(GcsPath path, String type, Integer uploadBufferSizeBytes) throws IOException {
        GoogleCloudStorageWriteChannel channel = new GoogleCloudStorageWriteChannel(this.executorService, this.storageClient, new ClientRequestHelper<StorageObject>(), path.getBucket(), path.getObject(), type, null, AsyncWriteChannelOptions.newBuilder().build(), new ObjectWriteConditions(), Collections.emptyMap());
        if (uploadBufferSizeBytes != null) {
            channel.setUploadBufferSize(uploadBufferSizeBytes);
        }
        channel.initialize();
        return channel;
    }

    public boolean bucketAccessible(GcsPath path) throws IOException {
        return this.bucketAccessible(path, GcsUtil.createBackOff(), Sleeper.DEFAULT);
    }

    public long bucketOwner(GcsPath path) throws IOException {
        return this.getBucket(path, GcsUtil.createBackOff(), Sleeper.DEFAULT).getProjectNumber().longValue();
    }

    public void createBucket(String projectId, Bucket bucket) throws IOException {
        this.createBucket(projectId, bucket, GcsUtil.createBackOff(), Sleeper.DEFAULT);
    }

    @VisibleForTesting
    boolean bucketAccessible(GcsPath path, BackOff backoff, Sleeper sleeper) throws IOException {
        try {
            return this.getBucket(path, backoff, sleeper) != null;
        }
        catch (FileNotFoundException | AccessDeniedException e) {
            return false;
        }
    }

    @Nullable
    @VisibleForTesting
    Bucket getBucket(GcsPath path, BackOff backoff, Sleeper sleeper) throws IOException {
        Storage.Buckets.Get getBucket = this.storageClient.buckets().get(path.getBucket());
        try {
            return ResilientOperation.retry(ResilientOperation.getGoogleRequestCallable(getBucket), backoff, new RetryDeterminer<IOException>(){

                @Override
                public boolean shouldRetry(IOException e) {
                    if (GcsUtil.this.errorExtractor.itemNotFound(e) || GcsUtil.this.errorExtractor.accessDenied(e)) {
                        return false;
                    }
                    return RetryDeterminer.SOCKET_ERRORS.shouldRetry(e);
                }
            }, IOException.class, sleeper);
        }
        catch (GoogleJsonResponseException e) {
            if (this.errorExtractor.accessDenied(e)) {
                throw new AccessDeniedException(path.toString(), null, e.getMessage());
            }
            if (this.errorExtractor.itemNotFound(e)) {
                throw new FileNotFoundException(e.getMessage());
            }
            throw e;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException(String.format("Error while attempting to verify existence of bucket gs://%s", path.getBucket()), e);
        }
    }

    @VisibleForTesting
    void createBucket(String projectId, Bucket bucket, BackOff backoff, Sleeper sleeper) throws IOException {
        Storage.Buckets.Insert insertBucket = this.storageClient.buckets().insert(projectId, bucket);
        insertBucket.setPredefinedAcl("projectPrivate");
        insertBucket.setPredefinedDefaultObjectAcl("projectPrivate");
        try {
            ResilientOperation.retry(ResilientOperation.getGoogleRequestCallable(insertBucket), backoff, new RetryDeterminer<IOException>(){

                @Override
                public boolean shouldRetry(IOException e) {
                    if (GcsUtil.this.errorExtractor.itemAlreadyExists(e) || GcsUtil.this.errorExtractor.accessDenied(e)) {
                        return false;
                    }
                    return RetryDeterminer.SOCKET_ERRORS.shouldRetry(e);
                }
            }, IOException.class, sleeper);
            return;
        }
        catch (GoogleJsonResponseException e) {
            if (this.errorExtractor.accessDenied(e)) {
                throw new AccessDeniedException(bucket.getName(), null, e.getMessage());
            }
            if (this.errorExtractor.itemAlreadyExists(e)) {
                throw new FileAlreadyExistsException(bucket.getName(), null, e.getMessage());
            }
            throw e;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException(String.format("Error while attempting to create bucket gs://%s for rproject %s", bucket.getName(), projectId), e);
        }
    }

    private static void executeBatches(List<BatchRequest> batches) throws IOException {
        ListeningExecutorService executor = MoreExecutors.listeningDecorator(new ThreadPoolExecutor(256, 256, 0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>()));
        ArrayList<CompletionStage<Void>> futures = new ArrayList<CompletionStage<Void>>();
        for (BatchRequest batch : batches) {
            futures.add(MoreFutures.runAsync(() -> batch.execute(), executor));
        }
        try {
            MoreFutures.get(MoreFutures.allAsList(futures));
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Interrupted while executing batch GCS request", e);
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof FileNotFoundException) {
                throw (FileNotFoundException)e.getCause();
            }
            throw new IOException("Error executing batch GCS request", e);
        }
        finally {
            executor.shutdown();
        }
    }

    @VisibleForTesting
    List<BatchRequest> makeGetBatches(Collection<GcsPath> paths, List<StorageObjectOrIOException[]> results) throws IOException {
        ArrayList<BatchRequest> batches = new ArrayList<BatchRequest>();
        for (List<GcsPath> filesToGet : Lists.partition(Lists.newArrayList(paths), 100)) {
            BatchRequest batch = this.createBatchRequest();
            for (GcsPath path : filesToGet) {
                results.add(this.enqueueGetFileSize(path, batch));
            }
            batches.add(batch);
        }
        return batches;
    }

    public void copy(Iterable<String> srcFilenames, Iterable<String> destFilenames) throws IOException {
        LinkedList<RewriteOp> rewrites = this.makeRewriteOps(srcFilenames, destFilenames);
        while (rewrites.size() > 0) {
            GcsUtil.executeBatches(this.makeCopyBatches(rewrites));
        }
    }

    LinkedList<RewriteOp> makeRewriteOps(Iterable<String> srcFilenames, Iterable<String> destFilenames) throws IOException {
        ArrayList<String> srcList = Lists.newArrayList(srcFilenames);
        ArrayList<String> destList = Lists.newArrayList(destFilenames);
        Preconditions.checkArgument(srcList.size() == destList.size(), "Number of source files %s must equal number of destination files %s", srcList.size(), destList.size());
        LinkedList<RewriteOp> rewrites = Lists.newLinkedList();
        for (int i = 0; i < srcList.size(); ++i) {
            GcsPath sourcePath = GcsPath.fromUri((String)srcList.get(i));
            GcsPath destPath = GcsPath.fromUri((String)destList.get(i));
            rewrites.addLast(new RewriteOp(sourcePath, destPath));
        }
        return rewrites;
    }

    List<BatchRequest> makeCopyBatches(LinkedList<RewriteOp> rewrites) throws IOException {
        ArrayList<BatchRequest> batches = new ArrayList<BatchRequest>();
        BatchRequest batch = this.createBatchRequest();
        Iterator it = rewrites.iterator();
        while (it.hasNext()) {
            RewriteOp rewrite = (RewriteOp)it.next();
            if (!rewrite.getReadyToEnqueue()) {
                it.remove();
                continue;
            }
            rewrite.enqueue(batch);
            if (batch.size() < 100) continue;
            batches.add(batch);
            batch = this.createBatchRequest();
        }
        if (batch.size() > 0) {
            batches.add(batch);
        }
        return batches;
    }

    List<BatchRequest> makeRemoveBatches(Collection<String> filenames) throws IOException {
        ArrayList<BatchRequest> batches = new ArrayList<BatchRequest>();
        for (List<String> filesToDelete : Lists.partition(Lists.newArrayList(filenames), 100)) {
            BatchRequest batch = this.createBatchRequest();
            for (String file : filesToDelete) {
                this.enqueueDelete(GcsPath.fromUri(file), batch);
            }
            batches.add(batch);
        }
        return batches;
    }

    public void remove(Collection<String> filenames) throws IOException {
        GcsUtil.executeBatches(this.makeRemoveBatches(filenames));
    }

    private StorageObjectOrIOException[] enqueueGetFileSize(final GcsPath path, BatchRequest batch) throws IOException {
        final StorageObjectOrIOException[] ret = new StorageObjectOrIOException[1];
        Storage.Objects.Get getRequest = this.storageClient.objects().get(path.getBucket(), path.getObject());
        getRequest.queue(batch, new JsonBatchCallback<StorageObject>(){

            @Override
            public void onSuccess(StorageObject response, HttpHeaders httpHeaders) throws IOException {
                ret[0] = StorageObjectOrIOException.create(response);
            }

            @Override
            public void onFailure(GoogleJsonError e, HttpHeaders httpHeaders) throws IOException {
                IOException ioException = GcsUtil.this.errorExtractor.itemNotFound(e) ? new FileNotFoundException(path.toString()) : new IOException(String.format("Error trying to get %s: %s", path, e));
                ret[0] = StorageObjectOrIOException.create(ioException);
            }
        });
        return ret;
    }

    private void enqueueDelete(final GcsPath file, BatchRequest batch) throws IOException {
        Storage.Objects.Delete deleteRequest = this.storageClient.objects().delete(file.getBucket(), file.getObject());
        deleteRequest.queue(batch, new JsonBatchCallback<Void>(){

            @Override
            public void onSuccess(Void obj, HttpHeaders responseHeaders) {
                LOG.debug("Successfully deleted {}", (Object)file);
            }

            @Override
            public void onFailure(GoogleJsonError e, HttpHeaders responseHeaders) throws IOException {
                if (e.getCode() != 404) {
                    throw new IOException(String.format("Error trying to delete %s: %s", file, e));
                }
                LOG.info("Ignoring failed deletion of file {} which already does not exist: {}", (Object)file, (Object)e);
            }
        });
    }

    private BatchRequest createBatchRequest() {
        return this.storageClient.batch(this.httpRequestInitializer);
    }

    private static int doubleSlashes(StringBuilder dst, char[] src, int i) {
        dst.append('\\');
        if (i - 1 != src.length) {
            dst.append(src[i]);
            ++i;
        } else {
            dst.append('\\');
        }
        return i;
    }

    @AutoValue
    public static abstract class StorageObjectOrIOException {
        @Nullable
        public abstract StorageObject storageObject();

        @Nullable
        public abstract IOException ioException();

        @VisibleForTesting
        public static StorageObjectOrIOException create(StorageObject storageObject) {
            return new AutoValue_GcsUtil_StorageObjectOrIOException(Preconditions.checkNotNull(storageObject, "storageObject"), null);
        }

        @VisibleForTesting
        public static StorageObjectOrIOException create(IOException ioException) {
            return new AutoValue_GcsUtil_StorageObjectOrIOException(null, Preconditions.checkNotNull(ioException, "ioException"));
        }
    }

    class RewriteOp
    extends JsonBatchCallback<RewriteResponse> {
        private GcsPath from;
        private GcsPath to;
        private boolean readyToEnqueue;
        @VisibleForTesting
        Storage.Objects.Rewrite rewriteRequest;

        public boolean getReadyToEnqueue() {
            return this.readyToEnqueue;
        }

        public void enqueue(BatchRequest batch) throws IOException {
            if (!this.readyToEnqueue) {
                throw new IOException(String.format("Invalid state for Rewrite, from=%s, to=%s, readyToEnqueue=%s", this.from, this.to, this.readyToEnqueue));
            }
            this.rewriteRequest.queue(batch, this);
            this.readyToEnqueue = false;
        }

        public RewriteOp(GcsPath from, GcsPath to) throws IOException {
            this.from = from;
            this.to = to;
            this.rewriteRequest = GcsUtil.this.storageClient.objects().rewrite(from.getBucket(), from.getObject(), to.getBucket(), to.getObject(), null);
            if (GcsUtil.this.maxBytesRewrittenPerCall != null) {
                this.rewriteRequest.setMaxBytesRewrittenPerCall(GcsUtil.this.maxBytesRewrittenPerCall);
            }
            this.readyToEnqueue = true;
        }

        @Override
        public void onSuccess(RewriteResponse rewriteResponse, HttpHeaders responseHeaders) throws IOException {
            if (rewriteResponse.getDone().booleanValue()) {
                LOG.debug("Rewrite done: {} to {}", (Object)this.from, (Object)this.to);
                this.readyToEnqueue = false;
            } else {
                LOG.debug("Rewrite progress: {} of {} bytes, {} to {}", rewriteResponse.getTotalBytesRewritten(), rewriteResponse.getObjectSize(), this.from, this.to);
                this.rewriteRequest.setRewriteToken(rewriteResponse.getRewriteToken());
                this.readyToEnqueue = true;
                if (GcsUtil.this.numRewriteTokensUsed != null) {
                    GcsUtil.this.numRewriteTokensUsed.incrementAndGet();
                }
            }
        }

        @Override
        public void onFailure(GoogleJsonError e, HttpHeaders responseHeaders) throws IOException {
            this.readyToEnqueue = false;
            throw new IOException(String.format("Error trying to rewrite %s to %s: %s", this.from, this.to, e));
        }
    }

    public static class GcsUtilFactory
    implements DefaultValueFactory<GcsUtil> {
        @Override
        public GcsUtil create(PipelineOptions options) {
            LOG.debug("Creating new GcsUtil");
            GcsOptions gcsOptions = options.as(GcsOptions.class);
            Storage.Builder storageBuilder = Transport.newStorageClient(gcsOptions);
            return new GcsUtil(storageBuilder.build(), storageBuilder.getHttpRequestInitializer(), gcsOptions.getExecutorService(), gcsOptions.getGcsUploadBufferSizeBytes());
        }

        public static GcsUtil create(Storage storageClient, HttpRequestInitializer httpRequestInitializer, ExecutorService executorService, @Nullable Integer uploadBufferSizeBytes) {
            return new GcsUtil(storageClient, httpRequestInitializer, executorService, uploadBufferSizeBytes);
        }
    }
}

