/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.utils.streamhist;

import com.google.common.math.IntMath;
import java.math.RoundingMode;
import java.util.Arrays;
import java.util.stream.Collectors;
import org.apache.cassandra.utils.streamhist.HistogramDataConsumer;
import org.apache.cassandra.utils.streamhist.TombstoneHistogram;

public class StreamingTombstoneHistogramBuilder {
    private final DataHolder bin;
    private final DistanceHolder distances;
    private final Spool spool;
    private final int roundSeconds;

    public StreamingTombstoneHistogramBuilder(int maxBinSize, int maxSpoolSize, int roundSeconds) {
        this.roundSeconds = roundSeconds;
        this.bin = new DataHolder(maxBinSize + 1, roundSeconds);
        this.distances = new DistanceHolder(maxBinSize);
        maxSpoolSize = maxSpoolSize == 0 ? 0 : IntMath.pow((int)2, (int)IntMath.log2((int)maxSpoolSize, (RoundingMode)RoundingMode.CEILING));
        this.spool = new Spool(maxSpoolSize);
    }

    public void update(int p) {
        this.update(p, 1);
    }

    public void update(int p, int m) {
        p = StreamingTombstoneHistogramBuilder.roundKey(p, this.roundSeconds);
        if (this.spool.capacity > 0) {
            if (!this.spool.tryAddOrAccumulate(p, m)) {
                this.flushHistogram();
                boolean success = this.spool.tryAddOrAccumulate(p, m);
                assert (success) : "Can not add value to spool";
            }
        } else {
            this.flushValue(p, m);
        }
    }

    public void flushHistogram() {
        this.spool.forEach(this::flushValue);
        this.spool.clear();
    }

    private void flushValue(int key, int spoolValue) {
        DataHolder.NeighboursAndResult addResult = this.bin.addValue(key, spoolValue);
        if (addResult.result == AddResult.INSERTED) {
            int prevPoint = addResult.prevPoint;
            int nextPoint = addResult.nextPoint;
            if (prevPoint != -1 && nextPoint != -1) {
                this.distances.remove(prevPoint, nextPoint);
            }
            if (prevPoint != -1) {
                this.distances.add(prevPoint, key);
            }
            if (nextPoint != -1) {
                this.distances.add(key, nextPoint);
            }
        }
        if (this.bin.isFull()) {
            this.mergeBin();
        }
    }

    private void mergeBin() {
        int[] smallestDifference = this.distances.getFirstAndRemove();
        int point1 = smallestDifference[0];
        int point2 = smallestDifference[1];
        DataHolder.MergeResult mergeResult = this.bin.merge(point1, point2);
        int nextPoint = mergeResult.nextPoint;
        int prevPoint = mergeResult.prevPoint;
        int newPoint = mergeResult.newPoint;
        if (nextPoint != -1) {
            this.distances.remove(point2, nextPoint);
            this.distances.add(newPoint, nextPoint);
        }
        if (prevPoint != -1) {
            this.distances.remove(prevPoint, point1);
            this.distances.add(prevPoint, newPoint);
        }
    }

    public TombstoneHistogram build() {
        this.flushHistogram();
        return new TombstoneHistogram(this.bin);
    }

    private static int roundKey(int p, int roundSeconds) {
        int d = p % roundSeconds;
        if (d > 0) {
            return p + (roundSeconds - d);
        }
        return p;
    }

    static class Spool {
        final int[] map;
        final int capacity;
        int size;

        Spool(int capacity) {
            this.capacity = capacity;
            if (capacity == 0) {
                this.map = new int[0];
            } else {
                assert (IntMath.isPowerOfTwo((int)capacity)) : "should be power of two";
                this.map = new int[capacity * 2 * 2];
                this.clear();
            }
        }

        void clear() {
            Arrays.fill(this.map, -1);
            this.size = 0;
        }

        boolean tryAddOrAccumulate(int point, int delta) {
            if (this.size > this.capacity) {
                return false;
            }
            int cell = 2 * (this.capacity - 1 & this.hash(point));
            for (int attempt = 0; attempt < 100; ++attempt) {
                if (!this.tryCell(cell + attempt * 2, point, delta)) continue;
                return true;
            }
            return false;
        }

        private int hash(int i) {
            long largePrime = 948701839L;
            return (int)((long)i * largePrime);
        }

        <E extends Exception> void forEach(HistogramDataConsumer<E> consumer) throws E {
            for (int i = 0; i < this.map.length; i += 2) {
                if (this.map[i] == -1) continue;
                consumer.consume(this.map[i], this.map[i + 1]);
            }
        }

        private boolean tryCell(int cell, int point, int delta) {
            if (this.map[cell %= this.map.length] == -1) {
                this.map[cell] = point;
                this.map[cell + 1] = delta;
                ++this.size;
                return true;
            }
            if (this.map[cell] == point) {
                int n = cell + 1;
                this.map[n] = this.map[n] + delta;
                return true;
            }
            return false;
        }
    }

    public static enum AddResult {
        INSERTED,
        ACCUMULATED;

    }

    static class DataHolder {
        private static final long EMPTY = Long.MAX_VALUE;
        private final long[] data;
        private final int roundSeconds;

        DataHolder(int maxCapacity, int roundSeconds) {
            this.data = new long[maxCapacity];
            Arrays.fill(this.data, Long.MAX_VALUE);
            this.roundSeconds = roundSeconds;
        }

        DataHolder(DataHolder holder) {
            this.data = Arrays.copyOf(holder.data, holder.data.length);
            this.roundSeconds = holder.roundSeconds;
        }

        NeighboursAndResult addValue(int point, int delta) {
            AddResult addResult;
            long key = this.wrap(point, 0);
            int index = Arrays.binarySearch(this.data, key);
            if (index < 0) {
                index = -index - 1;
                assert (index < this.data.length) : "No more space in array";
                if (this.unwrapPoint(this.data[index]) != point) {
                    assert (this.data[this.data.length - 1] == Long.MAX_VALUE) : "No more space in array";
                    System.arraycopy(this.data, index, this.data, index + 1, this.data.length - index - 1);
                    this.data[index] = this.wrap(point, delta);
                    addResult = AddResult.INSERTED;
                } else {
                    int n = index;
                    this.data[n] = this.data[n] + (long)delta;
                    addResult = AddResult.ACCUMULATED;
                }
            } else {
                int n = index;
                this.data[n] = this.data[n] + (long)delta;
                addResult = AddResult.ACCUMULATED;
            }
            return new NeighboursAndResult(this.getPrevPoint(index), this.getNextPoint(index), addResult);
        }

        public MergeResult merge(int point1, int point2) {
            long key = this.wrap(point1, 0);
            int index = Arrays.binarySearch(this.data, key);
            if (index < 0) {
                index = -index - 1;
                assert (index < this.data.length) : "Not found in array";
                assert (this.unwrapPoint(this.data[index]) == point1) : "Not found in array";
            }
            int prevPoint = this.getPrevPoint(index);
            int nextPoint = this.getNextPoint(index + 1);
            int value1 = this.unwrapValue(this.data[index]);
            int value2 = this.unwrapValue(this.data[index + 1]);
            assert (this.unwrapPoint(this.data[index + 1]) == point2) : "point2 should follow point1";
            int sum = value1 + value2;
            int newPoint = (int)(((long)point1 * (long)value1 + (long)point2 * (long)value2) / (long)(value1 + value2));
            newPoint = StreamingTombstoneHistogramBuilder.roundKey(newPoint, this.roundSeconds);
            this.data[index] = this.wrap(newPoint, sum);
            System.arraycopy(this.data, index + 2, this.data, index + 1, this.data.length - index - 2);
            this.data[this.data.length - 1] = Long.MAX_VALUE;
            return new MergeResult(prevPoint, newPoint, nextPoint);
        }

        private int getPrevPoint(int index) {
            if (index > 0) {
                if (this.data[index - 1] != Long.MAX_VALUE) {
                    return (int)(this.data[index - 1] >> 32);
                }
                return -1;
            }
            return -1;
        }

        private int getNextPoint(int index) {
            if (index < this.data.length - 1) {
                if (this.data[index + 1] != Long.MAX_VALUE) {
                    return (int)(this.data[index + 1] >> 32);
                }
                return -1;
            }
            return -1;
        }

        private int[] unwrap(long key) {
            int point = this.unwrapPoint(key);
            int value = this.unwrapValue(key);
            return new int[]{point, value};
        }

        private int unwrapPoint(long key) {
            return (int)(key >> 32);
        }

        private int unwrapValue(long key) {
            return (int)(key & 0xFFFFFFFFL);
        }

        private long wrap(int point, int value) {
            return (long)point << 32 | (long)value;
        }

        public String toString() {
            return Arrays.stream(this.data).filter(x -> x != Long.MAX_VALUE).boxed().map(this::unwrap).map(Arrays::toString).collect(Collectors.joining());
        }

        public boolean isFull() {
            return this.data[this.data.length - 1] != Long.MAX_VALUE;
        }

        public <E extends Exception> void forEach(HistogramDataConsumer<E> histogramDataConsumer) throws E {
            for (long datum : this.data) {
                if (datum == Long.MAX_VALUE) break;
                histogramDataConsumer.consume(this.unwrapPoint(datum), this.unwrapValue(datum));
            }
        }

        public int size() {
            int[] accumulator = new int[1];
            this.forEach((point, value) -> {
                accumulator[0] = accumulator[0] + 1;
            });
            return accumulator[0];
        }

        public double sum(int b) {
            long pointAndValue;
            double sum = 0.0;
            for (int i = 0; i < this.data.length && (pointAndValue = this.data[i]) != Long.MAX_VALUE; ++i) {
                int point = this.unwrapPoint(pointAndValue);
                int value = this.unwrapValue(pointAndValue);
                if (point > b) {
                    if (i == 0) {
                        return 0.0;
                    }
                    int prevPoint = this.unwrapPoint(this.data[i - 1]);
                    int prevValue = this.unwrapValue(this.data[i - 1]);
                    double weight = (double)(b - prevPoint) / (double)(point - prevPoint);
                    double mb = (double)prevValue + (double)(value - prevValue) * weight;
                    sum -= (double)prevValue;
                    sum += ((double)prevValue + mb) * weight / 2.0;
                    return sum += (double)prevValue / 2.0;
                }
                sum += (double)value;
            }
            return sum;
        }

        public int hashCode() {
            return Arrays.hashCode(this.data);
        }

        public boolean equals(Object o) {
            if (!(o instanceof DataHolder)) {
                return false;
            }
            DataHolder other = (DataHolder)o;
            if (this.size() != other.size()) {
                return false;
            }
            for (int i = 0; i < this.size(); ++i) {
                if (this.data[i] == other.data[i]) continue;
                return false;
            }
            return true;
        }

        static class NeighboursAndResult {
            int prevPoint;
            int nextPoint;
            AddResult result;

            NeighboursAndResult(int prevPoint, int nextPoint, AddResult result) {
                this.prevPoint = prevPoint;
                this.nextPoint = nextPoint;
                this.result = result;
            }
        }

        static class MergeResult {
            int prevPoint;
            int newPoint;
            int nextPoint;

            MergeResult(int prevPoint, int newPoint, int nextPoint) {
                this.prevPoint = prevPoint;
                this.newPoint = newPoint;
                this.nextPoint = nextPoint;
            }
        }
    }

    private static class DistanceHolder {
        private static final long EMPTY = Long.MAX_VALUE;
        private final long[] data;

        DistanceHolder(int maxCapacity) {
            this.data = new long[maxCapacity];
            Arrays.fill(this.data, Long.MAX_VALUE);
        }

        void add(int prev, int next) {
            long key = this.getKey(prev, next);
            int index = Arrays.binarySearch(this.data, key);
            assert (index < 0) : "Element already exists";
            assert (this.data[this.data.length - 1] == Long.MAX_VALUE) : "No more space in array";
            index = -index - 1;
            System.arraycopy(this.data, index, this.data, index + 1, this.data.length - index - 1);
            this.data[index] = key;
        }

        void remove(int prev, int next) {
            long key = this.getKey(prev, next);
            int index = Arrays.binarySearch(this.data, key);
            if (index >= 0) {
                if (index < this.data.length) {
                    System.arraycopy(this.data, index + 1, this.data, index, this.data.length - index - 1);
                }
                this.data[this.data.length - 1] = Long.MAX_VALUE;
            }
        }

        int[] getFirstAndRemove() {
            if (this.data[0] == Long.MAX_VALUE) {
                return null;
            }
            int[] result = this.unwrapKey(this.data[0]);
            System.arraycopy(this.data, 1, this.data, 0, this.data.length - 1);
            this.data[this.data.length - 1] = Long.MAX_VALUE;
            return result;
        }

        private int[] unwrapKey(long key) {
            int distance = (int)(key >> 32);
            int prev = (int)(key & 0xFFFFFFFFL);
            return new int[]{prev, prev + distance};
        }

        private long getKey(int prev, int next) {
            long distance = next - prev;
            return distance << 32 | (long)prev;
        }

        public String toString() {
            return Arrays.stream(this.data).filter(x -> x != Long.MAX_VALUE).boxed().map(this::unwrapKey).map(Arrays::toString).collect(Collectors.joining());
        }
    }
}

