/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.engine.storagegroup;

import java.io.File;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.commons.io.FileUtils;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.conf.directories.DirectoryManager;
import org.apache.iotdb.db.engine.modification.Deletion;
import org.apache.iotdb.db.engine.modification.ModificationFile;
import org.apache.iotdb.db.engine.querycontext.QueryDataSource;
import org.apache.iotdb.db.engine.querycontext.ReadOnlyMemChunk;
import org.apache.iotdb.db.engine.storagegroup.TsFileProcessor;
import org.apache.iotdb.db.engine.storagegroup.TsFileResource;
import org.apache.iotdb.db.engine.version.SimpleFileVersionController;
import org.apache.iotdb.db.engine.version.VersionController;
import org.apache.iotdb.db.exception.DiskSpaceInsufficientException;
import org.apache.iotdb.db.exception.ProcessorException;
import org.apache.iotdb.db.exception.StorageGroupProcessorException;
import org.apache.iotdb.db.exception.TsFileProcessorException;
import org.apache.iotdb.db.metadata.MManager;
import org.apache.iotdb.db.qp.physical.crud.DeletePlan;
import org.apache.iotdb.db.qp.physical.crud.InsertPlan;
import org.apache.iotdb.db.query.context.QueryContext;
import org.apache.iotdb.db.utils.CopyOnReadLinkedList;
import org.apache.iotdb.db.writelog.recover.TsFileRecoverPerformer;
import org.apache.iotdb.tsfile.file.metadata.ChunkMetaData;
import org.apache.iotdb.tsfile.file.metadata.enums.CompressionType;
import org.apache.iotdb.tsfile.file.metadata.enums.TSDataType;
import org.apache.iotdb.tsfile.file.metadata.enums.TSEncoding;
import org.apache.iotdb.tsfile.read.common.Path;
import org.apache.iotdb.tsfile.utils.Pair;
import org.apache.iotdb.tsfile.write.schema.FileSchema;
import org.apache.iotdb.tsfile.write.schema.MeasurementSchema;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StorageGroupProcessor {
    private static final Logger logger = LoggerFactory.getLogger(StorageGroupProcessor.class);
    private final ReadWriteLock insertLock = new ReentrantReadWriteLock();
    private final Object closeStorageGroupCondition = new Object();
    private final ReadWriteLock closeQueryLock = new ReentrantReadWriteLock();
    private FileSchema fileSchema;
    private List<TsFileResource> sequenceFileList = new ArrayList<TsFileResource>();
    private TsFileProcessor workSequenceTsFileProcessor = null;
    private CopyOnReadLinkedList<TsFileProcessor> closingSequenceTsFileProcessor = new CopyOnReadLinkedList();
    private List<TsFileResource> unSequenceFileList = new ArrayList<TsFileResource>();
    private TsFileProcessor workUnSequenceTsFileProcessor = null;
    private CopyOnReadLinkedList<TsFileProcessor> closingUnSequenceTsFileProcessor = new CopyOnReadLinkedList();
    private Map<String, Long> latestTimeForEachDevice = new HashMap<String, Long>();
    private Map<String, Long> latestFlushedTimeForEachDevice = new HashMap<String, Long>();
    private String storageGroupName;
    private VersionController versionController;
    private ReentrantLock mergeDeleteLock = new ReentrantLock();
    private ModificationFile mergingModification;

    public StorageGroupProcessor(String systemInfoDir, String storageGroupName) throws ProcessorException {
        this.storageGroupName = storageGroupName;
        this.fileSchema = this.constructFileSchema(storageGroupName);
        try {
            File storageGroupSysDir = new File(systemInfoDir, storageGroupName);
            if (storageGroupSysDir.mkdirs()) {
                logger.info("Storage Group system Directory {} doesn't exist, create it", (Object)storageGroupSysDir.getPath());
            } else if (!storageGroupSysDir.exists()) {
                logger.error("create Storage Group system Directory {} failed", (Object)storageGroupSysDir.getPath());
            }
            this.versionController = new SimpleFileVersionController(storageGroupSysDir.getPath());
        }
        catch (IOException e) {
            throw new StorageGroupProcessorException((Throwable)e);
        }
        this.recover();
    }

    private void recover() throws ProcessorException {
        logger.info("recover Storage Group  {}", (Object)this.storageGroupName);
        List<File> tsFiles = this.getAllFiles(DirectoryManager.getInstance().getAllSequenceFileFolders());
        this.recoverSeqFiles(tsFiles);
        tsFiles = this.getAllFiles(DirectoryManager.getInstance().getAllUnSequenceFileFolders());
        this.recoverUnseqFiles(tsFiles);
        for (TsFileResource resource : this.sequenceFileList) {
            this.latestTimeForEachDevice.putAll(resource.getEndTimeMap());
            this.latestFlushedTimeForEachDevice.putAll(resource.getEndTimeMap());
        }
    }

    private List<File> getAllFiles(List<String> folders) {
        ArrayList<File> tsFiles = new ArrayList<File>();
        for (String baseDir : folders) {
            File fileFolder = new File(baseDir, this.storageGroupName);
            if (!fileFolder.exists()) continue;
            Collections.addAll(tsFiles, fileFolder.listFiles(file -> file.getName().endsWith(".tsfile")));
        }
        return tsFiles;
    }

    private void recoverSeqFiles(List<File> tsFiles) throws ProcessorException {
        tsFiles.sort(this::compareFileName);
        for (File tsFile : tsFiles) {
            TsFileResource tsFileResource = new TsFileResource(tsFile);
            this.sequenceFileList.add(tsFileResource);
            TsFileRecoverPerformer recoverPerformer = new TsFileRecoverPerformer(this.storageGroupName + "-", this.fileSchema, this.versionController, tsFileResource, false);
            recoverPerformer.recover();
        }
    }

    private void recoverUnseqFiles(List<File> tsFiles) throws ProcessorException {
        tsFiles.sort(this::compareFileName);
        for (File tsFile : tsFiles) {
            TsFileResource tsFileResource = new TsFileResource(tsFile);
            this.unSequenceFileList.add(tsFileResource);
            TsFileRecoverPerformer recoverPerformer = new TsFileRecoverPerformer(this.storageGroupName + "-", this.fileSchema, this.versionController, tsFileResource, true);
            recoverPerformer.recover();
        }
    }

    public int compareFileName(File o1, File o2) {
        String[] items1 = o1.getName().replace(".tsfile", "").split("-");
        String[] items2 = o2.getName().replace(".tsfile", "").split("-");
        if (Long.valueOf(items1[0]) - Long.valueOf(items2[0]) == 0L) {
            return Long.compare(Long.valueOf(items1[1]), Long.valueOf(items2[1]));
        }
        return Long.compare(Long.valueOf(items1[0]), Long.valueOf(items2[0]));
    }

    private FileSchema constructFileSchema(String storageGroupName) {
        List<MeasurementSchema> columnSchemaList = MManager.getInstance().getSchemaForStorageGroup(storageGroupName);
        FileSchema schema = new FileSchema();
        for (MeasurementSchema measurementSchema : columnSchemaList) {
            schema.registerMeasurement(measurementSchema);
        }
        return schema;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addMeasurement(String measurementId, TSDataType dataType, TSEncoding encoding, CompressionType compressor, Map<String, String> props) {
        this.writeLock();
        try {
            this.fileSchema.registerMeasurement(new MeasurementSchema(measurementId, dataType, encoding, compressor, props));
        }
        finally {
            this.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean insert(InsertPlan insertPlan) {
        this.writeLock();
        try {
            this.latestTimeForEachDevice.putIfAbsent(insertPlan.getDeviceId(), Long.MIN_VALUE);
            this.latestFlushedTimeForEachDevice.putIfAbsent(insertPlan.getDeviceId(), Long.MIN_VALUE);
            boolean bl = this.insertToTsFileProcessor(insertPlan, insertPlan.getTime() > this.latestFlushedTimeForEachDevice.get(insertPlan.getDeviceId()));
            return bl;
        }
        catch (IOException e) {
            logger.error("insert tsRecord to unsealed data file failed, because {}", (Object)e.getMessage(), (Object)e);
            boolean bl = false;
            return bl;
        }
        finally {
            this.writeUnlock();
        }
    }

    private boolean insertToTsFileProcessor(InsertPlan insertPlan, boolean sequence) throws IOException {
        TsFileProcessor tsFileProcessor;
        try {
            if (sequence) {
                if (this.workSequenceTsFileProcessor == null) {
                    this.workSequenceTsFileProcessor = this.createTsFileProcessor(true);
                    this.sequenceFileList.add(this.workSequenceTsFileProcessor.getTsFileResource());
                }
                tsFileProcessor = this.workSequenceTsFileProcessor;
            } else {
                if (this.workUnSequenceTsFileProcessor == null) {
                    this.workUnSequenceTsFileProcessor = this.createTsFileProcessor(false);
                    this.unSequenceFileList.add(this.workUnSequenceTsFileProcessor.getTsFileResource());
                }
                tsFileProcessor = this.workUnSequenceTsFileProcessor;
            }
        }
        catch (DiskSpaceInsufficientException e) {
            logger.error("disk space is insufficient when creating TsFile processor, change system mode to read-only", (Throwable)e);
            IoTDBDescriptor.getInstance().getConfig().setReadOnly(true);
            return false;
        }
        boolean result = tsFileProcessor.insert(insertPlan);
        if (result && this.latestTimeForEachDevice.get(insertPlan.getDeviceId()) < insertPlan.getTime()) {
            this.latestTimeForEachDevice.put(insertPlan.getDeviceId(), insertPlan.getTime());
        }
        if (tsFileProcessor.shouldFlush()) {
            logger.info("The memtable size {} reaches the threshold, async flush it to tsfile: {}", (Object)tsFileProcessor.getWorkMemTableMemory(), (Object)tsFileProcessor.getTsFileResource().getFile().getAbsolutePath());
            if (tsFileProcessor.shouldClose()) {
                this.moveOneWorkProcessorToClosingList(sequence);
            } else {
                tsFileProcessor.asyncFlush();
            }
        }
        return result;
    }

    private TsFileProcessor createTsFileProcessor(boolean sequence) throws IOException, DiskSpaceInsufficientException {
        String baseDir = sequence ? DirectoryManager.getInstance().getNextFolderForSequenceFile() : DirectoryManager.getInstance().getNextFolderForUnSequenceFile();
        new File(baseDir, this.storageGroupName).mkdirs();
        String filePath = Paths.get(baseDir, this.storageGroupName, System.currentTimeMillis() + "-" + this.versionController.nextVersion()).toString() + ".tsfile";
        if (sequence) {
            return new TsFileProcessor(this.storageGroupName, new File(filePath), this.fileSchema, this.versionController, this::closeUnsealedTsFileProcessor, this::updateLatestFlushTimeCallback, sequence);
        }
        return new TsFileProcessor(this.storageGroupName, new File(filePath), this.fileSchema, this.versionController, this::closeUnsealedTsFileProcessor, () -> true, sequence);
    }

    private void moveOneWorkProcessorToClosingList(boolean sequence) {
        if (sequence) {
            this.closingSequenceTsFileProcessor.add(this.workSequenceTsFileProcessor);
            this.updateEndTimeMap(this.workSequenceTsFileProcessor);
            this.workSequenceTsFileProcessor.asyncClose();
            this.workSequenceTsFileProcessor = null;
        } else {
            this.closingUnSequenceTsFileProcessor.add(this.workUnSequenceTsFileProcessor);
            this.workUnSequenceTsFileProcessor.asyncClose();
            this.workUnSequenceTsFileProcessor = null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void syncDeleteDataFiles() {
        this.waitForAllCurrentTsFileProcessorsClosed();
        this.writeLock();
        try {
            for (TsFileResource tsFileResource : this.unSequenceFileList) {
                tsFileResource.close();
            }
            for (TsFileResource tsFileResource : this.sequenceFileList) {
                tsFileResource.close();
            }
            List<String> folder = DirectoryManager.getInstance().getAllSequenceFileFolders();
            folder.addAll(DirectoryManager.getInstance().getAllUnSequenceFileFolders());
            for (String tsfilePath : folder) {
                File storageGroupFolder = new File(tsfilePath, this.storageGroupName);
                if (!storageGroupFolder.exists()) continue;
                try {
                    FileUtils.deleteDirectory((File)storageGroupFolder);
                }
                catch (IOException e) {
                    logger.error("Delete tsfiles failed", (Throwable)e);
                }
            }
            this.workSequenceTsFileProcessor = null;
            this.workUnSequenceTsFileProcessor = null;
            this.sequenceFileList.clear();
            this.unSequenceFileList.clear();
            this.latestFlushedTimeForEachDevice.clear();
            this.latestTimeForEachDevice.clear();
        }
        catch (IOException e) {
            logger.error("Cannot delete files in storage group {}, because", (Object)this.storageGroupName, (Object)e);
        }
        finally {
            this.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void waitForAllCurrentTsFileProcessorsClosed() {
        Object object = this.closeStorageGroupCondition;
        synchronized (object) {
            try {
                this.putAllWorkingTsFileProcessorIntoClosingList();
                while (!this.closingSequenceTsFileProcessor.isEmpty() || !this.closingUnSequenceTsFileProcessor.isEmpty()) {
                    this.closeStorageGroupCondition.wait();
                }
            }
            catch (InterruptedException e) {
                logger.error("CloseFileNodeCondition error occurs while waiting for closing the storage group {}", (Object)this.storageGroupName, (Object)e);
                Thread.currentThread().interrupt();
            }
        }
    }

    public void putAllWorkingTsFileProcessorIntoClosingList() {
        this.writeLock();
        try {
            logger.info("async force close all files in storage group: {}", (Object)this.storageGroupName);
            if (this.workSequenceTsFileProcessor != null) {
                this.moveOneWorkProcessorToClosingList(true);
            }
            if (this.workUnSequenceTsFileProcessor != null) {
                this.moveOneWorkProcessorToClosingList(false);
            }
        }
        finally {
            this.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public QueryDataSource query(String deviceId, String measurementId, QueryContext context) {
        this.insertLock.readLock().lock();
        try {
            List<TsFileResource> seqResources = this.getFileReSourceListForQuery(this.sequenceFileList, deviceId, measurementId, context);
            List<TsFileResource> unseqResources = this.getFileReSourceListForQuery(this.unSequenceFileList, deviceId, measurementId, context);
            QueryDataSource queryDataSource = new QueryDataSource(new Path(deviceId, measurementId), seqResources, unseqResources);
            return queryDataSource;
        }
        finally {
            this.insertLock.readLock().unlock();
        }
    }

    private void writeLock() {
        this.insertLock.writeLock().lock();
    }

    private void writeUnlock() {
        this.insertLock.writeLock().unlock();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<TsFileResource> getFileReSourceListForQuery(List<TsFileResource> tsFileResources, String deviceId, String measurementId, QueryContext context) {
        MeasurementSchema mSchema = this.fileSchema.getMeasurementSchema(measurementId);
        TSDataType dataType = mSchema.getType();
        ArrayList<TsFileResource> tsfileResourcesForQuery = new ArrayList<TsFileResource>();
        for (TsFileResource tsFileResource : tsFileResources) {
            if (!tsFileResource.containsDevice(deviceId) || tsFileResource.getStartTimeMap().isEmpty()) continue;
            this.closeQueryLock.readLock().lock();
            try {
                if (tsFileResource.isClosed()) {
                    tsfileResourcesForQuery.add(tsFileResource);
                    continue;
                }
                Pair<ReadOnlyMemChunk, List<ChunkMetaData>> pair = tsFileResource.getUnsealedFileProcessor().query(deviceId, measurementId, dataType, mSchema.getProps(), context);
                tsfileResourcesForQuery.add(new TsFileResource(tsFileResource.getFile(), tsFileResource.getStartTimeMap(), tsFileResource.getEndTimeMap(), (ReadOnlyMemChunk)pair.left, (List)pair.right));
            }
            finally {
                this.closeQueryLock.readLock().unlock();
            }
        }
        return tsfileResourcesForQuery;
    }

    public void delete(String deviceId, String measurementId, long timestamp) throws IOException {
        this.writeLock();
        ArrayList<ModificationFile> updatedModFiles = new ArrayList<ModificationFile>();
        try {
            Long lastUpdateTime = this.latestTimeForEachDevice.get(deviceId);
            if (lastUpdateTime == null) {
                logger.debug("No device {} in SG {}, deletion invalid", (Object)deviceId, (Object)this.storageGroupName);
                return;
            }
            if (IoTDBDescriptor.getInstance().getConfig().isEnableWal()) {
                if (this.workSequenceTsFileProcessor != null) {
                    this.workSequenceTsFileProcessor.getLogNode().write(new DeletePlan(timestamp, new Path(deviceId, measurementId)));
                }
                if (this.workUnSequenceTsFileProcessor != null) {
                    this.workUnSequenceTsFileProcessor.getLogNode().write(new DeletePlan(timestamp, new Path(deviceId, measurementId)));
                }
            }
            Path fullPath = new Path(deviceId, measurementId);
            Deletion deletion = new Deletion(fullPath, this.versionController.nextVersion(), timestamp);
            if (this.mergingModification != null) {
                this.mergingModification.write(deletion);
                updatedModFiles.add(this.mergingModification);
            }
            this.deleteDataInFiles(this.sequenceFileList, deletion, updatedModFiles);
            this.deleteDataInFiles(this.unSequenceFileList, deletion, updatedModFiles);
        }
        catch (Exception e) {
            for (ModificationFile modFile : updatedModFiles) {
                modFile.abort();
            }
            throw new IOException(e);
        }
        finally {
            this.writeUnlock();
        }
    }

    private void deleteDataInFiles(List<TsFileResource> tsFileResourceList, Deletion deletion, List<ModificationFile> updatedModFiles) throws IOException {
        String deviceId = deletion.getDevice();
        for (TsFileResource tsFileResource : tsFileResourceList) {
            if (!tsFileResource.containsDevice(deviceId) || deletion.getTimestamp() < tsFileResource.getStartTimeMap().get(deviceId)) continue;
            tsFileResource.getModFile().write(deletion);
            if (!tsFileResource.isClosed()) {
                TsFileProcessor tsfileProcessor = tsFileResource.getUnsealedFileProcessor();
                tsfileProcessor.deleteDataInMemory(deletion);
            }
            updatedModFiles.add(tsFileResource.getModFile());
        }
    }

    private void updateEndTimeMap(TsFileProcessor tsFileProcessor) {
        TsFileResource resource = tsFileProcessor.getTsFileResource();
        for (Map.Entry<String, Long> startTime : resource.getStartTimeMap().entrySet()) {
            String deviceId = startTime.getKey();
            resource.forceUpdateEndTime(deviceId, this.latestTimeForEachDevice.get(deviceId));
        }
    }

    private boolean updateLatestFlushTimeCallback() {
        for (Map.Entry<String, Long> entry : this.latestTimeForEachDevice.entrySet()) {
            this.latestFlushedTimeForEachDevice.put(entry.getKey(), entry.getValue());
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void closeUnsealedTsFileProcessor(TsFileProcessor tsFileProcessor) throws TsFileProcessorException {
        this.closeQueryLock.writeLock().lock();
        try {
            tsFileProcessor.close();
        }
        finally {
            this.closeQueryLock.writeLock().unlock();
        }
        if (this.closingSequenceTsFileProcessor.contains(tsFileProcessor)) {
            this.closingSequenceTsFileProcessor.remove(tsFileProcessor);
        } else {
            this.closingUnSequenceTsFileProcessor.remove(tsFileProcessor);
        }
        logger.info("signal closing storage group condition in {}", (Object)this.storageGroupName);
        Object object = this.closeStorageGroupCondition;
        synchronized (object) {
            this.closeStorageGroupCondition.notifyAll();
        }
    }

    public TsFileProcessor getWorkSequenceTsFileProcessor() {
        return this.workSequenceTsFileProcessor;
    }

    @FunctionalInterface
    public static interface CloseTsFileCallBack {
        public void call(TsFileProcessor var1) throws TsFileProcessorException, IOException;
    }
}

