/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.query.control;

import java.time.ZoneId;
import java.util.Comparator;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.apache.iotdb.common.rpc.thrift.TSStatus;
import org.apache.iotdb.commons.auth.AuthException;
import org.apache.iotdb.commons.auth.entity.PrivilegeType;
import org.apache.iotdb.commons.conf.IoTDBConstant;
import org.apache.iotdb.commons.service.JMXService;
import org.apache.iotdb.db.auth.AuthorityChecker;
import org.apache.iotdb.db.auth.AuthorizerManager;
import org.apache.iotdb.db.conf.OperationType;
import org.apache.iotdb.db.mpp.common.SessionInfo;
import org.apache.iotdb.db.qp.physical.PhysicalPlan;
import org.apache.iotdb.db.qp.physical.sys.AuthorPlan;
import org.apache.iotdb.db.query.control.QueryResourceManager;
import org.apache.iotdb.db.query.control.SessionManagerMBean;
import org.apache.iotdb.db.query.control.clientsession.IClientSession;
import org.apache.iotdb.db.service.basic.BasicOpenSessionResp;
import org.apache.iotdb.db.utils.ErrorHandlingUtils;
import org.apache.iotdb.rpc.ConfigNodeConnectionException;
import org.apache.iotdb.rpc.RpcUtils;
import org.apache.iotdb.rpc.TSStatusCode;
import org.apache.iotdb.service.rpc.thrift.TSConnectionInfo;
import org.apache.iotdb.service.rpc.thrift.TSConnectionInfoResp;
import org.apache.iotdb.service.rpc.thrift.TSProtocolVersion;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SessionManager
implements SessionManagerMBean {
    private static final Logger LOGGER = LoggerFactory.getLogger(SessionManager.class);
    public static final Logger AUDIT_LOGGER = LoggerFactory.getLogger((String)"IoTDB_AUDIT_LOGGER");
    private final ThreadLocal<IClientSession> currSession = new ThreadLocal();
    private final Map<IClientSession, Object> sessions = new ConcurrentHashMap<IClientSession, Object>();
    private final Object placeHolder = new Object();
    private final AtomicLong sessionIdGenerator = new AtomicLong();
    private final AtomicLong statementIdGenerator = new AtomicLong();
    public static final TSProtocolVersion CURRENT_RPC_VERSION = TSProtocolVersion.IOTDB_SERVICE_PROTOCOL_V3;

    protected SessionManager() {
        String mbeanName = String.format("%s:%s=%s", "org.apache.iotdb.service", "type", "RpcSession");
        JMXService.registerMBean((Object)this, (String)mbeanName);
    }

    public BasicOpenSessionResp login(IClientSession session, String username, String password, String zoneId, TSProtocolVersion tsProtocolVersion, IoTDBConstant.ClientVersion clientVersion) throws TException {
        BasicOpenSessionResp openSessionResp = new BasicOpenSessionResp();
        try {
            TSStatus loginStatus = AuthorizerManager.getInstance().checkUser(username, password);
            if (loginStatus.getCode() == TSStatusCode.SUCCESS_STATUS.getStatusCode()) {
                if (!tsProtocolVersion.equals((Object)CURRENT_RPC_VERSION)) {
                    openSessionResp.sessionId(-1L).setCode(TSStatusCode.INCOMPATIBLE_VERSION.getStatusCode()).setMessage("The version is incompatible, please upgrade to " + IoTDBConstant.VERSION);
                } else {
                    this.supplySession(session, username, zoneId, clientVersion);
                    openSessionResp.sessionId(session.getId()).setCode(TSStatusCode.SUCCESS_STATUS.getStatusCode()).setMessage("Login successfully");
                    LOGGER.info("{}: Login status: {}. User : {}, opens Session-{}", new Object[]{"IoTDB", openSessionResp.getMessage(), username, session});
                }
            } else {
                AUDIT_LOGGER.info("User {} opens Session failed with an incorrect password", (Object)username);
                openSessionResp.sessionId(-1L).setMessage(loginStatus.message).setCode(loginStatus.code);
            }
        }
        catch (ConfigNodeConnectionException e) {
            LOGGER.error("Failed to connect to ConfigNode, because ", (Throwable)e);
            openSessionResp.sessionId(-1L).setCode(TSStatusCode.AUTHENTICATION_ERROR.getStatusCode()).setMessage(e.getMessage());
        }
        return openSessionResp;
    }

    public boolean closeSession(IClientSession session, Consumer<Long> releaseByQueryId) {
        this.releaseSessionResource(session, releaseByQueryId);
        IClientSession session1 = this.currSession.get();
        if (session1 != null && session != session1) {
            AUDIT_LOGGER.error("The client-{} is trying to close another session {}, pls check if it's a bug", (Object)session, (Object)session1);
            return false;
        }
        AUDIT_LOGGER.info("Session-{} is closing", (Object)session);
        return true;
    }

    private void releaseSessionResource(IClientSession session, Consumer<Long> releaseQueryResource) {
        Iterable<Long> statementIds = session.getStatementIds();
        if (statementIds != null) {
            for (Long statementId : statementIds) {
                Set<Long> queryIdSet = session.removeStatementId(statementId);
                if (queryIdSet == null) continue;
                for (Long queryId : queryIdSet) {
                    releaseQueryResource.accept(queryId);
                }
            }
        }
    }

    public TSStatus closeOperation(IClientSession session, long queryId, long statementId, boolean haveStatementId, boolean haveSetQueryId, Consumer<Long> releaseByQueryId) {
        if (!this.checkLogin(session)) {
            return RpcUtils.getStatus((TSStatusCode)TSStatusCode.NOT_LOGIN, (String)"Log in failed. Either you are not authorized or the session has timed out.");
        }
        if (AUDIT_LOGGER.isDebugEnabled()) {
            AUDIT_LOGGER.debug("{}: receive close operation from Session {}", (Object)"IoTDB", (Object)this.currSession.get());
        }
        try {
            if (haveStatementId) {
                if (haveSetQueryId) {
                    this.closeDataset(session, statementId, queryId, releaseByQueryId);
                } else {
                    this.closeStatement(session, statementId, releaseByQueryId);
                }
                return RpcUtils.getStatus((TSStatusCode)TSStatusCode.SUCCESS_STATUS);
            }
            return RpcUtils.getStatus((TSStatusCode)TSStatusCode.CLOSE_OPERATION_ERROR, (String)"statement id not set by client.");
        }
        catch (Exception e) {
            return ErrorHandlingUtils.onNPEOrUnexpectedException(e, OperationType.CLOSE_OPERATION, TSStatusCode.CLOSE_OPERATION_ERROR);
        }
    }

    public boolean checkLogin(IClientSession session) {
        boolean isLoggedIn;
        boolean bl = isLoggedIn = session != null && session.isLogin();
        if (!isLoggedIn) {
            LOGGER.info("{}: Not login. ", (Object)"IoTDB");
        }
        return isLoggedIn;
    }

    public long requestStatementId(IClientSession session) {
        long statementId = this.statementIdGenerator.incrementAndGet();
        session.addStatementId(statementId);
        return statementId;
    }

    public void closeStatement(IClientSession session, long statementId, Consumer<Long> releaseByQueryId) {
        Set<Long> queryIdSet = session.removeStatementId(statementId);
        if (queryIdSet != null) {
            for (Long queryId : queryIdSet) {
                releaseByQueryId.accept(queryId);
            }
        }
        session.removeStatementId(statementId);
    }

    public long requestQueryId(IClientSession session, Long statementId) {
        long queryId = this.requestQueryId();
        session.addQueryId(statementId, queryId);
        return queryId;
    }

    public long requestQueryId() {
        return QueryResourceManager.getInstance().assignQueryId();
    }

    public boolean checkAuthorization(PhysicalPlan plan, String username) throws AuthException {
        if (!plan.isAuthenticationRequired()) {
            return true;
        }
        String targetUser = null;
        if (plan instanceof AuthorPlan) {
            targetUser = ((AuthorPlan)plan).getUserName();
        }
        return AuthorityChecker.check(username, plan.getAuthPaths(), plan.getOperatorType(), targetUser);
    }

    public TSStatus checkAuthority(PhysicalPlan plan, IClientSession session) {
        try {
            if (!this.checkAuthorization(plan, session.getUsername())) {
                return RpcUtils.getStatus((TSStatusCode)TSStatusCode.NO_PERMISSION, (String)("No permissions for this operation, please add privilege " + PrivilegeType.values()[AuthorityChecker.translateToPermissionId(plan.getOperatorType())]));
            }
        }
        catch (AuthException e) {
            LOGGER.warn("meet error while checking authorization.", (Throwable)e);
            return RpcUtils.getStatus((TSStatusCode)TSStatusCode.UNINITIALIZED_AUTH_ERROR, (String)e.getMessage());
        }
        catch (Exception e) {
            return ErrorHandlingUtils.onQueryException(e, OperationType.CHECK_AUTHORITY.getName(), TSStatusCode.EXECUTE_STATEMENT_ERROR);
        }
        return null;
    }

    public IClientSession getCurrSession() {
        return this.currSession.get();
    }

    public TimeZone getSessionTimeZone() {
        IClientSession session = this.currSession.get();
        if (session != null) {
            return session.getTimeZone();
        }
        return TimeZone.getTimeZone("+08:00");
    }

    public void removeCurrSession() {
        IClientSession session = this.currSession.get();
        this.sessions.remove(session);
        this.currSession.remove();
    }

    public boolean registerSession(IClientSession session) {
        if (this.currSession.get() != null) {
            LOGGER.error("the client session is registered repeatedly, pls check whether this is a bug.");
            return false;
        }
        this.currSession.set(session);
        this.sessions.put(session, this.placeHolder);
        return true;
    }

    public void supplySession(IClientSession session, String username, String zoneId, IoTDBConstant.ClientVersion clientVersion) {
        session.setId(this.sessionIdGenerator.incrementAndGet());
        session.setUsername(username);
        session.setZoneId(ZoneId.of(zoneId));
        session.setClientVersion(clientVersion);
        session.setLogin(true);
        session.setLogInTime(System.currentTimeMillis());
    }

    public void closeDataset(IClientSession session, Long statementId, Long queryId, Consumer<Long> releaseByQueryId) {
        releaseByQueryId.accept(queryId);
        session.removeQueryId(statementId, queryId);
    }

    public static SessionManager getInstance() {
        return SessionManagerHelper.INSTANCE;
    }

    public SessionInfo getSessionInfo(IClientSession session) {
        return new SessionInfo(session.getId(), session.getUsername(), session.getZoneId().getId());
    }

    @Override
    public Set<String> getAllRpcClients() {
        return this.sessions.keySet().stream().map(IClientSession::toString).collect(Collectors.toSet());
    }

    public TSConnectionInfoResp getAllConnectionInfo() {
        return new TSConnectionInfoResp(this.sessions.keySet().stream().map(IClientSession::convertToTSConnectionInfo).sorted(Comparator.comparingLong(TSConnectionInfo::getLogInTime)).collect(Collectors.toList()));
    }

    private static class SessionManagerHelper {
        private static final SessionManager INSTANCE = new SessionManager();

        private SessionManagerHelper() {
        }
    }
}

