/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.task;

import com.github.fge.lambdas.functions.ThrowingFunction;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.james.task.CompletedTask;
import org.apache.james.task.FailedTask;
import org.apache.james.task.MemoryReferenceTask;
import org.apache.james.task.MemoryReferenceWithCounterTask;
import org.apache.james.task.Task;
import org.apache.james.task.TaskExecutionDetails;
import org.apache.james.task.TaskId;
import org.apache.james.task.TaskManager;
import org.apache.james.task.TaskNotFoundException;
import org.apache.james.task.ThrowingTask;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.SoftAssertions;
import org.awaitility.Awaitility;
import org.awaitility.core.ConditionFactory;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.function.ThrowingSupplier;

public interface TaskManagerContract {
    public static final Duration UPDATE_INFORMATION_POLLING_INTERVAL = Duration.ofSeconds(1L);
    public static final Duration slowPacedPollInterval = Duration.ofMillis(100L);
    public static final ConditionFactory calmlyAwait = Awaitility.with().pollInterval(slowPacedPollInterval).and().with().pollDelay(slowPacedPollInterval).await();
    public static final ConditionFactory awaitAtMostTwoSeconds = calmlyAwait.atMost(Duration.ofSeconds(2L));
    public static final Duration TIMEOUT = Duration.ofMinutes(15L);

    public TaskManager taskManager() throws Exception;

    @Test
    default public void submitShouldReturnATaskId() throws Exception {
        TaskId taskId = this.taskManager().submit((Task)new CompletedTask());
        Assertions.assertThat((Object)taskId).isNotNull();
    }

    @Test
    default public void getStatusShouldReturnUnknownWhenUnknownId() {
        TaskId unknownId = TaskId.generateTaskId();
        Assertions.assertThatThrownBy(() -> this.taskManager().getExecutionDetails(unknownId)).isInstanceOf(TaskNotFoundException.class);
    }

    @Test
    default public void getStatusShouldReturnWaitingWhenNotYetProcessed(CountDownLatch waitingForResultLatch) throws Exception {
        TaskManager taskManager = this.taskManager();
        taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            waitingForResultLatch.await();
            return Task.Result.COMPLETED;
        })));
        TaskId taskId = taskManager.submit((Task)new CompletedTask());
        Assertions.assertThat((Comparable)taskManager.getExecutionDetails(taskId).getStatus()).isEqualTo((Object)TaskManager.Status.WAITING);
    }

    @Test
    default public void taskCodeAfterCancelIsNotRun(CountDownLatch waitingForResultLatch) throws Exception {
        TaskManager taskManager = this.taskManager();
        CountDownLatch waitForTaskToBeLaunched = new CountDownLatch(1);
        AtomicInteger count = new AtomicInteger(0);
        TaskId id = taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            waitForTaskToBeLaunched.countDown();
            waitingForResultLatch.await();
            Thread.sleep(1L);
            count.incrementAndGet();
            return Task.Result.COMPLETED;
        })));
        waitForTaskToBeLaunched.await();
        taskManager.cancel(id);
        Assertions.assertThat((int)count.get()).isEqualTo(0);
    }

    @Test
    default public void completedTaskShouldNotBeCancelled() throws Exception {
        TaskManager taskManager = this.taskManager();
        TaskId id = taskManager.submit((Task)new CompletedTask());
        this.awaitUntilTaskHasStatus(id, TaskManager.Status.COMPLETED, taskManager);
        taskManager.cancel(id);
        try {
            this.awaitUntilTaskHasStatus(id, TaskManager.Status.CANCELLED, taskManager);
        }
        catch (Exception exception) {
            // empty catch block
        }
        Assertions.assertThat((Comparable)taskManager.getExecutionDetails(id).getStatus()).isEqualTo((Object)TaskManager.Status.COMPLETED);
    }

    @Test
    default public void failedTaskShouldNotBeCancelled() throws Exception {
        TaskManager taskManager = this.taskManager();
        TaskId id = taskManager.submit((Task)new FailedTask());
        this.awaitUntilTaskHasStatus(id, TaskManager.Status.FAILED, taskManager);
        taskManager.cancel(id);
        try {
            this.awaitUntilTaskHasStatus(id, TaskManager.Status.CANCELLED, taskManager);
        }
        catch (Exception exception) {
            // empty catch block
        }
        Assertions.assertThat((Comparable)taskManager.getExecutionDetails(id).getStatus()).isEqualTo((Object)TaskManager.Status.FAILED);
    }

    @Test
    default public void getStatusShouldBeCancelledWhenCancelled(CountDownLatch countDownLatch) throws Exception {
        TaskManager taskManager = this.taskManager();
        TaskId id = taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            countDownLatch.await();
            return Task.Result.COMPLETED;
        })));
        this.awaitUntilTaskHasStatus(id, TaskManager.Status.IN_PROGRESS, taskManager);
        taskManager.cancel(id);
        awaitAtMostTwoSeconds.untilAsserted(() -> Assertions.assertThat((Comparable)taskManager.getExecutionDetails(id).getStatus()).isIn(new Object[]{TaskManager.Status.CANCELLED, TaskManager.Status.CANCEL_REQUESTED}));
        countDownLatch.countDown();
        this.awaitUntilTaskHasStatus(id, TaskManager.Status.CANCELLED, taskManager);
        Assertions.assertThat((Comparable)taskManager.getExecutionDetails(id).getStatus()).isEqualTo((Object)TaskManager.Status.CANCELLED);
    }

    @Test
    default public void aWaitingTaskShouldBeCancelled(CountDownLatch countDownLatch) throws Exception {
        TaskManager taskManager = this.taskManager();
        TaskId id = taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            countDownLatch.await();
            return Task.Result.COMPLETED;
        })));
        TaskId idTaskToCancel = taskManager.submit((Task)new CompletedTask());
        taskManager.cancel(idTaskToCancel);
        this.awaitUntilTaskHasStatus(id, TaskManager.Status.IN_PROGRESS, taskManager);
        Assertions.assertThat((Comparable)taskManager.getExecutionDetails(idTaskToCancel).getStatus()).isIn(new Object[]{TaskManager.Status.CANCELLED, TaskManager.Status.CANCEL_REQUESTED});
        countDownLatch.countDown();
        this.awaitUntilTaskHasStatus(idTaskToCancel, TaskManager.Status.CANCELLED, taskManager);
        Assertions.assertThat((Comparable)taskManager.getExecutionDetails(idTaskToCancel).getStatus()).isEqualTo((Object)TaskManager.Status.CANCELLED);
    }

    @Test
    default public void cancelShouldBeIdempotent(CountDownLatch waitingForResultLatch) throws Exception {
        TaskManager taskManager = this.taskManager();
        TaskId id = taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            waitingForResultLatch.await();
            return Task.Result.COMPLETED;
        })));
        this.awaitUntilTaskHasStatus(id, TaskManager.Status.IN_PROGRESS, taskManager);
        taskManager.cancel(id);
        Assertions.assertThatCode(() -> taskManager.cancel(id)).doesNotThrowAnyException();
    }

    @Test
    default public void getStatusShouldReturnInProgressWhenProcessingIsInProgress(CountDownLatch waitingForResultLatch) throws Exception {
        TaskManager taskManager = this.taskManager();
        TaskId taskId = taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            waitingForResultLatch.await();
            return Task.Result.COMPLETED;
        })));
        this.awaitUntilTaskHasStatus(taskId, TaskManager.Status.IN_PROGRESS, taskManager);
        Assertions.assertThat((Comparable)taskManager.getExecutionDetails(taskId).getStatus()).isEqualTo((Object)TaskManager.Status.IN_PROGRESS);
    }

    @Test
    default public void getStatusShouldReturnCompletedWhenRunSuccessfully() throws Exception {
        TaskManager taskManager = this.taskManager();
        TaskId taskId = taskManager.submit((Task)new CompletedTask());
        this.awaitUntilTaskHasStatus(taskId, TaskManager.Status.COMPLETED, taskManager);
        Assertions.assertThat((Comparable)taskManager.getExecutionDetails(taskId).getStatus()).isEqualTo((Object)TaskManager.Status.COMPLETED);
    }

    @Test
    default public void additionalInformationShouldBeUpdatedWhenRunSuccessfully() throws Exception {
        TaskManager taskManager = this.taskManager();
        TaskId taskId = taskManager.submit((Task)new MemoryReferenceWithCounterTask((ThrowingFunction<AtomicLong, Task.Result>)((ThrowingFunction)counter -> {
            counter.incrementAndGet();
            return Task.Result.COMPLETED;
        })));
        this.awaitUntilTaskHasStatus(taskId, TaskManager.Status.COMPLETED, taskManager);
        MemoryReferenceWithCounterTask.AdditionalInformation additionalInformation = (MemoryReferenceWithCounterTask.AdditionalInformation)taskManager.getExecutionDetails(taskId).getAdditionalInformation().get();
        Assertions.assertThat((long)additionalInformation.getCount()).isEqualTo(1L);
    }

    @Test
    default public void additionalInformationShouldBeUpdatedWhenFailed() throws Exception {
        TaskManager taskManager = this.taskManager();
        TaskId taskId = taskManager.submit((Task)new MemoryReferenceWithCounterTask((ThrowingFunction<AtomicLong, Task.Result>)((ThrowingFunction)counter -> {
            counter.incrementAndGet();
            throw new RuntimeException();
        })));
        this.awaitUntilTaskHasStatus(taskId, TaskManager.Status.FAILED, taskManager);
        MemoryReferenceWithCounterTask.AdditionalInformation additionalInformation = (MemoryReferenceWithCounterTask.AdditionalInformation)taskManager.getExecutionDetails(taskId).getAdditionalInformation().get();
        Assertions.assertThat((long)additionalInformation.getCount()).isEqualTo(1L);
    }

    @Test
    default public void additionalInformationShouldBeUpdatedWhenCancelled(CountDownLatch countDownLatch) throws Exception {
        TaskManager taskManager = this.taskManager();
        TaskId id = taskManager.submit((Task)new MemoryReferenceWithCounterTask((ThrowingFunction<AtomicLong, Task.Result>)((ThrowingFunction)counter -> {
            counter.incrementAndGet();
            countDownLatch.await();
            return Task.Result.COMPLETED;
        })));
        this.awaitUntilTaskHasStatus(id, TaskManager.Status.IN_PROGRESS, taskManager);
        taskManager.cancel(id);
        this.awaitUntilTaskHasStatus(id, TaskManager.Status.CANCELLED, taskManager);
        Assertions.assertThat((Comparable)taskManager.getExecutionDetails(id).getStatus()).isEqualTo((Object)TaskManager.Status.CANCELLED);
        MemoryReferenceWithCounterTask.AdditionalInformation additionalInformation = (MemoryReferenceWithCounterTask.AdditionalInformation)taskManager.getExecutionDetails(id).getAdditionalInformation().get();
        Assertions.assertThat((long)additionalInformation.getCount()).isEqualTo(1L);
    }

    @Test
    default public void additionalInformationShouldBeUpdatedDuringExecution(CountDownLatch countDownLatch) throws Exception {
        TaskManager taskManager = this.taskManager();
        TaskId id = taskManager.submit((Task)new MemoryReferenceWithCounterTask((ThrowingFunction<AtomicLong, Task.Result>)((ThrowingFunction)counter -> {
            counter.incrementAndGet();
            countDownLatch.await();
            return Task.Result.COMPLETED;
        })));
        this.awaitUntilTaskHasStatus(id, TaskManager.Status.IN_PROGRESS, taskManager);
        calmlyAwait.atMost(Duration.ofSeconds(5L)).untilAsserted(() -> Assertions.assertThat((long)this.getAdditionalInformation(taskManager, id).getCount()).isEqualTo(1L));
    }

    @Test
    default public void additionalInformationShouldBeAvailableOnAnyTaskManagerDuringExecution(CountDownLatch countDownLatch) throws Exception {
        TaskManager taskManager = this.taskManager();
        TaskManager otherTaskManager = this.taskManager();
        TaskId id = taskManager.submit((Task)new MemoryReferenceWithCounterTask((ThrowingFunction<AtomicLong, Task.Result>)((ThrowingFunction)counter -> {
            counter.incrementAndGet();
            countDownLatch.await();
            return Task.Result.COMPLETED;
        })));
        this.awaitUntilTaskHasStatus(id, TaskManager.Status.IN_PROGRESS, taskManager);
        calmlyAwait.atMost(Duration.ofSeconds(5L)).untilAsserted(() -> Assertions.assertThat((long)this.getAdditionalInformation(taskManager, id).getCount()).isEqualTo(1L));
        Assertions.assertThat((long)this.getAdditionalInformation(otherTaskManager, id).getCount()).isEqualTo(1L);
    }

    default public MemoryReferenceWithCounterTask.AdditionalInformation getAdditionalInformation(TaskManager taskManager, TaskId id) {
        return (MemoryReferenceWithCounterTask.AdditionalInformation)taskManager.getExecutionDetails(id).getAdditionalInformation().get();
    }

    @Test
    default public void getStatusShouldReturnFailedWhenRunPartially() throws Exception {
        TaskManager taskManager = this.taskManager();
        TaskId taskId = taskManager.submit((Task)new FailedTask());
        this.awaitUntilTaskHasStatus(taskId, TaskManager.Status.FAILED, taskManager);
        Assertions.assertThat((Comparable)taskManager.getExecutionDetails(taskId).getStatus()).isEqualTo((Object)TaskManager.Status.FAILED);
    }

    @Test
    default public void listShouldReturnTaskStatus(CountDownLatch waitingForResultLatch) throws Exception {
        TaskManager taskManager = this.taskManager();
        SoftAssertions softly = new SoftAssertions();
        CountDownLatch latch1 = new CountDownLatch(1);
        TaskId failedId = taskManager.submit((Task)new FailedTask());
        TaskId successfulId = taskManager.submit((Task)new CompletedTask());
        TaskId inProgressId = taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            latch1.countDown();
            waitingForResultLatch.await();
            return Task.Result.COMPLETED;
        })));
        TaskId waitingId = taskManager.submit((Task)new CompletedTask());
        latch1.await();
        List list = taskManager.list();
        softly.assertThat(list).hasSize(4);
        softly.assertThat((Comparable)this.entryWithId(list, failedId)).isEqualTo((Object)TaskManager.Status.FAILED);
        softly.assertThat((Comparable)this.entryWithId(list, waitingId)).isEqualTo((Object)TaskManager.Status.WAITING);
        softly.assertThat((Comparable)this.entryWithId(list, successfulId)).isEqualTo((Object)TaskManager.Status.COMPLETED);
        softly.assertThat((Comparable)this.entryWithId(list, inProgressId)).isEqualTo((Object)TaskManager.Status.IN_PROGRESS);
    }

    default public TaskManager.Status entryWithId(List<TaskExecutionDetails> list, TaskId taskId) {
        return list.stream().filter(e -> e.getTaskId().equals((Object)taskId)).findFirst().get().getStatus();
    }

    @Test
    default public void listShouldAllowToSeeWaitingTasks(CountDownLatch waitingForResultLatch) throws Exception {
        TaskManager taskManager = this.taskManager();
        CountDownLatch latch1 = new CountDownLatch(1);
        CountDownLatch latch2 = new CountDownLatch(1);
        taskManager.submit((Task)new FailedTask());
        taskManager.submit((Task)new CompletedTask());
        taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            latch1.await();
            latch2.countDown();
            waitingForResultLatch.await();
            return Task.Result.COMPLETED;
        })));
        TaskId waitingId = taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            waitingForResultLatch.await();
            latch2.countDown();
            return Task.Result.COMPLETED;
        })));
        latch1.countDown();
        latch2.await();
        Assertions.assertThat((List)taskManager.list(TaskManager.Status.WAITING)).extracting(TaskExecutionDetails::getTaskId).containsOnly((Object[])new TaskId[]{waitingId});
    }

    @Test
    default public void listShouldAllowToSeeCompletedTasks(CountDownLatch waitingForResultLatch) throws Exception {
        TaskManager taskManager = this.taskManager();
        CountDownLatch latch1 = new CountDownLatch(1);
        CountDownLatch latch2 = new CountDownLatch(1);
        taskManager.submit((Task)new FailedTask());
        TaskId successfulId = taskManager.submit((Task)new CompletedTask());
        taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            latch1.await();
            latch2.countDown();
            waitingForResultLatch.await();
            return Task.Result.COMPLETED;
        })));
        taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            waitingForResultLatch.await();
            latch2.countDown();
            return Task.Result.COMPLETED;
        })));
        latch1.countDown();
        latch2.await();
        Assertions.assertThat((List)taskManager.list(TaskManager.Status.COMPLETED)).extracting(TaskExecutionDetails::getTaskId).containsOnly((Object[])new TaskId[]{successfulId});
    }

    @Test
    default public void listShouldAllowToSeeFailedTasks(CountDownLatch waitingForResultLatch) throws Exception {
        TaskManager taskManager = this.taskManager();
        CountDownLatch latch1 = new CountDownLatch(1);
        CountDownLatch latch2 = new CountDownLatch(1);
        TaskId failedId = taskManager.submit((Task)new FailedTask());
        taskManager.submit((Task)new CompletedTask());
        taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            latch1.await();
            latch2.countDown();
            waitingForResultLatch.await();
            return Task.Result.COMPLETED;
        })));
        taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            waitingForResultLatch.await();
            latch2.countDown();
            return Task.Result.COMPLETED;
        })));
        latch1.countDown();
        latch2.await();
        Assertions.assertThat((List)taskManager.list(TaskManager.Status.FAILED)).extracting(TaskExecutionDetails::getTaskId).containsOnly((Object[])new TaskId[]{failedId});
    }

    @Test
    default public void listShouldAllowToSeeInProgressfulTasks(CountDownLatch waitingForResultLatch) throws Exception {
        TaskManager taskManager = this.taskManager();
        CountDownLatch latch1 = new CountDownLatch(1);
        CountDownLatch latch2 = new CountDownLatch(1);
        taskManager.submit((Task)new FailedTask());
        taskManager.submit((Task)new CompletedTask());
        TaskId inProgressId = taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            latch1.await();
            latch2.countDown();
            waitingForResultLatch.await();
            return Task.Result.COMPLETED;
        })));
        taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            waitingForResultLatch.await();
            latch2.countDown();
            return Task.Result.COMPLETED;
        })));
        latch1.countDown();
        latch2.await();
        Assertions.assertThat((List)taskManager.list(TaskManager.Status.IN_PROGRESS)).extracting(TaskExecutionDetails::getTaskId).containsOnly((Object[])new TaskId[]{inProgressId});
    }

    @Test
    default public void listShouldBeEmptyWhenNoTasks() throws Exception {
        Assertions.assertThat((List)this.taskManager().list()).isEmpty();
    }

    @Test
    default public void listCancelledShouldBeEmptyWhenNoTasks() throws Exception {
        Assertions.assertThat((List)this.taskManager().list(TaskManager.Status.CANCELLED)).isEmpty();
    }

    @Test
    default public void listCancelRequestedShouldBeEmptyWhenNoTasks() throws Exception {
        Assertions.assertThat((List)this.taskManager().list(TaskManager.Status.CANCEL_REQUESTED)).isEmpty();
    }

    @Test
    default public void awaitShouldNotThrowWhenCompletedTask() throws Exception {
        TaskManager taskManager = this.taskManager();
        TaskId taskId = taskManager.submit((Task)new CompletedTask());
        taskManager.await(taskId, TIMEOUT);
        taskManager.await(taskId, TIMEOUT);
    }

    @Test
    default public void awaitShouldAwaitWaitingTask() throws Exception {
        TaskManager taskManager = this.taskManager();
        CountDownLatch latch = new CountDownLatch(1);
        taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            latch.await();
            return Task.Result.COMPLETED;
        })));
        latch.countDown();
        TaskId task2 = taskManager.submit((Task)new CompletedTask());
        Assertions.assertThat((Comparable)taskManager.await(task2, TIMEOUT).getStatus()).isEqualTo((Object)TaskManager.Status.COMPLETED);
    }

    @Test
    default public void awaitANonExistingTaskShouldReturnAnUnknownAwaitedTaskExecutionDetailsAndThrow() {
        Assertions.assertThatCode(() -> this.taskManager().await(TaskId.generateTaskId(), TIMEOUT)).isInstanceOf(TaskNotFoundException.class);
    }

    @Test
    default public void awaitWithATooShortTimeoutShouldReturnATimeoutAwaitedTaskExecutionDetailsAndThrow() throws Exception {
        TaskManager taskManager = this.taskManager();
        TaskId taskId = taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            Thread.sleep(1000L);
            return Task.Result.COMPLETED;
        })));
        Assertions.assertThatCode(() -> taskManager.await(taskId, Duration.ofMillis(10L))).isInstanceOf(TaskManager.ReachedTimeoutException.class);
    }

    @Test
    default public void submittedTaskShouldExecuteSequentially() throws Exception {
        TaskManager taskManager = this.taskManager();
        ConcurrentLinkedQueue queue = new ConcurrentLinkedQueue();
        taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            queue.add(1);
            Thread.sleep(50L);
            queue.add(2);
            return Task.Result.COMPLETED;
        })));
        taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            queue.add(3);
            Thread.sleep(50L);
            queue.add(4);
            return Task.Result.COMPLETED;
        })));
        taskManager.submit((Task)new MemoryReferenceTask((ThrowingSupplier<Task.Result>)((ThrowingSupplier)() -> {
            queue.add(5);
            Thread.sleep(50L);
            queue.add(6);
            return Task.Result.COMPLETED;
        })));
        awaitAtMostTwoSeconds.until(() -> queue.contains(6));
        Assertions.assertThat(queue).containsExactly((Object[])new Integer[]{1, 2, 3, 4, 5, 6});
    }

    @Test
    default public void awaitShouldReturnFailedWhenExceptionThrown() throws Exception {
        TaskManager taskManager = this.taskManager();
        TaskId taskId = taskManager.submit((Task)new ThrowingTask());
        this.awaitUntilTaskHasStatus(taskId, TaskManager.Status.FAILED, taskManager);
        Assertions.assertThat((Comparable)taskManager.getExecutionDetails(taskId).getStatus()).isEqualTo((Object)TaskManager.Status.FAILED);
    }

    @Test
    default public void getStatusShouldReturnFailedWhenExceptionThrown() throws Exception {
        TaskManager taskManager = this.taskManager();
        TaskId taskId = taskManager.submit((Task)new ThrowingTask());
        this.awaitUntilTaskHasStatus(taskId, TaskManager.Status.FAILED, taskManager);
        Assertions.assertThat((Comparable)taskManager.getExecutionDetails(taskId).getStatus()).isEqualTo((Object)TaskManager.Status.FAILED);
    }

    default public void awaitUntilTaskHasStatus(TaskId id, TaskManager.Status status, TaskManager taskManager) {
        awaitAtMostTwoSeconds.until(() -> taskManager.getExecutionDetails(id).getStatus(), Matchers.equalTo((Object)status));
    }
}

