/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.metadata;

import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.util.CollectionUtil;
import org.apache.lucene.util.automaton.Automaton;
import org.apache.lucene.util.automaton.Operations;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.indices.alias.Alias;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateTaskListener;
import org.elasticsearch.cluster.SimpleBatchedExecutor;
import org.elasticsearch.cluster.metadata.AliasMetadata;
import org.elasticsearch.cluster.metadata.AliasValidator;
import org.elasticsearch.cluster.metadata.ComponentTemplate;
import org.elasticsearch.cluster.metadata.ComposableIndexTemplate;
import org.elasticsearch.cluster.metadata.DataStream;
import org.elasticsearch.cluster.metadata.DataStreamFailureStoreDefinition;
import org.elasticsearch.cluster.metadata.DataStreamGlobalRetention;
import org.elasticsearch.cluster.metadata.DataStreamGlobalRetentionSettings;
import org.elasticsearch.cluster.metadata.DataStreamLifecycle;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.IndexNameExpressionResolver;
import org.elasticsearch.cluster.metadata.IndexTemplateMetadata;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.metadata.MetadataCreateDataStreamService;
import org.elasticsearch.cluster.metadata.MetadataCreateIndexService;
import org.elasticsearch.cluster.metadata.Template;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.cluster.service.MasterServiceTaskQueue;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.ValidationException;
import org.elasticsearch.common.compress.CompressedXContent;
import org.elasticsearch.common.logging.DeprecationCategory;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.logging.HeaderWarning;
import org.elasticsearch.common.regex.Regex;
import org.elasticsearch.common.settings.IndexScopedSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.index.AbstractIndexComponent;
import org.elasticsearch.index.CloseUtils;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.IndexService;
import org.elasticsearch.index.IndexSettingProvider;
import org.elasticsearch.index.IndexSettingProviders;
import org.elasticsearch.index.IndexVersion;
import org.elasticsearch.index.mapper.MapperParsingException;
import org.elasticsearch.index.mapper.MapperService;
import org.elasticsearch.index.shard.IndexLongFieldRange;
import org.elasticsearch.indices.IndexTemplateMissingException;
import org.elasticsearch.indices.IndicesService;
import org.elasticsearch.indices.InvalidIndexTemplateException;
import org.elasticsearch.indices.SystemIndices;
import org.elasticsearch.indices.cluster.IndicesClusterStateService;
import org.elasticsearch.ingest.IngestMetadata;
import org.elasticsearch.ingest.PipelineConfiguration;
import org.elasticsearch.injection.guice.Inject;
import org.elasticsearch.xcontent.NamedXContentRegistry;

public class MetadataIndexTemplateService {
    public static final String DEFAULT_TIMESTAMP_FIELD = "@timestamp";
    public static final CompressedXContent DEFAULT_TIMESTAMP_MAPPING_WITHOUT_ROUTING;
    private static final CompressedXContent DEFAULT_TIMESTAMP_MAPPING_WITH_ROUTING;
    private static final Logger logger;
    private static final DeprecationLogger deprecationLogger;
    private final ClusterService clusterService;
    private final MasterServiceTaskQueue<TemplateClusterStateUpdateTask> taskQueue;
    private final IndicesService indicesService;
    private final MetadataCreateIndexService metadataCreateIndexService;
    private final IndexScopedSettings indexScopedSettings;
    private final NamedXContentRegistry xContentRegistry;
    private final SystemIndices systemIndices;
    private final Set<IndexSettingProvider> indexSettingProviders;
    private final DataStreamGlobalRetentionSettings globalRetentionSettings;
    private static final SimpleBatchedExecutor<TemplateClusterStateUpdateTask, Void> TEMPLATE_TASK_EXECUTOR;

    @Inject
    public MetadataIndexTemplateService(ClusterService clusterService, MetadataCreateIndexService metadataCreateIndexService, IndicesService indicesService, IndexScopedSettings indexScopedSettings, NamedXContentRegistry xContentRegistry, SystemIndices systemIndices, IndexSettingProviders indexSettingProviders, DataStreamGlobalRetentionSettings globalRetentionSettings) {
        this.clusterService = clusterService;
        this.taskQueue = clusterService.createTaskQueue("index-templates", Priority.URGENT, TEMPLATE_TASK_EXECUTOR);
        this.indicesService = indicesService;
        this.metadataCreateIndexService = metadataCreateIndexService;
        this.indexScopedSettings = indexScopedSettings;
        this.xContentRegistry = xContentRegistry;
        this.systemIndices = systemIndices;
        this.indexSettingProviders = indexSettingProviders.getIndexSettingProviders();
        this.globalRetentionSettings = globalRetentionSettings;
    }

    public void removeTemplates(final String templatePattern, TimeValue timeout, ActionListener<AcknowledgedResponse> listener) {
        this.taskQueue.submitTask("remove-index-template [" + templatePattern + "]", new TemplateClusterStateUpdateTask(listener){

            @Override
            public ClusterState execute(ClusterState currentState) {
                HashSet<String> templateNames = new HashSet<String>();
                for (Map.Entry<String, IndexTemplateMetadata> cursor : currentState.metadata().templates().entrySet()) {
                    String templateName = cursor.getKey();
                    if (!Regex.simpleMatch(templatePattern, templateName)) continue;
                    templateNames.add(templateName);
                }
                if (templateNames.isEmpty()) {
                    if (Regex.isMatchAllPattern(templatePattern)) {
                        return currentState;
                    }
                    throw new IndexTemplateMissingException(templatePattern);
                }
                Metadata.Builder metadata = Metadata.builder(currentState.metadata());
                for (String templateName : templateNames) {
                    logger.info("removing template [{}]", (Object)templateName);
                    metadata.removeTemplate(templateName);
                }
                return ClusterState.builder(currentState).metadata(metadata).build();
            }
        }, timeout);
    }

    public void putComponentTemplate(String cause, final boolean create, final String name, TimeValue masterTimeout, final ComponentTemplate template, ActionListener<AcknowledgedResponse> listener) {
        this.taskQueue.submitTask("create-component-template [" + name + "], cause [" + cause + "]", new TemplateClusterStateUpdateTask(listener){

            @Override
            public ClusterState execute(ClusterState currentState) throws Exception {
                return MetadataIndexTemplateService.this.addComponentTemplate(currentState, create, name, template);
            }
        }, masterTimeout);
    }

    public ClusterState addComponentTemplate(ClusterState currentState, boolean create, String name, ComponentTemplate template) throws Exception {
        Template finalTemplate;
        ComponentTemplate finalComponentTemplate;
        ComponentTemplate existing = currentState.metadata().componentTemplates().get(name);
        if (create && existing != null) {
            throw new IllegalArgumentException("component template [" + name + "] already exists");
        }
        CompressedXContent mappings = template.template().mappings();
        CompressedXContent wrappedMappings = MetadataIndexTemplateService.wrapMappingsIfNecessary(mappings, this.xContentRegistry);
        Settings finalSettings = template.template().settings();
        if (finalSettings != null) {
            finalSettings = Settings.builder().put(finalSettings).normalizePrefix("index.").build();
        }
        Map<String, ComposableIndexTemplate> templatesUsingComponent = currentState.metadata().templatesV2().entrySet().stream().filter(e -> ((ComposableIndexTemplate)e.getValue()).composedOf().contains(name)).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        if (!create && finalSettings != null && IndexMetadata.INDEX_HIDDEN_SETTING.exists(finalSettings)) {
            ArrayList<String> globalTemplatesThatUseThisComponent = new ArrayList<String>();
            for (Map.Entry<String, ComposableIndexTemplate> entry : templatesUsingComponent.entrySet()) {
                ComposableIndexTemplate templateV2 = entry.getValue();
                if (!templateV2.indexPatterns().stream().anyMatch(Regex::isMatchAllPattern)) continue;
                globalTemplatesThatUseThisComponent.add(entry.getKey());
            }
            if (!globalTemplatesThatUseThisComponent.isEmpty()) {
                throw new IllegalArgumentException("cannot update component template [" + name + "] because the following global templates would resolve to specifying the [index.hidden] setting: [" + String.join((CharSequence)",", globalTemplatesThatUseThisComponent) + "]");
            }
        }
        if ((finalComponentTemplate = new ComponentTemplate(finalTemplate = new Template(finalSettings, wrappedMappings, template.template().aliases(), template.template().lifecycle()), template.version(), template.metadata(), template.deprecated())).equals(existing)) {
            return currentState;
        }
        MetadataIndexTemplateService.validateTemplate(finalSettings, wrappedMappings, this.indicesService);
        this.validate(name, finalComponentTemplate);
        if (templatesUsingComponent.size() > 0) {
            ClusterState tempStateWithComponentTemplateAdded = ClusterState.builder(currentState).metadata(Metadata.builder(currentState.metadata()).put(name, finalComponentTemplate)).build();
            Throwable validationFailure = null;
            for (Map.Entry<String, ComposableIndexTemplate> entry : templatesUsingComponent.entrySet()) {
                String composableTemplateName = entry.getKey();
                ComposableIndexTemplate composableTemplate = entry.getValue();
                try {
                    MetadataIndexTemplateService.validateLifecycle(tempStateWithComponentTemplateAdded.metadata(), composableTemplateName, composableTemplate, this.globalRetentionSettings.get());
                    this.validateIndexTemplateV2(composableTemplateName, composableTemplate, tempStateWithComponentTemplateAdded);
                }
                catch (Exception e2) {
                    if (validationFailure == null) {
                        validationFailure = new IllegalArgumentException("updating component template [" + name + "] results in invalid composable template [" + composableTemplateName + "] after templates are merged", e2);
                        continue;
                    }
                    validationFailure.addSuppressed(e2);
                }
            }
            if (validationFailure != null) {
                throw validationFailure;
            }
        }
        if (finalComponentTemplate.template().lifecycle() != null) {
            finalComponentTemplate.template().lifecycle().addWarningHeaderIfDataRetentionNotEffective(this.globalRetentionSettings.get(), false);
        }
        logger.info("{} component template [{}]", (Object)(existing == null ? "adding" : "updating"), (Object)name);
        return ClusterState.builder(currentState).metadata(Metadata.builder(currentState.metadata()).put(name, finalComponentTemplate)).build();
    }

    @Nullable
    private static CompressedXContent wrapMappingsIfNecessary(@Nullable CompressedXContent mappings, NamedXContentRegistry xContentRegistry) throws IOException {
        Map<String, Object> parsedMappings;
        CompressedXContent wrapped = mappings;
        if (wrapped != null && (parsedMappings = MapperService.parseMapping(xContentRegistry, mappings)).size() > 0) {
            if (parsedMappings.size() == 1) {
                String keyName = parsedMappings.keySet().iterator().next();
                if (!"_doc".equals(keyName)) {
                    wrapped = new CompressedXContent((builder, params) -> builder.field("_doc", parsedMappings));
                }
            } else {
                wrapped = new CompressedXContent((builder, params) -> builder.field("_doc", parsedMappings));
            }
        }
        return wrapped;
    }

    public void removeComponentTemplate(final String[] names, TimeValue masterTimeout, ClusterState state, ActionListener<AcknowledgedResponse> listener) {
        MetadataIndexTemplateService.validateCanBeRemoved(state.metadata(), names);
        this.taskQueue.submitTask("remove-component-template [" + String.join((CharSequence)",", names) + "]", new TemplateClusterStateUpdateTask(listener){

            @Override
            public ClusterState execute(ClusterState currentState) {
                return MetadataIndexTemplateService.innerRemoveComponentTemplate(currentState, names);
            }
        }, masterTimeout);
    }

    public static ClusterState innerRemoveComponentTemplate(ClusterState currentState, String ... names) {
        MetadataIndexTemplateService.validateCanBeRemoved(currentState.metadata(), names);
        HashSet<String> templateNames = new HashSet<String>();
        if (names.length > 1) {
            Object missingNames = null;
            for (String name : names) {
                if (currentState.metadata().componentTemplates().containsKey(name)) {
                    templateNames.add(name);
                    continue;
                }
                if (missingNames == null) {
                    missingNames = new LinkedHashSet();
                }
                missingNames.add((String)name);
            }
            if (missingNames != null) {
                throw new ResourceNotFoundException(String.join((CharSequence)",", missingNames), new Object[0]);
            }
        } else {
            for (String templateName : currentState.metadata().componentTemplates().keySet()) {
                if (!Regex.simpleMatch(names[0], templateName)) continue;
                templateNames.add(templateName);
            }
            if (templateNames.isEmpty()) {
                if (Regex.isMatchAllPattern(names[0])) {
                    return currentState;
                }
                throw new ResourceNotFoundException(names[0], new Object[0]);
            }
        }
        Metadata.Builder metadata = Metadata.builder(currentState.metadata());
        for (String templateName : templateNames) {
            logger.info("removing component template [{}]", (Object)templateName);
            metadata.removeComponentTemplate(templateName);
        }
        return ClusterState.builder(currentState).metadata(metadata).build();
    }

    static void validateCanBeRemoved(Metadata metadata, String ... templateNameOrWildcard) {
        Predicate<String> predicate = templateNameOrWildcard.length > 1 ? name -> Arrays.asList(templateNameOrWildcard).contains(name) : name -> Regex.simpleMatch(templateNameOrWildcard[0], name);
        Set matchingComponentTemplates = metadata.componentTemplates().keySet().stream().filter(predicate).collect(Collectors.toSet());
        HashSet componentsBeingUsed = new HashSet();
        List<String> templatesStillUsing = metadata.templatesV2().entrySet().stream().filter(e -> {
            Set<String> intersecting = Sets.intersection(new HashSet<String>(((ComposableIndexTemplate)e.getValue()).getRequiredComponentTemplates()), matchingComponentTemplates);
            if (intersecting.size() > 0) {
                componentsBeingUsed.addAll(intersecting);
                return true;
            }
            return false;
        }).map(Map.Entry::getKey).toList();
        if (templatesStillUsing.size() > 0) {
            throw new IllegalArgumentException("component templates " + componentsBeingUsed + " cannot be removed as they are still in use by index templates " + templatesStillUsing);
        }
    }

    public void putIndexTemplateV2(String cause, final boolean create, final String name, TimeValue masterTimeout, final ComposableIndexTemplate template, ActionListener<AcknowledgedResponse> listener) {
        MetadataIndexTemplateService.validateV2TemplateRequest(this.clusterService.state().metadata(), name, template);
        this.taskQueue.submitTask("create-index-template-v2 [" + name + "], cause [" + cause + "]", new TemplateClusterStateUpdateTask(listener){

            @Override
            public ClusterState execute(ClusterState currentState) throws Exception {
                return MetadataIndexTemplateService.this.addIndexTemplateV2(currentState, create, name, template);
            }
        }, masterTimeout);
    }

    public static void validateV2TemplateRequest(Metadata metadata, String name, ComposableIndexTemplate template) {
        Settings mergedSettings;
        if (template.indexPatterns().stream().anyMatch(Regex::isMatchAllPattern) && IndexMetadata.INDEX_HIDDEN_SETTING.exists(mergedSettings = MetadataIndexTemplateService.resolveSettings(template, metadata.componentTemplates()))) {
            throw new InvalidIndexTemplateException(name, "global composable templates may not specify the setting " + IndexMetadata.INDEX_HIDDEN_SETTING.getKey());
        }
        Map<String, ComponentTemplate> componentTemplates = metadata.componentTemplates();
        List<Object> ignoreMissingComponentTemplates = template.getIgnoreMissingComponentTemplates() == null ? List.of() : template.getIgnoreMissingComponentTemplates();
        List<String> missingComponentTemplates = template.composedOf().stream().filter(componentTemplate -> !componentTemplates.containsKey(componentTemplate)).filter(componentTemplate -> !ignoreMissingComponentTemplates.contains(componentTemplate)).toList();
        if (missingComponentTemplates.size() > 0 && ignoreMissingComponentTemplates.size() == 0) {
            throw new InvalidIndexTemplateException(name, "index template [" + name + "] specifies component templates " + missingComponentTemplates + " that do not exist");
        }
        if (missingComponentTemplates.size() > 0 && ignoreMissingComponentTemplates.size() > 0) {
            throw new InvalidIndexTemplateException(name, "index template [" + name + "] specifies a missing component templates " + missingComponentTemplates + " that does not exist and is not part of 'ignore_missing_component_templates'");
        }
    }

    public ClusterState addIndexTemplateV2(ClusterState currentState, boolean create, String name, ComposableIndexTemplate template) throws Exception {
        return this.addIndexTemplateV2(currentState, create, name, template, true);
    }

    public ClusterState addIndexTemplateV2(ClusterState currentState, boolean create, String name, ComposableIndexTemplate template, boolean validateV2Overlaps) throws Exception {
        ComposableIndexTemplate existing = currentState.metadata().templatesV2().get(name);
        if (create && existing != null) {
            throw new IllegalArgumentException("index template [" + name + "] already exists");
        }
        Map<String, List<String>> overlaps = MetadataIndexTemplateService.v2TemplateOverlaps(currentState, name, template, validateV2Overlaps);
        overlaps = MetadataIndexTemplateService.findConflictingV1Templates(currentState, name, template.indexPatterns());
        if (overlaps.size() > 0) {
            String warning = String.format(Locale.ROOT, "index template [%s] has index patterns %s matching patterns from existing older templates [%s] with patterns (%s); this template [%s] will take precedence during new index creation", name, template.indexPatterns(), Strings.collectionToCommaDelimitedString(overlaps.keySet()), overlaps.entrySet().stream().map(e -> (String)e.getKey() + " => " + e.getValue()).collect(Collectors.joining(",")), name);
            logger.warn(warning);
            HeaderWarning.addWarning(warning, new Object[0]);
        }
        ComposableIndexTemplate finalIndexTemplate = template;
        Template innerTemplate = template.template();
        if (innerTemplate != null) {
            Settings finalSettings = innerTemplate.settings();
            if (finalSettings != null) {
                finalSettings = Settings.builder().put(finalSettings).normalizePrefix("index.").build();
            }
            CompressedXContent mappings = innerTemplate.mappings();
            CompressedXContent wrappedMappings = MetadataIndexTemplateService.wrapMappingsIfNecessary(mappings, this.xContentRegistry);
            Template finalTemplate = new Template(finalSettings, wrappedMappings, innerTemplate.aliases(), innerTemplate.lifecycle());
            finalIndexTemplate = template.toBuilder().template(finalTemplate).build();
        }
        if (finalIndexTemplate.equals(existing)) {
            return currentState;
        }
        this.validateIndexTemplateV2(name, finalIndexTemplate, currentState);
        logger.info("{} index template [{}] for index patterns {}", (Object)(existing == null ? "adding" : "updating"), (Object)name, template.indexPatterns());
        return ClusterState.builder(currentState).metadata(Metadata.builder(currentState.metadata()).put(name, finalIndexTemplate)).build();
    }

    public static Map<String, List<String>> v2TemplateOverlaps(ClusterState currentState, String name, ComposableIndexTemplate template, boolean validate) {
        Map<String, List<String>> overlaps = MetadataIndexTemplateService.findConflictingV2Templates(currentState, name, template.indexPatterns(), true, template.priorityOrZero());
        overlaps.remove(name);
        if (validate && overlaps.size() > 0) {
            String error = String.format(Locale.ROOT, "index template [%s] has index patterns %s matching patterns from existing templates [%s] with patterns (%s) that have the same priority [%d], multiple index templates may not match during index creation, please use a different priority", name, template.indexPatterns(), Strings.collectionToCommaDelimitedString(overlaps.keySet()), overlaps.entrySet().stream().map(e -> (String)e.getKey() + " => " + e.getValue()).collect(Collectors.joining(",")), template.priorityOrZero());
            throw new IllegalArgumentException(error);
        }
        return overlaps;
    }

    private void validateIndexTemplateV2(String name, ComposableIndexTemplate indexTemplate, ClusterState currentState) {
        Template finalTemplate = indexTemplate.template();
        Instant now = Instant.now();
        Metadata metadata = currentState.getMetadata();
        List<CompressedXContent> combinedMappings = MetadataIndexTemplateService.collectMappings(indexTemplate, metadata.componentTemplates(), "tmp_idx");
        Settings combinedSettings = MetadataIndexTemplateService.resolveSettings(indexTemplate, metadata.componentTemplates());
        Settings.Builder finalSettings = Settings.builder();
        for (IndexSettingProvider provider : this.indexSettingProviders) {
            Settings newAdditionalSettings = provider.getAdditionalIndexSettings("validate-index-name", indexTemplate.getDataStreamTemplate() != null ? "validate-data-stream-name" : null, metadata.retrieveIndexModeFromTemplate(indexTemplate), currentState.getMetadata(), now, combinedSettings, combinedMappings);
            MetadataCreateIndexService.validateAdditionalSettings(provider, newAdditionalSettings, finalSettings);
            finalSettings.put(newAdditionalSettings);
        }
        finalSettings.put(combinedSettings);
        if (finalTemplate != null && finalTemplate.settings() != null) {
            finalSettings.put(finalTemplate.settings());
        }
        ComposableIndexTemplate templateToValidate = indexTemplate.toBuilder().template(Template.builder(finalTemplate).settings(finalSettings)).build();
        this.validate(name, templateToValidate);
        MetadataIndexTemplateService.validateDataStreamsStillReferenced(currentState, name, templateToValidate);
        MetadataIndexTemplateService.validateLifecycle(currentState.metadata(), name, templateToValidate, this.globalRetentionSettings.get());
        if (!templateToValidate.isDeprecated()) {
            this.validateUseOfDeprecatedComponentTemplates(name, templateToValidate, currentState.metadata().componentTemplates());
            this.validateUseOfDeprecatedIngestPipelines(name, (IngestMetadata)currentState.metadata().custom("ingest"), combinedSettings);
        }
        try {
            MetadataIndexTemplateService.validateCompositeTemplate(currentState, name, templateToValidate, this.indicesService, this.xContentRegistry, this.systemIndices);
        }
        catch (Exception e) {
            throw new IllegalArgumentException("composable template [" + name + "] template after composition " + (String)(indexTemplate.composedOf().size() > 0 ? "with component templates " + indexTemplate.composedOf() + " " : "") + "is invalid", e);
        }
    }

    private void validateUseOfDeprecatedComponentTemplates(String name, ComposableIndexTemplate template, Map<String, ComponentTemplate> componentTemplates) {
        template.composedOf().stream().map(ct -> Tuple.tuple((Object)ct, (Object)((ComponentTemplate)componentTemplates.get(ct)))).filter(ct -> Objects.nonNull(ct.v2())).filter(ct -> ((ComponentTemplate)ct.v2()).isDeprecated()).forEach(ct -> deprecationLogger.warn(DeprecationCategory.TEMPLATES, "use_of_deprecated_component_template", "index template [{}] uses deprecated component template [{}]", name, ct.v1()));
    }

    private void validateUseOfDeprecatedIngestPipelines(String name, IngestMetadata ingestMetadata, Settings combinedSettings) {
        Map<String, PipelineConfiguration> pipelines = Optional.ofNullable(ingestMetadata).map(IngestMetadata::getPipelines).orElse(Map.of());
        this.emitWarningIfPipelineIsDeprecated(name, pipelines, combinedSettings.get("index.default_pipeline"));
        this.emitWarningIfPipelineIsDeprecated(name, pipelines, combinedSettings.get("index.final_pipeline"));
    }

    private void emitWarningIfPipelineIsDeprecated(String name, Map<String, PipelineConfiguration> pipelines, String pipelineName) {
        Optional.ofNullable(pipelineName).map(pipelines::get).filter(p -> Boolean.TRUE.equals(p.getConfig().get("deprecated"))).ifPresent(p -> deprecationLogger.warn(DeprecationCategory.TEMPLATES, "use_of_deprecated_ingest_pipeline", "index template [{}] uses deprecated ingest pipeline [{}]", name, p.getId()));
    }

    static void validateLifecycle(Metadata metadata, String indexTemplateName, ComposableIndexTemplate template, @Nullable DataStreamGlobalRetention globalRetention) {
        DataStreamLifecycle lifecycle = MetadataIndexTemplateService.resolveLifecycle(template, metadata.componentTemplates());
        if (lifecycle != null) {
            if (template.getDataStreamTemplate() == null) {
                throw new IllegalArgumentException("index template [" + indexTemplateName + "] specifies lifecycle configuration that can only be used in combination with a data stream");
            }
            if (globalRetention != null) {
                boolean isInternalDataStream = template.indexPatterns().stream().allMatch(indexPattern -> indexPattern.charAt(0) == '.');
                lifecycle.addWarningHeaderIfDataRetentionNotEffective(globalRetention, isInternalDataStream);
            }
        }
    }

    private static void validateDataStreamsStillReferenced(ClusterState state, String templateName, ComposableIndexTemplate newTemplate) {
        Set<String> dataStreams = state.metadata().dataStreams().keySet();
        Function<Metadata, Set> findUnreferencedDataStreams = meta -> {
            HashSet<String> unreferenced = new HashSet<String>();
            for (String dataStream : dataStreams) {
                String matchingTemplate = MetadataIndexTemplateService.findV2Template(meta, dataStream, false);
                if (matchingTemplate == null) {
                    unreferenced.add(dataStream);
                    continue;
                }
                if (meta.templatesV2().get(matchingTemplate).getDataStreamTemplate() != null) continue;
                unreferenced.add(dataStream);
            }
            return unreferenced;
        };
        Set currentlyUnreferenced = findUnreferencedDataStreams.apply(state.metadata());
        Metadata updatedMetadata = Metadata.builder(state.metadata()).put(templateName, newTemplate).build();
        Set newlyUnreferenced = findUnreferencedDataStreams.apply(updatedMetadata);
        if (newlyUnreferenced.size() > currentlyUnreferenced.size()) {
            throw new IllegalArgumentException("composable template [" + templateName + "] with index patterns " + newTemplate.indexPatterns() + ", priority [" + newTemplate.priority() + "] " + (newTemplate.getDataStreamTemplate() == null ? "and no data stream configuration " : "") + "would cause data streams " + newlyUnreferenced + " to no longer match a data stream template");
        }
    }

    public static Map<String, List<String>> findConflictingV1Templates(ClusterState state, String candidateName, List<String> indexPatterns) {
        Automaton v2automaton = Regex.simpleMatchToAutomaton(indexPatterns.toArray(Strings.EMPTY_ARRAY));
        HashMap<String, List<String>> overlappingTemplates = new HashMap<String, List<String>>();
        for (Map.Entry<String, IndexTemplateMetadata> cursor : state.metadata().templates().entrySet()) {
            String name = cursor.getKey();
            IndexTemplateMetadata template = cursor.getValue();
            Automaton v1automaton = Regex.simpleMatchToAutomaton(template.patterns().toArray(Strings.EMPTY_ARRAY));
            if (Operations.isEmpty((Automaton)Operations.intersection((Automaton)v2automaton, (Automaton)v1automaton))) continue;
            logger.debug("composable template {} and legacy template {} would overlap: {} <=> {}", (Object)candidateName, (Object)name, indexPatterns, template.patterns());
            overlappingTemplates.put(name, template.patterns());
        }
        return overlappingTemplates;
    }

    public static Map<String, List<String>> findConflictingV2Templates(ClusterState state, String candidateName, List<String> indexPatterns) {
        return MetadataIndexTemplateService.findConflictingV2Templates(state, candidateName, indexPatterns, false, 0L);
    }

    static Map<String, List<String>> findConflictingV2Templates(ClusterState state, String candidateName, List<String> indexPatterns, boolean checkPriority, long priority) {
        Automaton v1automaton = Regex.simpleMatchToAutomaton(indexPatterns.toArray(Strings.EMPTY_ARRAY));
        TreeMap<String, List<String>> overlappingTemplates = new TreeMap<String, List<String>>();
        for (Map.Entry<String, ComposableIndexTemplate> entry : state.metadata().templatesV2().entrySet()) {
            String name = entry.getKey();
            ComposableIndexTemplate template = entry.getValue();
            Automaton v2automaton = Regex.simpleMatchToAutomaton(template.indexPatterns().toArray(Strings.EMPTY_ARRAY));
            if (Operations.isEmpty((Automaton)Operations.intersection((Automaton)v1automaton, (Automaton)v2automaton)) || checkPriority && priority != template.priorityOrZero()) continue;
            logger.debug("legacy template {} and composable template {} would overlap: {} <=> {}", (Object)candidateName, (Object)name, indexPatterns, template.indexPatterns());
            overlappingTemplates.put(name, template.indexPatterns());
        }
        overlappingTemplates.remove(candidateName);
        return overlappingTemplates;
    }

    public void removeIndexTemplateV2(final String[] names, TimeValue masterTimeout, ActionListener<AcknowledgedResponse> listener) {
        this.taskQueue.submitTask("remove-index-template-v2 [" + String.join((CharSequence)",", names) + "]", new TemplateClusterStateUpdateTask(listener){

            @Override
            public ClusterState execute(ClusterState currentState) {
                return MetadataIndexTemplateService.innerRemoveIndexTemplateV2(currentState, names);
            }
        }, masterTimeout);
    }

    public static ClusterState innerRemoveIndexTemplateV2(ClusterState currentState, String ... names) {
        Set<String> dataStreamsUsingTemplates;
        HashSet<String> templateNames = new HashSet<String>();
        if (names.length > 1) {
            LinkedHashSet<Object> missingNames = null;
            for (String name : names) {
                if (currentState.metadata().templatesV2().containsKey(name)) {
                    templateNames.add(name);
                    continue;
                }
                if (missingNames == null) {
                    missingNames = new LinkedHashSet<Object>();
                }
                missingNames.add(name);
            }
            if (missingNames != null) {
                throw new IndexTemplateMissingException(String.join((CharSequence)",", missingNames));
            }
        } else {
            String name = names[0];
            for (String templateName : currentState.metadata().templatesV2().keySet()) {
                if (!Regex.simpleMatch(name, templateName)) continue;
                templateNames.add(templateName);
            }
            if (templateNames.isEmpty()) {
                boolean isMatchAll = false;
                if (Regex.isMatchAllPattern(name)) {
                    isMatchAll = true;
                }
                if (isMatchAll) {
                    return currentState;
                }
                throw new IndexTemplateMissingException(name);
            }
        }
        if ((dataStreamsUsingTemplates = MetadataIndexTemplateService.dataStreamsExclusivelyUsingTemplates(currentState, templateNames)).size() > 0) {
            throw new IllegalArgumentException("unable to remove composable templates " + new TreeSet<String>(templateNames) + " as they are in use by a data streams " + new TreeSet<String>(dataStreamsUsingTemplates));
        }
        Metadata.Builder metadata = Metadata.builder(currentState.metadata());
        for (String templateName : templateNames) {
            logger.info("removing index template [{}]", (Object)templateName);
            metadata.removeIndexTemplate(templateName);
        }
        return ClusterState.builder(currentState).metadata(metadata).build();
    }

    static Set<String> dataStreamsExclusivelyUsingTemplates(ClusterState state, Set<String> templateNames) {
        Metadata metadata = state.metadata();
        Set namePatterns = templateNames.stream().map(templateName -> metadata.templatesV2().get(templateName)).filter(Objects::nonNull).map(ComposableIndexTemplate::indexPatterns).map(Set::copyOf).reduce(Sets::union).orElse(Set.of());
        return metadata.dataStreams().values().stream().filter(ds -> namePatterns.stream().anyMatch(pattern -> Regex.simpleMatch(pattern, ds.getName()))).filter(ds -> {
            List<Tuple<String, ComposableIndexTemplate>> candidates = MetadataIndexTemplateService.findV2CandidateTemplates(metadata, ds.getName(), ds.isHidden());
            if (candidates.isEmpty()) {
                throw new IllegalStateException("Data stream " + ds.getName() + " did not match any composable index templates.");
            }
            return candidates.stream().filter(template -> !templateNames.contains(template.v1()) && !MetadataIndexTemplateService.isGlobalAndHasIndexHiddenSetting(metadata, (ComposableIndexTemplate)template.v2(), (String)template.v1())).map(Tuple::v1).toList().isEmpty();
        }).map(DataStream::getName).collect(Collectors.toSet());
    }

    public void putTemplate(final PutRequest request, TimeValue timeout, ActionListener<AcknowledgedResponse> listener) {
        Settings.Builder updatedSettingsBuilder = Settings.builder();
        updatedSettingsBuilder.put(request.settings).normalizePrefix("index.");
        request.settings(updatedSettingsBuilder.build());
        if (request.name == null) {
            listener.onFailure(new IllegalArgumentException("index_template must provide a name"));
            return;
        }
        if (request.indexPatterns == null) {
            listener.onFailure(new IllegalArgumentException("index_template must provide a template"));
            return;
        }
        try {
            this.validate(request);
        }
        catch (Exception e) {
            listener.onFailure(e);
            return;
        }
        final IndexTemplateMetadata.Builder templateBuilder = IndexTemplateMetadata.builder(request.name);
        this.taskQueue.submitTask("create-index-template [" + request.name + "], cause [" + request.cause + "]", new TemplateClusterStateUpdateTask(listener){

            @Override
            public ClusterState execute(ClusterState currentState) throws Exception {
                MetadataIndexTemplateService.validateTemplate(request.settings, request.mappings, MetadataIndexTemplateService.this.indicesService);
                return MetadataIndexTemplateService.innerPutTemplate(currentState, request, templateBuilder);
            }
        }, timeout);
    }

    static ClusterState innerPutTemplate(ClusterState currentState, PutRequest request, IndexTemplateMetadata.Builder templateBuilder) {
        IndexTemplateMetadata existingTemplate;
        boolean isUpdate = currentState.metadata().templates().containsKey(request.name);
        if (request.create && isUpdate) {
            throw new IllegalArgumentException("index_template [" + request.name + "] already exists");
        }
        boolean isUpdateAndPatternsAreUnchanged = isUpdate && currentState.metadata().templates().get(request.name).patterns().equals(request.indexPatterns);
        Map<String, List<String>> overlaps = MetadataIndexTemplateService.findConflictingV2Templates(currentState, request.name, request.indexPatterns);
        if (overlaps.size() > 0) {
            if (isUpdateAndPatternsAreUnchanged || request.indexPatterns.stream().anyMatch(Regex::isMatchAllPattern)) {
                String warning = String.format(Locale.ROOT, "legacy template [%s] has index patterns %s matching patterns from existing composable templates [%s] with patterns (%s); this template [%s] may be ignored in favor of a composable template at index creation time", request.name, request.indexPatterns, Strings.collectionToCommaDelimitedString(overlaps.keySet()), overlaps.entrySet().stream().map(e -> (String)e.getKey() + " => " + e.getValue()).collect(Collectors.joining(",")), request.name);
                logger.warn(warning);
                HeaderWarning.addWarning(warning, new Object[0]);
            } else {
                String error = String.format(Locale.ROOT, "legacy template [%s] has index patterns %s matching patterns from existing composable templates [%s] with patterns (%s), use composable templates (/_index_template) instead", request.name, request.indexPatterns, Strings.collectionToCommaDelimitedString(overlaps.keySet()), overlaps.entrySet().stream().map(e -> (String)e.getKey() + " => " + e.getValue()).collect(Collectors.joining(",")));
                logger.error(error);
                throw new IllegalArgumentException(error);
            }
        }
        templateBuilder.order(request.order);
        templateBuilder.version(request.version);
        templateBuilder.patterns(request.indexPatterns);
        templateBuilder.settings(request.settings);
        if (request.mappings != null) {
            try {
                templateBuilder.putMapping("_doc", request.mappings);
            }
            catch (Exception e2) {
                throw new MapperParsingException("Failed to parse mapping: {}", (Throwable)e2, request.mappings);
            }
        }
        for (Alias alias : request.aliases) {
            AliasMetadata aliasMetadata = AliasMetadata.builder(alias.name()).filter(alias.filter()).indexRouting(alias.indexRouting()).searchRouting(alias.searchRouting()).writeIndex(alias.writeIndex()).isHidden(alias.isHidden()).build();
            templateBuilder.putAlias(aliasMetadata);
        }
        IndexTemplateMetadata template = templateBuilder.build();
        if (template.equals(existingTemplate = currentState.metadata().templates().get(request.name))) {
            return currentState;
        }
        Metadata.Builder builder = Metadata.builder(currentState.metadata()).put(template);
        logger.info("adding template [{}] for index patterns {}", (Object)request.name, request.indexPatterns);
        return ClusterState.builder(currentState).metadata(builder).build();
    }

    private static <T> boolean anyMatch(List<T> elements, Predicate<T> predicate) {
        for (T e : elements) {
            if (!predicate.test(e)) continue;
            return true;
        }
        return false;
    }

    private static <T> boolean noneMatch(List<T> elements, Predicate<T> predicate) {
        for (T e : elements) {
            if (!predicate.test(e)) continue;
            return false;
        }
        return true;
    }

    private static <T> Optional<T> findFirst(List<T> elements, Predicate<T> predicate) {
        for (T e : elements) {
            if (!predicate.test(e)) continue;
            return Optional.of(e);
        }
        return Optional.empty();
    }

    public static List<IndexTemplateMetadata> findV1Templates(Metadata metadata, String indexName, @Nullable Boolean isHidden) {
        Optional<IndexTemplateMetadata> templateWithHiddenSetting;
        String resolvedIndexName = IndexNameExpressionResolver.DateMathExpressionResolver.resolveExpression(indexName);
        Predicate<String> patternMatchPredicate = pattern -> Regex.simpleMatch(pattern, resolvedIndexName);
        ArrayList<IndexTemplateMetadata> matchedTemplates = new ArrayList<IndexTemplateMetadata>();
        for (IndexTemplateMetadata template2 : metadata.templates().values()) {
            if (isHidden == null || isHidden == Boolean.FALSE) {
                if (!MetadataIndexTemplateService.anyMatch(template2.patterns(), patternMatchPredicate)) continue;
                matchedTemplates.add(template2);
                continue;
            }
            assert (isHidden == Boolean.TRUE);
            boolean isNotMatchAllTemplate = MetadataIndexTemplateService.noneMatch(template2.patterns(), Regex::isMatchAllPattern);
            if (!isNotMatchAllTemplate || !MetadataIndexTemplateService.anyMatch(template2.patterns(), patternMatchPredicate)) continue;
            matchedTemplates.add(template2);
        }
        CollectionUtil.timSort(matchedTemplates, Comparator.comparingInt(IndexTemplateMetadata::order).reversed());
        if (isHidden == null && (templateWithHiddenSetting = MetadataIndexTemplateService.findFirst(matchedTemplates, template -> IndexMetadata.INDEX_HIDDEN_SETTING.exists(template.settings()))).isPresent()) {
            Optional<IndexTemplateMetadata> templateWithHiddenSettingPostRemoval;
            boolean templatedIsHidden = IndexMetadata.INDEX_HIDDEN_SETTING.get(templateWithHiddenSetting.get().settings());
            if (templatedIsHidden) {
                matchedTemplates.removeIf(current -> MetadataIndexTemplateService.anyMatch(current.patterns(), Regex::isMatchAllPattern));
            }
            if ((templateWithHiddenSettingPostRemoval = MetadataIndexTemplateService.findFirst(matchedTemplates, template -> IndexMetadata.INDEX_HIDDEN_SETTING.exists(template.settings()))).isEmpty() || templateWithHiddenSetting.get() != templateWithHiddenSettingPostRemoval.get()) {
                throw new IllegalStateException("A global index template [" + templateWithHiddenSetting.get().name() + "] defined the index hidden setting, which is not allowed");
            }
        }
        return Collections.unmodifiableList(matchedTemplates);
    }

    @Nullable
    public static String findV2Template(Metadata metadata, String indexName, boolean isHidden) {
        String winnerName;
        List<Tuple<String, ComposableIndexTemplate>> candidates = MetadataIndexTemplateService.findV2CandidateTemplates(metadata, indexName, isHidden);
        if (candidates.isEmpty()) {
            return null;
        }
        ComposableIndexTemplate winner = (ComposableIndexTemplate)candidates.get(0).v2();
        if (MetadataIndexTemplateService.isGlobalAndHasIndexHiddenSetting(metadata, winner, winnerName = (String)candidates.get(0).v1())) {
            throw new IllegalStateException("global index template [" + winnerName + "], composed of component templates [" + String.join((CharSequence)",", winner.composedOf()) + "] defined the index.hidden setting, which is not allowed");
        }
        return winnerName;
    }

    static List<Tuple<String, ComposableIndexTemplate>> findV2CandidateTemplates(Metadata metadata, String indexName, boolean isHidden) {
        String resolvedIndexName = IndexNameExpressionResolver.DateMathExpressionResolver.resolveExpression(indexName);
        Predicate<String> patternMatchPredicate = pattern -> Regex.simpleMatch(pattern, resolvedIndexName);
        ArrayList<Tuple<String, ComposableIndexTemplate>> candidates = new ArrayList<Tuple<String, ComposableIndexTemplate>>();
        for (Map.Entry<String, ComposableIndexTemplate> entry : metadata.templatesV2().entrySet()) {
            String name = entry.getKey();
            ComposableIndexTemplate template = entry.getValue();
            if (!isHidden || template.getDataStreamTemplate() != null) {
                if (!MetadataIndexTemplateService.anyMatch(template.indexPatterns(), patternMatchPredicate)) continue;
                candidates.add((Tuple<String, ComposableIndexTemplate>)Tuple.tuple((Object)name, (Object)template));
                continue;
            }
            boolean isNotMatchAllTemplate = MetadataIndexTemplateService.noneMatch(template.indexPatterns(), Regex::isMatchAllPattern);
            if (!isNotMatchAllTemplate || !MetadataIndexTemplateService.anyMatch(template.indexPatterns(), patternMatchPredicate)) continue;
            candidates.add((Tuple<String, ComposableIndexTemplate>)Tuple.tuple((Object)name, (Object)template));
        }
        CollectionUtil.timSort(candidates, Comparator.comparing(candidate -> ((ComposableIndexTemplate)candidate.v2()).priorityOrZero(), Comparator.reverseOrder()));
        return candidates;
    }

    private static boolean isGlobalAndHasIndexHiddenSetting(Metadata metadata, ComposableIndexTemplate template, String templateName) {
        return MetadataIndexTemplateService.anyMatch(template.indexPatterns(), Regex::isMatchAllPattern) && IndexMetadata.INDEX_HIDDEN_SETTING.exists(MetadataIndexTemplateService.resolveSettings(metadata, templateName));
    }

    public static List<CompressedXContent> collectMappings(ClusterState state, String templateName, String indexName) {
        ComposableIndexTemplate template = state.metadata().templatesV2().get(templateName);
        assert (template != null) : "attempted to resolve mappings for a template [" + templateName + "] that did not exist in the cluster state";
        if (template == null) {
            return List.of();
        }
        Map<String, ComponentTemplate> componentTemplates = state.metadata().componentTemplates();
        return MetadataIndexTemplateService.collectMappings(template, componentTemplates, indexName);
    }

    public static List<CompressedXContent> collectMappings(ComposableIndexTemplate template, Map<String, ComponentTemplate> componentTemplates, String indexName) {
        Objects.requireNonNull(template, "Composable index template must be provided");
        if (template.getDataStreamTemplate() != null && indexName.startsWith(".fs-")) {
            return List.of(DataStreamFailureStoreDefinition.DATA_STREAM_FAILURE_STORE_MAPPING, ComposableIndexTemplate.DataStreamTemplate.DATA_STREAM_MAPPING_SNIPPET);
        }
        List mappings = template.composedOf().stream().map(componentTemplates::get).filter(Objects::nonNull).map(ComponentTemplate::template).map(Template::mappings).filter(Objects::nonNull).collect(Collectors.toCollection(LinkedList::new));
        Optional.ofNullable(template.template()).map(Template::mappings).ifPresent(mappings::add);
        if (template.getDataStreamTemplate() != null && MetadataIndexTemplateService.isDataStreamIndex(indexName)) {
            if (template.getDataStreamTemplate().isAllowCustomRouting()) {
                mappings.add(0, DEFAULT_TIMESTAMP_MAPPING_WITH_ROUTING);
            } else {
                mappings.add(0, DEFAULT_TIMESTAMP_MAPPING_WITHOUT_ROUTING);
            }
        }
        if (MetadataIndexTemplateService.isDataStreamIndex(indexName) && template.getDataStreamTemplate() != null) {
            mappings.add(ComposableIndexTemplate.DataStreamTemplate.DATA_STREAM_MAPPING_SNIPPET);
        }
        return Collections.unmodifiableList(mappings);
    }

    private static boolean isDataStreamIndex(String indexName) {
        return indexName.startsWith(".ds-") || indexName.startsWith(".fs-");
    }

    public static Settings resolveSettings(List<IndexTemplateMetadata> templates) {
        Settings.Builder templateSettings = Settings.builder();
        for (int i = templates.size() - 1; i >= 0; --i) {
            templateSettings.put(templates.get(i).settings());
        }
        return templateSettings.build();
    }

    public static Settings resolveSettings(Metadata metadata, String templateName) {
        ComposableIndexTemplate template = metadata.templatesV2().get(templateName);
        assert (template != null) : "attempted to resolve settings for a template [" + templateName + "] that did not exist in the cluster state";
        if (template == null) {
            return Settings.EMPTY;
        }
        return MetadataIndexTemplateService.resolveSettings(template, metadata.componentTemplates());
    }

    public static Settings resolveSettings(ComposableIndexTemplate template, Map<String, ComponentTemplate> componentTemplates) {
        Objects.requireNonNull(template, "attempted to resolve settings for a null template");
        Objects.requireNonNull(componentTemplates, "attempted to resolve settings with null component templates");
        List<Settings> componentSettings = template.composedOf().stream().map(componentTemplates::get).filter(Objects::nonNull).map(ComponentTemplate::template).map(Template::settings).filter(Objects::nonNull).toList();
        Settings.Builder templateSettings = Settings.builder();
        componentSettings.forEach(templateSettings::put);
        Optional.ofNullable(template.template()).map(Template::settings).ifPresent(templateSettings::put);
        return templateSettings.build();
    }

    public static List<Map<String, AliasMetadata>> resolveAliases(List<IndexTemplateMetadata> templates) {
        ArrayList resolvedAliases = new ArrayList();
        templates.forEach(template -> {
            if (template.aliases() != null) {
                HashMap<String, AliasMetadata> aliasMeta = new HashMap<String, AliasMetadata>();
                for (Map.Entry<String, AliasMetadata> cursor : template.aliases().entrySet()) {
                    aliasMeta.put(cursor.getKey(), cursor.getValue());
                }
                resolvedAliases.add(aliasMeta);
            }
        });
        return Collections.unmodifiableList(resolvedAliases);
    }

    public static List<Map<String, AliasMetadata>> resolveAliases(Metadata metadata, String templateName) {
        ComposableIndexTemplate template = metadata.templatesV2().get(templateName);
        assert (template != null) : "attempted to resolve aliases for a template [" + templateName + "] that did not exist in the cluster state";
        return MetadataIndexTemplateService.resolveAliases(metadata, template);
    }

    static List<Map<String, AliasMetadata>> resolveAliases(Metadata metadata, ComposableIndexTemplate template) {
        if (template == null) {
            return List.of();
        }
        Map<String, ComponentTemplate> componentTemplates = metadata.componentTemplates();
        return MetadataIndexTemplateService.resolveAliases(template, componentTemplates);
    }

    static List<Map<String, AliasMetadata>> resolveAliases(ComposableIndexTemplate template, Map<String, ComponentTemplate> componentTemplates) {
        Objects.requireNonNull(template, "attempted to resolve aliases for a null template");
        Objects.requireNonNull(componentTemplates, "attempted to resolve aliases with null component templates");
        List aliases = template.composedOf().stream().map(componentTemplates::get).filter(Objects::nonNull).map(ComponentTemplate::template).map(Template::aliases).filter(Objects::nonNull).collect(Collectors.toCollection(ArrayList::new));
        Optional.ofNullable(template.template()).map(Template::aliases).ifPresent(aliases::add);
        Collections.reverse(aliases);
        return Collections.unmodifiableList(aliases);
    }

    @Nullable
    public static DataStreamLifecycle resolveLifecycle(Metadata metadata, String templateName) {
        ComposableIndexTemplate template = metadata.templatesV2().get(templateName);
        assert (template != null) : "attempted to resolve settings for a template [" + templateName + "] that did not exist in the cluster state";
        if (template == null) {
            return null;
        }
        return MetadataIndexTemplateService.resolveLifecycle(template, metadata.componentTemplates());
    }

    @Nullable
    public static DataStreamLifecycle resolveLifecycle(ComposableIndexTemplate template, Map<String, ComponentTemplate> componentTemplates) {
        Objects.requireNonNull(template, "attempted to resolve lifecycle for a null template");
        Objects.requireNonNull(componentTemplates, "attempted to resolve lifecycle with null component templates");
        ArrayList<DataStreamLifecycle> lifecycles = new ArrayList<DataStreamLifecycle>();
        for (String componentTemplateName : template.composedOf()) {
            DataStreamLifecycle lifecycle;
            if (!componentTemplates.containsKey(componentTemplateName) || (lifecycle = componentTemplates.get(componentTemplateName).template().lifecycle()) == null) continue;
            lifecycles.add(lifecycle);
        }
        if (template.template() != null && template.template().lifecycle() != null) {
            lifecycles.add(template.template().lifecycle());
        }
        return MetadataIndexTemplateService.composeDataLifecycles(lifecycles);
    }

    @Nullable
    public static DataStreamLifecycle composeDataLifecycles(List<DataStreamLifecycle> lifecycles) {
        DataStreamLifecycle.Builder builder = null;
        for (DataStreamLifecycle current : lifecycles) {
            if (builder == null) {
                builder = DataStreamLifecycle.newBuilder(current);
                continue;
            }
            builder.enabled(current.isEnabled());
            if (current.getDataRetention() != null) {
                builder.dataRetention(current.getDataRetention());
            }
            if (current.getDownsampling() == null) continue;
            builder.downsampling(current.getDownsampling());
        }
        return builder == null ? null : builder.build();
    }

    private static void validateCompositeTemplate(ClusterState state, String templateName, ComposableIndexTemplate template, IndicesService indicesService, NamedXContentRegistry xContentRegistry, SystemIndices systemIndices) throws Exception {
        Settings resolvedSettings;
        ClusterState stateWithTemplate = ClusterState.builder(state).metadata(Metadata.builder(state.metadata()).put(templateName, template)).build();
        String temporaryIndexName = "validate-template-" + UUIDs.randomBase64UUID().toLowerCase(Locale.ROOT);
        int dummyPartitionSize = IndexMetadata.INDEX_ROUTING_PARTITION_SIZE_SETTING.get(resolvedSettings = MetadataIndexTemplateService.resolveSettings(stateWithTemplate.metadata(), templateName));
        int dummyShards = resolvedSettings.getAsInt("index.number_of_shards", dummyPartitionSize == 1 ? 1 : dummyPartitionSize + 1);
        int shardReplicas = resolvedSettings.getAsInt("index.number_of_replicas", 0);
        Settings finalResolvedSettings = Settings.builder().put("index.version.created", IndexVersion.current()).put(resolvedSettings).put("index.number_of_shards", dummyShards).put("index.number_of_replicas", shardReplicas).put("index.uuid", UUIDs.randomBase64UUID()).build();
        ClusterState stateWithIndex = ClusterState.builder(stateWithTemplate).metadata(Metadata.builder(stateWithTemplate.metadata()).put(IndexMetadata.builder(temporaryIndexName).eventIngestedRange(IndexLongFieldRange.UNKNOWN, state.getMinTransportVersion()).settings(finalResolvedSettings)).build()).build();
        IndexMetadata tmpIndexMetadata = stateWithIndex.metadata().index(temporaryIndexName);
        indicesService.withTempIndexService(tmpIndexMetadata, tempIndexService -> {
            MetadataCreateIndexService.resolveAndValidateAliases(temporaryIndexName, Collections.emptySet(), MetadataIndexTemplateService.resolveAliases(stateWithIndex.metadata(), templateName), stateWithIndex.metadata(), xContentRegistry, tempIndexService.newSearchExecutionContext(0, 0, null, () -> 0L, null, Collections.emptyMap()), IndexService.dateMathExpressionResolverAt(System.currentTimeMillis()), systemIndices::isSystemName);
            String indexName = ".ds-" + temporaryIndexName;
            List<CompressedXContent> mappings = MetadataIndexTemplateService.collectMappings(stateWithIndex, templateName, indexName);
            try {
                MapperService mapperService = tempIndexService.mapperService();
                mapperService.merge("_doc", mappings, MapperService.MergeReason.INDEX_TEMPLATE);
                if (template.getDataStreamTemplate() != null) {
                    MetadataCreateDataStreamService.validateTimestampFieldMapping(mapperService.mappingLookup());
                }
            }
            catch (Exception e) {
                throw new IllegalArgumentException("invalid composite mappings for [" + templateName + "]", e);
            }
            return null;
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void validateTemplate(Settings validateSettings, CompressedXContent mappings, IndicesService indicesService) throws Exception {
        Settings settings = validateSettings;
        if (settings == null) {
            settings = Settings.EMPTY;
        }
        Index createdIndex = null;
        String temporaryIndexName = UUIDs.randomBase64UUID();
        try {
            int dummyPartitionSize = IndexMetadata.INDEX_ROUTING_PARTITION_SIZE_SETTING.get(settings);
            int dummyShards = settings.getAsInt("index.number_of_shards", dummyPartitionSize == 1 ? 1 : dummyPartitionSize + 1);
            int shardReplicas = settings.getAsInt("index.number_of_replicas", 0);
            Settings dummySettings = Settings.builder().put("index.version.created", IndexVersion.current()).put(settings).put("index.number_of_shards", dummyShards).put("index.number_of_replicas", shardReplicas).put("index.uuid", UUIDs.randomBase64UUID()).build();
            IndexMetadata tmpIndexMetadata = IndexMetadata.builder(temporaryIndexName).settings(dummySettings).build();
            IndicesClusterStateService.AllocatedIndex dummyIndexService = indicesService.createIndex(tmpIndexMetadata, Collections.emptyList(), false);
            createdIndex = ((AbstractIndexComponent)((Object)dummyIndexService)).index();
            if (mappings != null) {
                ((IndexService)dummyIndexService).mapperService().merge("_doc", mappings, MapperService.MergeReason.MAPPING_UPDATE);
            }
            if (createdIndex != null) {
                indicesService.removeIndex(createdIndex, IndicesClusterStateService.AllocatedIndices.IndexRemovalReason.NO_LONGER_ASSIGNED, " created for parsing template mapping", CloseUtils.NO_SHARDS_CREATED_EXECUTOR, ActionListener.noop());
            }
        }
        catch (Throwable throwable) {
            if (createdIndex != null) {
                indicesService.removeIndex(createdIndex, IndicesClusterStateService.AllocatedIndices.IndexRemovalReason.NO_LONGER_ASSIGNED, " created for parsing template mapping", CloseUtils.NO_SHARDS_CREATED_EXECUTOR, ActionListener.noop());
            }
            throw throwable;
        }
    }

    private void validate(String name, ComponentTemplate template) {
        this.validate(name, template.template(), Collections.emptyList());
    }

    private void validate(String name, ComposableIndexTemplate template) {
        this.validate(name, template.template(), template.indexPatterns());
    }

    private void validate(String name, Template template, List<String> indexPatterns) {
        Optional<Template> maybeTemplate = Optional.ofNullable(template);
        this.validate(name, maybeTemplate.map(Template::settings).orElse(Settings.EMPTY), indexPatterns, maybeTemplate.map(Template::aliases).orElse(Collections.emptyMap()).values().stream().map(MetadataIndexTemplateService::toAlias).toList());
    }

    private static Alias toAlias(AliasMetadata aliasMeta) {
        Alias a = new Alias(aliasMeta.alias());
        if (aliasMeta.filter() != null) {
            a.filter(aliasMeta.filter().string());
        }
        a.searchRouting(aliasMeta.searchRouting());
        a.indexRouting(aliasMeta.indexRouting());
        a.isHidden(aliasMeta.isHidden());
        a.writeIndex(aliasMeta.writeIndex());
        return a;
    }

    private void validate(PutRequest putRequest) {
        this.validate(putRequest.name, putRequest.settings, putRequest.indexPatterns, putRequest.aliases);
    }

    private void validate(String name, @Nullable Settings settings, List<String> indexPatterns, List<Alias> aliases) {
        ArrayList<String> validationErrors = new ArrayList<String>();
        if (name.contains(" ")) {
            validationErrors.add("name must not contain a space");
        }
        if (name.contains(",")) {
            validationErrors.add("name must not contain a ','");
        }
        if (name.contains("#")) {
            validationErrors.add("name must not contain a '#'");
        }
        if (name.contains("*")) {
            validationErrors.add("name must not contain a '*'");
        }
        if (name.startsWith("_")) {
            validationErrors.add("name must not start with '_'");
        }
        if (!name.toLowerCase(Locale.ROOT).equals(name)) {
            validationErrors.add("name must be lower cased");
        }
        for (String string : indexPatterns) {
            if (string.contains(" ")) {
                validationErrors.add("index_patterns [" + string + "] must not contain a space");
            }
            if (string.contains(",")) {
                validationErrors.add("index_pattern [" + string + "] must not contain a ','");
            }
            if (string.contains("#")) {
                validationErrors.add("index_pattern [" + string + "] must not contain a '#'");
            }
            if (string.contains(":")) {
                validationErrors.add("index_pattern [" + string + "] must not contain a ':'");
            }
            if (string.startsWith("_")) {
                validationErrors.add("index_pattern [" + string + "] must not start with '_'");
            }
            if (Strings.validFileNameExcludingAstrix(string)) continue;
            validationErrors.add("index_pattern [" + string + "] must not contain the following characters " + Strings.INVALID_FILENAME_CHARS);
        }
        if (settings != null) {
            try {
                this.indexScopedSettings.validate(settings, true);
            }
            catch (IllegalArgumentException iae) {
                validationErrors.add(iae.getMessage());
                for (Throwable t : iae.getSuppressed()) {
                    validationErrors.add(t.getMessage());
                }
            }
            List<String> indexSettingsValidation = this.metadataCreateIndexService.getIndexSettingsValidationErrors(settings, true);
            validationErrors.addAll(indexSettingsValidation);
        }
        if (indexPatterns.stream().anyMatch(Regex::isMatchAllPattern) && settings != null && IndexMetadata.INDEX_HIDDEN_SETTING.exists(settings)) {
            validationErrors.add("global templates may not specify the setting " + IndexMetadata.INDEX_HIDDEN_SETTING.getKey());
        }
        if (validationErrors.size() > 0) {
            ValidationException validationException = new ValidationException();
            validationException.addValidationErrors(validationErrors);
            throw new InvalidIndexTemplateException(name, validationException.getMessage());
        }
        for (Alias alias : aliases) {
            AliasValidator.validateAliasStandalone(alias);
            if (!indexPatterns.contains(alias.name())) continue;
            throw new IllegalArgumentException("alias [" + alias.name() + "] cannot be the same as any pattern in [" + String.join((CharSequence)", ", indexPatterns) + "]");
        }
    }

    static {
        Map<String, Map<String, String>> defaultTimestampField = Map.of(DEFAULT_TIMESTAMP_FIELD, Map.of("type", "date", "ignore_malformed", "false"));
        try {
            DEFAULT_TIMESTAMP_MAPPING_WITHOUT_ROUTING = new CompressedXContent((builder, params) -> builder.startObject("_doc").startObject("_routing").field("required", false).endObject().field("properties", defaultTimestampField).endObject());
            DEFAULT_TIMESTAMP_MAPPING_WITH_ROUTING = new CompressedXContent((builder, params) -> builder.startObject("_doc").startObject("_routing").field("required", true).endObject().field("properties").map(defaultTimestampField).endObject());
        }
        catch (IOException e) {
            throw new AssertionError((Object)e);
        }
        logger = LogManager.getLogger(MetadataIndexTemplateService.class);
        deprecationLogger = DeprecationLogger.getLogger(MetadataIndexTemplateService.class);
        TEMPLATE_TASK_EXECUTOR = new SimpleBatchedExecutor<TemplateClusterStateUpdateTask, Void>(){

            @Override
            public Tuple<ClusterState, Void> executeTask(TemplateClusterStateUpdateTask task, ClusterState clusterState) throws Exception {
                return Tuple.tuple((Object)task.execute(clusterState), null);
            }

            @Override
            public void taskSucceeded(TemplateClusterStateUpdateTask task, Void unused) {
                task.listener.onResponse(AcknowledgedResponse.TRUE);
            }
        };
    }

    public static class PutRequest {
        final String name;
        final String cause;
        boolean create;
        int order;
        Integer version;
        List<String> indexPatterns;
        Settings settings = Settings.EMPTY;
        CompressedXContent mappings = null;
        List<Alias> aliases = new ArrayList<Alias>();

        public PutRequest(String cause, String name) {
            this.cause = cause;
            this.name = name;
        }

        public PutRequest order(int order) {
            this.order = order;
            return this;
        }

        public PutRequest patterns(List<String> indexPatterns) {
            this.indexPatterns = indexPatterns;
            return this;
        }

        public PutRequest create(boolean create) {
            this.create = create;
            return this;
        }

        public PutRequest settings(Settings settings) {
            this.settings = settings;
            return this;
        }

        public PutRequest mappings(CompressedXContent mappings) {
            this.mappings = mappings;
            return this;
        }

        public PutRequest aliases(Set<Alias> aliases) {
            this.aliases.addAll(aliases);
            return this;
        }

        public PutRequest version(Integer version) {
            this.version = version;
            return this;
        }
    }

    private static abstract class TemplateClusterStateUpdateTask
    implements ClusterStateTaskListener {
        final ActionListener<AcknowledgedResponse> listener;

        TemplateClusterStateUpdateTask(ActionListener<AcknowledgedResponse> listener) {
            this.listener = listener;
        }

        public abstract ClusterState execute(ClusterState var1) throws Exception;

        @Override
        public void onFailure(Exception e) {
            this.listener.onFailure(e);
        }
    }
}

