/**
 * @license
 * Copyright (c) 2019 - 2023 Vaadin Ltd.
 * This program is available under Apache License Version 2.0, available at https://vaadin.com/license/
 */
import './vaadin-menu-bar-submenu.js';
import './vaadin-menu-bar-button.js';
import { html, PolymerElement } from '@polymer/polymer/polymer-element.js';
import { DisabledMixin } from '@vaadin/component-base/src/disabled-mixin.js';
import { ElementMixin } from '@vaadin/component-base/src/element-mixin.js';
import { TooltipController } from '@vaadin/component-base/src/tooltip-controller.js';
import { ThemableMixin } from '@vaadin/vaadin-themable-mixin/vaadin-themable-mixin.js';
import { MenuBarMixin } from './vaadin-menu-bar-mixin.js';

/**
 * `<vaadin-menu-bar>` is a Web Component providing a set of horizontally stacked buttons offering
 * the user quick access to a consistent set of commands. Each button can toggle a submenu with
 * support for additional levels of nested menus.
 *
 * To create the menu bar, first add the component to the page:
 *
 * ```
 * <vaadin-menu-bar></vaadin-menu-bar>
 * ```
 *
 * And then use [`items`](#/elements/vaadin-menu-bar#property-items) property to initialize the structure:
 *
 * ```
 * document.querySelector('vaadin-menu-bar').items = [{text: 'File'}, {text: 'Edit'}];
 * ```
 *
 * ### Styling
 *
 * The following shadow DOM parts are exposed for styling:
 *
 * Part name         | Description
 * ------------------|----------------
 * `container`       | The container wrapping menu bar buttons.
 *
 * The following state attributes are available for styling:
 *
 * Attribute           | Description
 * --------------------|----------------------------------
 * `disabled`          | Set when the menu bar is disabled
 * `has-single-button` | Set when there is only one button visible
 *
 * See [Styling Components](https://vaadin.com/docs/latest/styling/custom-theme/styling-components) documentation.
 *
 * ### Internal components
 *
 * In addition to `<vaadin-menu-bar>` itself, the following internal
 * components are themable:
 *
 * - `<vaadin-menu-bar-button>` - has the same API as [`<vaadin-button>`](#/elements/vaadin-button).
 * - `<vaadin-menu-bar-item>` - has the same API as [`<vaadin-item>`](#/elements/vaadin-item).
 * - `<vaadin-menu-bar-list-box>` - has the same API as [`<vaadin-list-box>`](#/elements/vaadin-list-box).
 * - `<vaadin-menu-bar-overlay>` - has the same API as [`<vaadin-overlay>`](#/elements/vaadin-overlay).
 *
 * @fires {CustomEvent<boolean>} item-selected - Fired when a submenu item or menu bar button without children is clicked.
 *
 * @extends HTMLElement
 * @mixes DisabledMixin
 * @mixes ElementMixin
 * @mixes MenuBarMixin
 * @mixes ThemableMixin
 */
class MenuBar extends MenuBarMixin(DisabledMixin(ElementMixin(ThemableMixin(PolymerElement)))) {
  static get template() {
    return html`
      <style>
        :host {
          display: block;
        }

        :host([hidden]) {
          display: none !important;
        }

        [part='container'] {
          position: relative;
          display: flex;
          width: 100%;
          flex-wrap: nowrap;
          overflow: hidden;
        }
      </style>

      <div part="container">
        <slot></slot>
        <slot name="overflow"></slot>
      </div>
      <vaadin-menu-bar-submenu is-root overlay-class="[[overlayClass]]"></vaadin-menu-bar-submenu>

      <slot name="tooltip"></slot>
    `;
  }

  static get is() {
    return 'vaadin-menu-bar';
  }

  static get properties() {
    return {
      /**
       * @typedef MenuBarItem
       * @type {object}
       * @property {string} text - Text to be set as the menu button component's textContent.
       * @property {string} tooltip - Text to be set as the menu button's tooltip.
       * Requires a `<vaadin-tooltip slot="tooltip">` element to be added inside the `<vaadin-menu-bar>`.
       * @property {union: string | object} component - The component to represent the button content.
       * Either a tagName or an element instance. Defaults to "vaadin-menu-bar-item".
       * @property {boolean} disabled - If true, the button is disabled and cannot be activated.
       * @property {union: string | string[]} theme - Theme(s) to be set as the theme attribute of the button, overriding any theme set on the menu bar.
       * @property {SubMenuItem[]} children - Array of submenu items.
       */

      /**
       * @typedef SubMenuItem
       * @type {object}
       * @property {string} text - Text to be set as the menu item component's textContent.
       * @property {union: string | object} component - The component to represent the item.
       * Either a tagName or an element instance. Defaults to "vaadin-menu-bar-item".
       * @property {boolean} disabled - If true, the item is disabled and cannot be selected.
       * @property {boolean} checked - If true, the item shows a checkmark next to it.
       * @property {SubMenuItem[]} children - Array of child submenu items.
       */

      /**
       * Defines a hierarchical structure, where root level items represent menu bar buttons,
       * and `children` property configures a submenu with items to be opened below
       * the button on click, Enter, Space, Up and Down arrow keys.
       *
       * #### Example
       *
       * ```js
       * menubar.items = [
       *   {
       *     text: 'File',
       *     children: [
       *       {text: 'Open'}
       *       {text: 'Auto Save', checked: true},
       *     ]
       *   },
       *   {component: 'hr'},
       *   {
       *     text: 'Edit',
       *     children: [
       *       {text: 'Undo', disabled: true},
       *       {text: 'Redo'}
       *     ]
       *   },
       *   {text: 'Help'}
       * ];
       * ```
       *
       * @type {!Array<!MenuBarItem>}
       */
      items: {
        type: Array,
        value: () => [],
      },

      /**
       * The object used to localize this component.
       * To change the default localization, replace the entire
       * `i18n` object with a custom one.
       *
       * To update individual properties, extend the existing i18n object like so:
       * ```
       * menuBar.i18n = {
       *   ...menuBar.i18n,
       *   moreOptions: 'More options'
       * }
       * ```
       *
       * The object has the following JSON structure and default values:
       * ```
       * {
       *   moreOptions: 'More options'
       * }
       * ```
       *
       * @type {!MenuBarI18n}
       * @default {English/US}
       */
      i18n: {
        type: Object,
        value: () => {
          return {
            moreOptions: 'More options',
          };
        },
      },

      /**
       * A space-delimited list of CSS class names
       * to set on each sub-menu overlay element.
       *
       * @attr {string} overlay-class
       */
      overlayClass: {
        type: String,
      },
    };
  }

  static get observers() {
    return ['_themeChanged(_theme, _overflow, _container)'];
  }

  /** @protected */
  ready() {
    super.ready();

    this._tooltipController = new TooltipController(this);
    this._tooltipController.setManual(true);
    this.addController(this._tooltipController);
  }

  /**
   * Override method inherited from `DisabledMixin`
   * to update the `disabled` property for the buttons
   * whenever the property changes on the menu bar.
   *
   * @param {boolean} newValue the new disabled value
   * @param {boolean} oldValue the previous disabled value
   * @override
   * @protected
   */
  _disabledChanged(newValue, oldValue) {
    super._disabledChanged(newValue, oldValue);
    if (oldValue !== newValue) {
      this.__updateButtonsDisabled(newValue);
    }
  }

  /**
   * A callback for the `_theme` property observer.
   * It propagates the host theme to the buttons and the sub menu.
   *
   * @param {string | null} theme
   * @protected
   */
  _themeChanged(theme, overflow, container) {
    if (overflow && container) {
      this._buttons.forEach((btn) => this._setButtonTheme(btn, theme));
      this.__detectOverflow();
    }

    if (theme) {
      this._subMenu.setAttribute('theme', theme);
    } else {
      this._subMenu.removeAttribute('theme');
    }
  }

  /** @private */
  __updateButtonsDisabled(disabled) {
    this._buttons.forEach((btn) => {
      // Disable the button if the entire menu-bar is disabled or the item alone is disabled
      btn.disabled = disabled || (btn.item && btn.item.disabled);
    });
  }

  /**
   * Fired when either a submenu item or menu bar button without nested children is clicked.
   *
   * @event item-selected
   * @param {Object} detail
   * @param {Object} detail.value the selected menu bar item
   */
}

customElements.define(MenuBar.is, MenuBar);

export { MenuBar };
