/*
 * Decompiled with CFR 0.152.
 */
package org.wso2.carbon.secvault;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.wso2.carbon.secvault.MasterKey;
import org.wso2.carbon.secvault.exception.SecureVaultException;
import org.wso2.carbon.secvault.internal.SecureVaultDataHolder;
import org.wso2.carbon.secvault.model.SecureVaultConfiguration;
import org.wso2.carbon.utils.StringUtils;
import org.yaml.snakeyaml.Yaml;
import org.yaml.snakeyaml.constructor.CustomClassLoaderConstructor;
import org.yaml.snakeyaml.introspector.BeanAccess;

public class SecureVaultUtils {
    private static final Logger logger = LoggerFactory.getLogger(SecureVaultUtils.class);
    private static final String DEFAULT_CHARSET = StandardCharsets.UTF_8.name();
    private static final Pattern VAR_PATTERN_ENV = Pattern.compile("\\$\\{env:([^}]*)}");
    private static final Pattern VAR_PATTERN_SYS = Pattern.compile("\\$\\{sys:([^}]*)}");
    private static final String YAML_EXTENSION = ".yaml";

    private SecureVaultUtils() {
        throw new AssertionError((Object)"Trying to a instantiate a constant class");
    }

    public static Optional<SecureVaultConfiguration> getSecureVaultConfig(Path secureVaultConfigPath) throws SecureVaultException {
        if (secureVaultConfigPath == null || !secureVaultConfigPath.toFile().exists()) {
            throw new SecureVaultException("Error while loading secure vault configuration. secure vault configuration file path is not provided");
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Loading Secure Vault Configurations from the file: " + secureVaultConfigPath.toString());
        }
        String resolvedFileContent = SecureVaultUtils.resolveFileToString(secureVaultConfigPath);
        Yaml yaml = new Yaml(new CustomClassLoaderConstructor(SecureVaultConfiguration.class, SecureVaultConfiguration.class.getClassLoader()));
        yaml.setBeanAccess(BeanAccess.FIELD);
        SecureVaultConfiguration secureVaultConfiguration = yaml.loadAs(resolvedFileContent, SecureVaultConfiguration.class);
        logger.debug("Secure vault configurations loaded successfully.");
        return Optional.ofNullable(secureVaultConfiguration);
    }

    public static MasterKey getSecret(List<MasterKey> masterKeys, String secretName) throws SecureVaultException {
        return masterKeys.stream().filter(masterKey -> masterKey.getMasterKeyName().equals(secretName)).findFirst().orElseThrow(() -> new SecureVaultException("No secret found with given secret name '" + secretName + "'"));
    }

    public static byte[] base64Decode(byte[] base64Encoded) {
        return Base64.getDecoder().decode(base64Encoded);
    }

    public static byte[] base64Encode(byte[] original) {
        return Base64.getEncoder().encode(original);
    }

    public static char[] toChars(byte[] bytes) {
        Charset charset = Charset.forName(DEFAULT_CHARSET);
        return charset.decode(ByteBuffer.wrap(bytes)).array();
    }

    public static byte[] toBytes(String value) {
        return value.getBytes(Charset.forName(DEFAULT_CHARSET));
    }

    public static Properties loadSecretFile(Path secretsFilePath) throws SecureVaultException {
        Properties properties = new Properties();
        try (FileInputStream inputStream = new FileInputStream(secretsFilePath.toFile());
             BufferedReader bufferedReader = new BufferedReader(new InputStreamReader((InputStream)inputStream, DEFAULT_CHARSET));){
            properties.load(bufferedReader);
        }
        catch (FileNotFoundException e) {
            throw new SecureVaultException("Cannot find secrets file in given location. (location: " + secretsFilePath + ")", e);
        }
        catch (IOException e) {
            throw new SecureVaultException("Cannot access secrets file in given location. (location: " + secretsFilePath + ")", e);
        }
        return properties;
    }

    public static void updateSecretFile(Path secretsFilePath, Properties properties) throws SecureVaultException {
        try (FileOutputStream outputStream = new FileOutputStream(secretsFilePath.toFile());
             OutputStreamWriter outputStreamWriter = new OutputStreamWriter((OutputStream)outputStream, DEFAULT_CHARSET);){
            properties.store(outputStreamWriter, null);
        }
        catch (FileNotFoundException e) {
            throw new SecureVaultException("Cannot find secrets file in given location. (location: " + secretsFilePath + ")", e);
        }
        catch (IOException e) {
            throw new SecureVaultException("Cannot access secrets file in given location. (location: " + secretsFilePath + ")", e);
        }
    }

    public static String readUpdatedValue(String alias) {
        if (alias != null) {
            if (alias.startsWith("${env:")) {
                return SecureVaultUtils.readFromEnvironment(alias.substring(6, alias.length() - 1));
            }
            if (alias.startsWith("${sys:")) {
                return SecureVaultUtils.readFromSystem(alias.substring(6, alias.length() - 1));
            }
        }
        return alias;
    }

    public static String substituteVariables(String value) throws SecureVaultException {
        if (VAR_PATTERN_ENV.matcher(value).find()) {
            value = SecureVaultUtils.substituteVariables(VAR_PATTERN_ENV.matcher(value), System::getenv);
        }
        if (VAR_PATTERN_SYS.matcher(value).find()) {
            value = SecureVaultUtils.substituteVariables(VAR_PATTERN_SYS.matcher(value), System::getProperty);
        }
        return value;
    }

    public static String substituteVariables(Matcher matcher, Function<String, String> function) throws SecureVaultException {
        StringBuffer sb = new StringBuffer();
        while (matcher.find()) {
            String sysPropKey = matcher.group(1);
            String sysPropValue = function.apply(sysPropKey);
            if (StringUtils.isNullOrEmpty(sysPropValue)) {
                String msg = "A value for placeholder '" + sysPropKey + "' is not specified";
                logger.error(msg);
                throw new SecureVaultException(msg);
            }
            sysPropValue = sysPropValue.replace("\\", "\\\\");
            matcher.appendReplacement(sb, sysPropValue);
        }
        matcher.appendTail(sb);
        return sb.toString();
    }

    public static String resolveFileToString(Path configFilePath) throws SecureVaultException {
        try {
            byte[] contentBytes = Files.readAllBytes(configFilePath);
            String stringContent = new String(contentBytes, StandardCharsets.UTF_8);
            if (configFilePath.toString().endsWith(YAML_EXTENSION)) {
                stringContent = SecureVaultUtils.getSecureVaultConfiguration(stringContent);
            }
            return SecureVaultUtils.substituteVariables(stringContent);
        }
        catch (IOException e) {
            throw new SecureVaultException("Failed to read filepath : " + configFilePath, e);
        }
    }

    private static String getSecureVaultConfiguration(String configFileContent) throws SecureVaultException {
        Yaml yaml = new Yaml();
        Map configurationMap = yaml.loadAs(configFileContent, Map.class);
        if (configurationMap == null || configurationMap.isEmpty() || configurationMap.get("wso2.securevault") == null) {
            throw new SecureVaultException("Error initializing securevault, secure configuration does not exist");
        }
        return yaml.dumpAsMap(configurationMap.get("wso2.securevault"));
    }

    public static Optional<Path> getPathFromSystemVariable(String systemProperty, String environmentProperty) {
        Optional<String> path = Optional.ofNullable(System.getProperty(systemProperty));
        if (!path.isPresent() && !(path = Optional.ofNullable(System.getenv(environmentProperty))).isPresent()) {
            return Optional.empty();
        }
        return Optional.of(Paths.get(path.get(), new String[0]));
    }

    public static boolean isOSGIEnv() {
        return SecureVaultDataHolder.getInstance().getBundleContext().isPresent();
    }

    private static String readFromEnvironment(String alias) {
        return Optional.ofNullable(alias).map(System::getenv).orElse(alias);
    }

    private static String readFromSystem(String alias) {
        return Optional.ofNullable(alias).map(System::getProperty).orElse(alias);
    }
}

