/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.beans.Transient;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class StartLabelDetectionRequest extends RekognitionRequest implements
        ToCopyableBuilder<StartLabelDetectionRequest.Builder, StartLabelDetectionRequest> {
    private static final SdkField<Video> VIDEO_FIELD = SdkField.<Video> builder(MarshallingType.SDK_POJO).memberName("Video")
            .getter(getter(StartLabelDetectionRequest::video)).setter(setter(Builder::video)).constructor(Video::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Video").build()).build();

    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ClientRequestToken").getter(getter(StartLabelDetectionRequest::clientRequestToken))
            .setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientRequestToken").build())
            .build();

    private static final SdkField<Float> MIN_CONFIDENCE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("MinConfidence").getter(getter(StartLabelDetectionRequest::minConfidence))
            .setter(setter(Builder::minConfidence))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinConfidence").build()).build();

    private static final SdkField<NotificationChannel> NOTIFICATION_CHANNEL_FIELD = SdkField
            .<NotificationChannel> builder(MarshallingType.SDK_POJO).memberName("NotificationChannel")
            .getter(getter(StartLabelDetectionRequest::notificationChannel)).setter(setter(Builder::notificationChannel))
            .constructor(NotificationChannel::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationChannel").build())
            .build();

    private static final SdkField<String> JOB_TAG_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("JobTag")
            .getter(getter(StartLabelDetectionRequest::jobTag)).setter(setter(Builder::jobTag))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobTag").build()).build();

    private static final SdkField<List<String>> FEATURES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Features")
            .getter(getter(StartLabelDetectionRequest::featuresAsStrings))
            .setter(setter(Builder::featuresWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Features").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<LabelDetectionSettings> SETTINGS_FIELD = SdkField
            .<LabelDetectionSettings> builder(MarshallingType.SDK_POJO).memberName("Settings")
            .getter(getter(StartLabelDetectionRequest::settings)).setter(setter(Builder::settings))
            .constructor(LabelDetectionSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Settings").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VIDEO_FIELD,
            CLIENT_REQUEST_TOKEN_FIELD, MIN_CONFIDENCE_FIELD, NOTIFICATION_CHANNEL_FIELD, JOB_TAG_FIELD, FEATURES_FIELD,
            SETTINGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final Video video;

    private final String clientRequestToken;

    private final Float minConfidence;

    private final NotificationChannel notificationChannel;

    private final String jobTag;

    private final List<String> features;

    private final LabelDetectionSettings settings;

    private StartLabelDetectionRequest(BuilderImpl builder) {
        super(builder);
        this.video = builder.video;
        this.clientRequestToken = builder.clientRequestToken;
        this.minConfidence = builder.minConfidence;
        this.notificationChannel = builder.notificationChannel;
        this.jobTag = builder.jobTag;
        this.features = builder.features;
        this.settings = builder.settings;
    }

    /**
     * <p>
     * The video in which you want to detect labels. The video must be stored in an Amazon S3 bucket.
     * </p>
     * 
     * @return The video in which you want to detect labels. The video must be stored in an Amazon S3 bucket.
     */
    public final Video video() {
        return video;
    }

    /**
     * <p>
     * Idempotent token used to identify the start request. If you use the same token with multiple
     * <code>StartLabelDetection</code> requests, the same <code>JobId</code> is returned. Use
     * <code>ClientRequestToken</code> to prevent the same job from being accidently started more than once.
     * </p>
     * 
     * @return Idempotent token used to identify the start request. If you use the same token with multiple
     *         <code>StartLabelDetection</code> requests, the same <code>JobId</code> is returned. Use
     *         <code>ClientRequestToken</code> to prevent the same job from being accidently started more than once.
     */
    public final String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * <p>
     * Specifies the minimum confidence that Amazon Rekognition Video must have in order to return a detected label.
     * Confidence represents how certain Amazon Rekognition is that a label is correctly identified.0 is the lowest
     * confidence. 100 is the highest confidence. Amazon Rekognition Video doesn't return any labels with a confidence
     * level lower than this specified value.
     * </p>
     * <p>
     * If you don't specify <code>MinConfidence</code>, the operation returns labels and bounding boxes (if detected)
     * with confidence values greater than or equal to 50 percent.
     * </p>
     * 
     * @return Specifies the minimum confidence that Amazon Rekognition Video must have in order to return a detected
     *         label. Confidence represents how certain Amazon Rekognition is that a label is correctly identified.0 is
     *         the lowest confidence. 100 is the highest confidence. Amazon Rekognition Video doesn't return any labels
     *         with a confidence level lower than this specified value.</p>
     *         <p>
     *         If you don't specify <code>MinConfidence</code>, the operation returns labels and bounding boxes (if
     *         detected) with confidence values greater than or equal to 50 percent.
     */
    public final Float minConfidence() {
        return minConfidence;
    }

    /**
     * <p>
     * The Amazon SNS topic ARN you want Amazon Rekognition Video to publish the completion status of the label
     * detection operation to. The Amazon SNS topic must have a topic name that begins with <i>AmazonRekognition</i> if
     * you are using the AmazonRekognitionServiceRole permissions policy.
     * </p>
     * 
     * @return The Amazon SNS topic ARN you want Amazon Rekognition Video to publish the completion status of the label
     *         detection operation to. The Amazon SNS topic must have a topic name that begins with
     *         <i>AmazonRekognition</i> if you are using the AmazonRekognitionServiceRole permissions policy.
     */
    public final NotificationChannel notificationChannel() {
        return notificationChannel;
    }

    /**
     * <p>
     * An identifier you specify that's returned in the completion notification that's published to your Amazon Simple
     * Notification Service topic. For example, you can use <code>JobTag</code> to group related jobs and identify them
     * in the completion notification.
     * </p>
     * 
     * @return An identifier you specify that's returned in the completion notification that's published to your Amazon
     *         Simple Notification Service topic. For example, you can use <code>JobTag</code> to group related jobs and
     *         identify them in the completion notification.
     */
    public final String jobTag() {
        return jobTag;
    }

    /**
     * <p>
     * The features to return after video analysis. You can specify that GENERAL_LABELS are returned.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFeatures} method.
     * </p>
     * 
     * @return The features to return after video analysis. You can specify that GENERAL_LABELS are returned.
     */
    public final List<LabelDetectionFeatureName> features() {
        return LabelDetectionFeatureListCopier.copyStringToEnum(features);
    }

    /**
     * For responses, this returns true if the service returned a value for the Features property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasFeatures() {
        return features != null && !(features instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The features to return after video analysis. You can specify that GENERAL_LABELS are returned.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFeatures} method.
     * </p>
     * 
     * @return The features to return after video analysis. You can specify that GENERAL_LABELS are returned.
     */
    public final List<String> featuresAsStrings() {
        return features;
    }

    /**
     * <p>
     * The settings for a StartLabelDetection request.Contains the specified parameters for the label detection request
     * of an asynchronous label analysis operation. Settings can include filters for GENERAL_LABELS.
     * </p>
     * 
     * @return The settings for a StartLabelDetection request.Contains the specified parameters for the label detection
     *         request of an asynchronous label analysis operation. Settings can include filters for GENERAL_LABELS.
     */
    public final LabelDetectionSettings settings() {
        return settings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(video());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(minConfidence());
        hashCode = 31 * hashCode + Objects.hashCode(notificationChannel());
        hashCode = 31 * hashCode + Objects.hashCode(jobTag());
        hashCode = 31 * hashCode + Objects.hashCode(hasFeatures() ? featuresAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(settings());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StartLabelDetectionRequest)) {
            return false;
        }
        StartLabelDetectionRequest other = (StartLabelDetectionRequest) obj;
        return Objects.equals(video(), other.video()) && Objects.equals(clientRequestToken(), other.clientRequestToken())
                && Objects.equals(minConfidence(), other.minConfidence())
                && Objects.equals(notificationChannel(), other.notificationChannel()) && Objects.equals(jobTag(), other.jobTag())
                && hasFeatures() == other.hasFeatures() && Objects.equals(featuresAsStrings(), other.featuresAsStrings())
                && Objects.equals(settings(), other.settings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StartLabelDetectionRequest").add("Video", video())
                .add("ClientRequestToken", clientRequestToken()).add("MinConfidence", minConfidence())
                .add("NotificationChannel", notificationChannel()).add("JobTag", jobTag())
                .add("Features", hasFeatures() ? featuresAsStrings() : null).add("Settings", settings()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Video":
            return Optional.ofNullable(clazz.cast(video()));
        case "ClientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        case "MinConfidence":
            return Optional.ofNullable(clazz.cast(minConfidence()));
        case "NotificationChannel":
            return Optional.ofNullable(clazz.cast(notificationChannel()));
        case "JobTag":
            return Optional.ofNullable(clazz.cast(jobTag()));
        case "Features":
            return Optional.ofNullable(clazz.cast(featuresAsStrings()));
        case "Settings":
            return Optional.ofNullable(clazz.cast(settings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Video", VIDEO_FIELD);
        map.put("ClientRequestToken", CLIENT_REQUEST_TOKEN_FIELD);
        map.put("MinConfidence", MIN_CONFIDENCE_FIELD);
        map.put("NotificationChannel", NOTIFICATION_CHANNEL_FIELD);
        map.put("JobTag", JOB_TAG_FIELD);
        map.put("Features", FEATURES_FIELD);
        map.put("Settings", SETTINGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<StartLabelDetectionRequest, T> g) {
        return obj -> g.apply((StartLabelDetectionRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends RekognitionRequest.Builder, SdkPojo, CopyableBuilder<Builder, StartLabelDetectionRequest> {
        /**
         * <p>
         * The video in which you want to detect labels. The video must be stored in an Amazon S3 bucket.
         * </p>
         * 
         * @param video
         *        The video in which you want to detect labels. The video must be stored in an Amazon S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder video(Video video);

        /**
         * <p>
         * The video in which you want to detect labels. The video must be stored in an Amazon S3 bucket.
         * </p>
         * This is a convenience method that creates an instance of the {@link Video.Builder} avoiding the need to
         * create one manually via {@link Video#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Video.Builder#build()} is called immediately and its result is
         * passed to {@link #video(Video)}.
         * 
         * @param video
         *        a consumer that will call methods on {@link Video.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #video(Video)
         */
        default Builder video(Consumer<Video.Builder> video) {
            return video(Video.builder().applyMutation(video).build());
        }

        /**
         * <p>
         * Idempotent token used to identify the start request. If you use the same token with multiple
         * <code>StartLabelDetection</code> requests, the same <code>JobId</code> is returned. Use
         * <code>ClientRequestToken</code> to prevent the same job from being accidently started more than once.
         * </p>
         * 
         * @param clientRequestToken
         *        Idempotent token used to identify the start request. If you use the same token with multiple
         *        <code>StartLabelDetection</code> requests, the same <code>JobId</code> is returned. Use
         *        <code>ClientRequestToken</code> to prevent the same job from being accidently started more than once.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * <p>
         * Specifies the minimum confidence that Amazon Rekognition Video must have in order to return a detected label.
         * Confidence represents how certain Amazon Rekognition is that a label is correctly identified.0 is the lowest
         * confidence. 100 is the highest confidence. Amazon Rekognition Video doesn't return any labels with a
         * confidence level lower than this specified value.
         * </p>
         * <p>
         * If you don't specify <code>MinConfidence</code>, the operation returns labels and bounding boxes (if
         * detected) with confidence values greater than or equal to 50 percent.
         * </p>
         * 
         * @param minConfidence
         *        Specifies the minimum confidence that Amazon Rekognition Video must have in order to return a detected
         *        label. Confidence represents how certain Amazon Rekognition is that a label is correctly identified.0
         *        is the lowest confidence. 100 is the highest confidence. Amazon Rekognition Video doesn't return any
         *        labels with a confidence level lower than this specified value.</p>
         *        <p>
         *        If you don't specify <code>MinConfidence</code>, the operation returns labels and bounding boxes (if
         *        detected) with confidence values greater than or equal to 50 percent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minConfidence(Float minConfidence);

        /**
         * <p>
         * The Amazon SNS topic ARN you want Amazon Rekognition Video to publish the completion status of the label
         * detection operation to. The Amazon SNS topic must have a topic name that begins with <i>AmazonRekognition</i>
         * if you are using the AmazonRekognitionServiceRole permissions policy.
         * </p>
         * 
         * @param notificationChannel
         *        The Amazon SNS topic ARN you want Amazon Rekognition Video to publish the completion status of the
         *        label detection operation to. The Amazon SNS topic must have a topic name that begins with
         *        <i>AmazonRekognition</i> if you are using the AmazonRekognitionServiceRole permissions policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationChannel(NotificationChannel notificationChannel);

        /**
         * <p>
         * The Amazon SNS topic ARN you want Amazon Rekognition Video to publish the completion status of the label
         * detection operation to. The Amazon SNS topic must have a topic name that begins with <i>AmazonRekognition</i>
         * if you are using the AmazonRekognitionServiceRole permissions policy.
         * </p>
         * This is a convenience method that creates an instance of the {@link NotificationChannel.Builder} avoiding the
         * need to create one manually via {@link NotificationChannel#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NotificationChannel.Builder#build()} is called immediately and
         * its result is passed to {@link #notificationChannel(NotificationChannel)}.
         * 
         * @param notificationChannel
         *        a consumer that will call methods on {@link NotificationChannel.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notificationChannel(NotificationChannel)
         */
        default Builder notificationChannel(Consumer<NotificationChannel.Builder> notificationChannel) {
            return notificationChannel(NotificationChannel.builder().applyMutation(notificationChannel).build());
        }

        /**
         * <p>
         * An identifier you specify that's returned in the completion notification that's published to your Amazon
         * Simple Notification Service topic. For example, you can use <code>JobTag</code> to group related jobs and
         * identify them in the completion notification.
         * </p>
         * 
         * @param jobTag
         *        An identifier you specify that's returned in the completion notification that's published to your
         *        Amazon Simple Notification Service topic. For example, you can use <code>JobTag</code> to group
         *        related jobs and identify them in the completion notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobTag(String jobTag);

        /**
         * <p>
         * The features to return after video analysis. You can specify that GENERAL_LABELS are returned.
         * </p>
         * 
         * @param features
         *        The features to return after video analysis. You can specify that GENERAL_LABELS are returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featuresWithStrings(Collection<String> features);

        /**
         * <p>
         * The features to return after video analysis. You can specify that GENERAL_LABELS are returned.
         * </p>
         * 
         * @param features
         *        The features to return after video analysis. You can specify that GENERAL_LABELS are returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featuresWithStrings(String... features);

        /**
         * <p>
         * The features to return after video analysis. You can specify that GENERAL_LABELS are returned.
         * </p>
         * 
         * @param features
         *        The features to return after video analysis. You can specify that GENERAL_LABELS are returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder features(Collection<LabelDetectionFeatureName> features);

        /**
         * <p>
         * The features to return after video analysis. You can specify that GENERAL_LABELS are returned.
         * </p>
         * 
         * @param features
         *        The features to return after video analysis. You can specify that GENERAL_LABELS are returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder features(LabelDetectionFeatureName... features);

        /**
         * <p>
         * The settings for a StartLabelDetection request.Contains the specified parameters for the label detection
         * request of an asynchronous label analysis operation. Settings can include filters for GENERAL_LABELS.
         * </p>
         * 
         * @param settings
         *        The settings for a StartLabelDetection request.Contains the specified parameters for the label
         *        detection request of an asynchronous label analysis operation. Settings can include filters for
         *        GENERAL_LABELS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder settings(LabelDetectionSettings settings);

        /**
         * <p>
         * The settings for a StartLabelDetection request.Contains the specified parameters for the label detection
         * request of an asynchronous label analysis operation. Settings can include filters for GENERAL_LABELS.
         * </p>
         * This is a convenience method that creates an instance of the {@link LabelDetectionSettings.Builder} avoiding
         * the need to create one manually via {@link LabelDetectionSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LabelDetectionSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #settings(LabelDetectionSettings)}.
         * 
         * @param settings
         *        a consumer that will call methods on {@link LabelDetectionSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #settings(LabelDetectionSettings)
         */
        default Builder settings(Consumer<LabelDetectionSettings.Builder> settings) {
            return settings(LabelDetectionSettings.builder().applyMutation(settings).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends RekognitionRequest.BuilderImpl implements Builder {
        private Video video;

        private String clientRequestToken;

        private Float minConfidence;

        private NotificationChannel notificationChannel;

        private String jobTag;

        private List<String> features = DefaultSdkAutoConstructList.getInstance();

        private LabelDetectionSettings settings;

        private BuilderImpl() {
        }

        private BuilderImpl(StartLabelDetectionRequest model) {
            super(model);
            video(model.video);
            clientRequestToken(model.clientRequestToken);
            minConfidence(model.minConfidence);
            notificationChannel(model.notificationChannel);
            jobTag(model.jobTag);
            featuresWithStrings(model.features);
            settings(model.settings);
        }

        public final Video.Builder getVideo() {
            return video != null ? video.toBuilder() : null;
        }

        public final void setVideo(Video.BuilderImpl video) {
            this.video = video != null ? video.build() : null;
        }

        @Override
        public final Builder video(Video video) {
            this.video = video;
            return this;
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final Float getMinConfidence() {
            return minConfidence;
        }

        public final void setMinConfidence(Float minConfidence) {
            this.minConfidence = minConfidence;
        }

        @Override
        public final Builder minConfidence(Float minConfidence) {
            this.minConfidence = minConfidence;
            return this;
        }

        public final NotificationChannel.Builder getNotificationChannel() {
            return notificationChannel != null ? notificationChannel.toBuilder() : null;
        }

        public final void setNotificationChannel(NotificationChannel.BuilderImpl notificationChannel) {
            this.notificationChannel = notificationChannel != null ? notificationChannel.build() : null;
        }

        @Override
        public final Builder notificationChannel(NotificationChannel notificationChannel) {
            this.notificationChannel = notificationChannel;
            return this;
        }

        public final String getJobTag() {
            return jobTag;
        }

        public final void setJobTag(String jobTag) {
            this.jobTag = jobTag;
        }

        @Override
        public final Builder jobTag(String jobTag) {
            this.jobTag = jobTag;
            return this;
        }

        public final Collection<String> getFeatures() {
            if (features instanceof SdkAutoConstructList) {
                return null;
            }
            return features;
        }

        public final void setFeatures(Collection<String> features) {
            this.features = LabelDetectionFeatureListCopier.copy(features);
        }

        @Override
        public final Builder featuresWithStrings(Collection<String> features) {
            this.features = LabelDetectionFeatureListCopier.copy(features);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder featuresWithStrings(String... features) {
            featuresWithStrings(Arrays.asList(features));
            return this;
        }

        @Override
        public final Builder features(Collection<LabelDetectionFeatureName> features) {
            this.features = LabelDetectionFeatureListCopier.copyEnumToString(features);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder features(LabelDetectionFeatureName... features) {
            features(Arrays.asList(features));
            return this;
        }

        public final LabelDetectionSettings.Builder getSettings() {
            return settings != null ? settings.toBuilder() : null;
        }

        public final void setSettings(LabelDetectionSettings.BuilderImpl settings) {
            this.settings = settings != null ? settings.build() : null;
        }

        @Override
        @Transient
        public final Builder settings(LabelDetectionSettings settings) {
            this.settings = settings;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public StartLabelDetectionRequest build() {
            return new StartLabelDetectionRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
