/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jena.geosparql.spatial;

import java.lang.invoke.MethodHandles;
import java.util.HashSet;
import java.util.Objects;
import org.apache.jena.geosparql.implementation.GeometryWrapper;
import org.apache.jena.geosparql.implementation.SRSInfo;
import org.apache.jena.geosparql.implementation.UnitsOfMeasure;
import org.apache.jena.geosparql.implementation.great_circle.GreatCirclePointDistance;
import org.apache.jena.geosparql.implementation.great_circle.LatLonPoint;
import org.apache.jena.geosparql.spatial.CardinalDirection;
import org.apache.jena.geosparql.spatial.SpatialIndex;
import org.apache.jena.rdf.model.Resource;
import org.apache.jena.sparql.expr.ExprEvalException;
import org.locationtech.jts.geom.Envelope;
import org.opengis.geometry.MismatchedDimensionException;
import org.opengis.referencing.operation.TransformException;
import org.opengis.util.FactoryException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SearchEnvelope {
    private final Envelope mainEnvelope;
    private final Envelope wrapEnvelope;
    private final SRSInfo srsInfo;
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final double NORTH_BEARING = 0.0;
    private static final double EAST_BEARING = Math.toRadians(90.0);
    private static final double SOUTH_BEARING = Math.toRadians(180.0);
    private static final double WEST_BEARING = Math.toRadians(270.0);

    protected SearchEnvelope(Envelope envelope, SRSInfo srsInfo) {
        this.srsInfo = srsInfo;
        Envelope domainEnvelope = srsInfo.getDomainEnvelope();
        double domMinX = domainEnvelope.getMinX();
        double domMaxX = domainEnvelope.getMaxX();
        if (srsInfo.isGeographic().booleanValue()) {
            double domRange = srsInfo.getDomainRangeX();
            double minX = envelope.getMinX();
            double maxX = envelope.getMaxX();
            if (minX < domMinX) {
                this.mainEnvelope = new Envelope(domMinX, maxX, envelope.getMinY(), envelope.getMaxY());
                this.wrapEnvelope = new Envelope(minX + domRange, domMaxX, envelope.getMinY(), envelope.getMaxY());
            } else if (maxX > domMaxX) {
                this.mainEnvelope = new Envelope(minX, domMaxX, envelope.getMinY(), envelope.getMaxY());
                this.wrapEnvelope = new Envelope(domMinX, maxX - domRange, envelope.getMinY(), envelope.getMaxY());
            } else {
                this.mainEnvelope = envelope;
                this.wrapEnvelope = null;
            }
            double envRange = Math.abs(minX) + Math.abs(maxX);
            if (envRange > domRange * 2.0) {
                LOGGER.warn("Search Envelope {} is outside of valid domain {} for SRS URI {}.", new Object[]{envelope, domainEnvelope, srsInfo.getSrsURI()});
            }
        } else {
            this.mainEnvelope = envelope;
            this.wrapEnvelope = null;
            double minX = envelope.getMinX();
            double maxX = envelope.getMaxX();
            if (minX < domMinX || maxX > domMaxX) {
                LOGGER.warn("Search Envelope {} is outside of valid domain {} for SRS URI {}.", new Object[]{envelope, domainEnvelope, srsInfo.getSrsURI()});
            }
        }
    }

    public Envelope getMainEnvelope() {
        return this.mainEnvelope;
    }

    public Envelope getWrapEnvelope() {
        return this.wrapEnvelope;
    }

    public String getSrsURI() {
        return this.srsInfo.getSrsURI();
    }

    public SRSInfo getCrsInfo() {
        return this.srsInfo;
    }

    public HashSet<Resource> check(SpatialIndex spatialIndex) {
        HashSet<Resource> features = spatialIndex.query(this.mainEnvelope);
        if (this.wrapEnvelope != null) {
            HashSet<Resource> wrapFeatures = spatialIndex.query(this.wrapEnvelope);
            features.addAll(wrapFeatures);
        }
        return features;
    }

    public boolean check(Envelope envelope) {
        boolean result = this.mainEnvelope.intersects(envelope);
        if (!result && this.wrapEnvelope != null) {
            result = this.wrapEnvelope.intersects(envelope);
        }
        return result;
    }

    public String toString() {
        return "SearchEnvelope{envelope=" + this.mainEnvelope + ", wrappedEnvelope=" + this.wrapEnvelope + "}";
    }

    public int hashCode() {
        int hash = 5;
        hash = 47 * hash + Objects.hashCode(this.mainEnvelope);
        hash = 47 * hash + Objects.hashCode(this.wrapEnvelope);
        return hash;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        SearchEnvelope other = (SearchEnvelope)obj;
        if (!Objects.equals(this.mainEnvelope, other.mainEnvelope)) {
            return false;
        }
        return Objects.equals(this.wrapEnvelope, other.wrapEnvelope);
    }

    public static SearchEnvelope build(GeometryWrapper geometryWrapper, SRSInfo srsInfo, double radius, String unitsURI) {
        try {
            SearchEnvelope searchEnvelope;
            GeometryWrapper envelopeGeometryWrapper = geometryWrapper.envelope();
            GeometryWrapper srsGeometryWrapper = envelopeGeometryWrapper.convertSRS(srsInfo.getSrsURI());
            Envelope envelope = srsGeometryWrapper.getEnvelope();
            if (srsInfo.isGeographic().booleanValue()) {
                double radiusMetres = UnitsOfMeasure.convertToMetres(radius, unitsURI, srsGeometryWrapper.getLatitude());
                searchEnvelope = SearchEnvelope.expandGeographicEnvelope(envelope, radiusMetres, srsInfo);
            } else {
                String targetUnitsURI = srsInfo.getUnitsOfMeasure().getUnitURI();
                double targetRadius = UnitsOfMeasure.conversion(radius, unitsURI, targetUnitsURI);
                searchEnvelope = SearchEnvelope.expandEnvelope(envelope, targetRadius, srsInfo);
            }
            return searchEnvelope;
        }
        catch (MismatchedDimensionException | TransformException | FactoryException ex) {
            throw new ExprEvalException(ex.getMessage() + ": " + geometryWrapper.asLiteral(), ex);
        }
    }

    private static SearchEnvelope expandGeographicEnvelope(Envelope envelope, double distanceMetres, SRSInfo srsInfo) {
        double minLon = envelope.getMinX();
        double minLat = envelope.getMinY();
        double maxLon = envelope.getMaxX();
        double maxLat = envelope.getMaxY();
        double extLat = Math.abs(maxLat) > Math.abs(minLat) ? maxLat : minLat;
        double extLon = Math.abs(maxLon) > Math.abs(minLon) ? maxLon : minLon;
        double latBearing = extLat > 0.0 ? 0.0 : SOUTH_BEARING;
        double lonBearing = extLon > 0.0 ? EAST_BEARING : WEST_BEARING;
        GreatCirclePointDistance pointDistance = new GreatCirclePointDistance(extLat, extLon, distanceMetres);
        double latRad = pointDistance.latitude(latBearing);
        double lonRad = pointDistance.longitude(latRad, lonBearing);
        LatLonPoint point = GreatCirclePointDistance.radToPoint(latRad, lonRad, false);
        double latDiff = Math.abs(extLat - point.getLat());
        double lonDiff = Math.abs(extLon - point.getLon());
        double normMinLon = GreatCirclePointDistance.normaliseLongitude(minLon - lonDiff);
        double normMaxLon = GreatCirclePointDistance.normaliseLongitude(maxLon + lonDiff);
        Envelope expandedEnvelope = new Envelope(normMinLon, normMaxLon, minLat - latDiff, maxLat + latDiff);
        SearchEnvelope searchEnvelope = new SearchEnvelope(expandedEnvelope, srsInfo);
        return searchEnvelope;
    }

    private static SearchEnvelope expandEnvelope(Envelope envelope, double distance, SRSInfo srsInfo) {
        double x1 = envelope.getMinX() - distance;
        double y1 = envelope.getMinY() - distance;
        double x2 = envelope.getMaxX() + distance;
        double y2 = envelope.getMaxY() + distance;
        Envelope expandedEnvelope = new Envelope(x1, x2, y1, y2);
        SearchEnvelope searchEnvelope = new SearchEnvelope(expandedEnvelope, srsInfo);
        return searchEnvelope;
    }

    public static SearchEnvelope build(GeometryWrapper geometryWrapper, SRSInfo srsInfo) {
        try {
            GeometryWrapper envelopeGeometryWrapper = geometryWrapper.envelope();
            GeometryWrapper srsGeometryWrapper = envelopeGeometryWrapper.convertSRS(srsInfo.getSrsURI());
            Envelope envelope = srsGeometryWrapper.getEnvelope();
            SearchEnvelope searchEnvelope = new SearchEnvelope(envelope, srsInfo);
            return searchEnvelope;
        }
        catch (MismatchedDimensionException | TransformException | FactoryException ex) {
            throw new ExprEvalException(ex.getMessage() + ": " + geometryWrapper.asLiteral(), ex);
        }
    }

    public static SearchEnvelope build(GeometryWrapper geometryWrapper, SRSInfo srsInfo, CardinalDirection direction) {
        try {
            GeometryWrapper envelopeGeometryWrapper = geometryWrapper.envelope();
            GeometryWrapper srsGeometryWrapper = envelopeGeometryWrapper.convertSRS(srsInfo.getSrsURI());
            Envelope envelope = srsGeometryWrapper.getEnvelope();
            Envelope domEnvelope = srsInfo.getDomainEnvelope();
            double x1 = domEnvelope.getMinX();
            double x2 = domEnvelope.getMaxX();
            double y1 = domEnvelope.getMinY();
            double y2 = domEnvelope.getMaxY();
            switch (direction) {
                case NORTH: {
                    y1 = envelope.getMaxY();
                    break;
                }
                case SOUTH: {
                    y2 = envelope.getMinY();
                    break;
                }
                case EAST: {
                    x1 = envelope.getMaxX();
                    if (!srsInfo.isGeographic().booleanValue()) break;
                    x2 = x1 + domEnvelope.getMaxX();
                    break;
                }
                case WEST: {
                    x2 = envelope.getMinX();
                    if (!srsInfo.isGeographic().booleanValue()) break;
                    x1 = x2 + domEnvelope.getMinX();
                }
            }
            Envelope cardinalEnvelope = new Envelope(x1, x2, y1, y2);
            return new SearchEnvelope(cardinalEnvelope, srsInfo);
        }
        catch (MismatchedDimensionException | TransformException | FactoryException ex) {
            throw new ExprEvalException(ex.getMessage() + ": " + geometryWrapper.asLiteral(), ex);
        }
    }
}

