/*
 * Decompiled with CFR 0.152.
 */
package com.lambdaworks.redis.cluster;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.lambdaworks.redis.RedisCommandInterruptedException;
import com.lambdaworks.redis.RedisConnectionException;
import com.lambdaworks.redis.RedisFuture;
import com.lambdaworks.redis.RedisURI;
import com.lambdaworks.redis.api.StatefulRedisConnection;
import com.lambdaworks.redis.cluster.RedisClusterClient;
import com.lambdaworks.redis.cluster.models.partitions.ClusterPartitionParser;
import com.lambdaworks.redis.cluster.models.partitions.Partitions;
import com.lambdaworks.redis.cluster.models.partitions.RedisClusterNode;
import com.lambdaworks.redis.codec.Utf8StringCodec;
import com.lambdaworks.redis.output.StatusOutput;
import com.lambdaworks.redis.protocol.AsyncCommand;
import com.lambdaworks.redis.protocol.Command;
import com.lambdaworks.redis.protocol.CommandArgs;
import com.lambdaworks.redis.protocol.CommandKeyword;
import com.lambdaworks.redis.protocol.CommandType;
import com.lambdaworks.redis.protocol.RedisCommand;
import io.netty.buffer.ByteBuf;
import io.netty.util.internal.logging.InternalLogger;
import io.netty.util.internal.logging.InternalLoggerFactory;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

class ClusterTopologyRefresh {
    private static final Utf8StringCodec CODEC = new Utf8StringCodec();
    private static final InternalLogger logger = InternalLoggerFactory.getInstance(ClusterTopologyRefresh.class);
    private final RedisClusterClient client;

    public ClusterTopologyRefresh(RedisClusterClient client) {
        this.client = client;
    }

    static boolean isChanged(Partitions o1, Partitions o2) {
        if (o1.size() != o2.size()) {
            return true;
        }
        for (RedisClusterNode base : o2) {
            if (ClusterTopologyRefresh.essentiallyEqualsTo(base, o1.getPartitionByNodeId(base.getNodeId()))) continue;
            return true;
        }
        return false;
    }

    static List<RedisClusterNode> createSortedList(Iterable<RedisClusterNode> clusterNodes) {
        ArrayList ordered = Lists.newArrayList(clusterNodes);
        Collections.sort(ordered, (o1, o2) -> RedisUriComparator.INSTANCE.compare(o1.getUri(), o2.getUri()));
        return ordered;
    }

    static boolean essentiallyEqualsTo(RedisClusterNode o1, RedisClusterNode o2) {
        if (o2 == null) {
            return false;
        }
        if (!ClusterTopologyRefresh.sameFlags(o1, o2, RedisClusterNode.NodeFlag.MASTER)) {
            return false;
        }
        if (!ClusterTopologyRefresh.sameFlags(o1, o2, RedisClusterNode.NodeFlag.SLAVE)) {
            return false;
        }
        return Sets.newHashSet(o1.getSlots()).equals(Sets.newHashSet(o2.getSlots()));
    }

    private static boolean sameFlags(RedisClusterNode base, RedisClusterNode other, RedisClusterNode.NodeFlag flag) {
        return !(base.getFlags().contains((Object)flag) ? !other.getFlags().contains((Object)flag) : other.getFlags().contains((Object)flag));
    }

    public Map<RedisURI, Partitions> loadViews(Iterable<RedisURI> seed) {
        Map<RedisURI, StatefulRedisConnection<String, String>> connections = this.getConnections(seed);
        Map<RedisURI, TimedAsyncCommand<String, String, String>> rawViews = this.requestViews(connections);
        Map<RedisURI, Partitions> nodeSpecificViews = this.getNodeSpecificViews(rawViews);
        this.close(connections);
        return nodeSpecificViews;
    }

    protected Map<RedisURI, Partitions> getNodeSpecificViews(Map<RedisURI, TimedAsyncCommand<String, String, String>> rawViews) {
        TreeMap nodeSpecificViews = Maps.newTreeMap((Comparator)RedisUriComparator.INSTANCE);
        long timeout = this.client.getFirstUri().getUnit().toNanos(this.client.getFirstUri().getTimeout());
        long waitTime = 0L;
        HashMap latencies = Maps.newHashMap();
        for (Map.Entry<RedisURI, TimedAsyncCommand<String, String, String>> entry : rawViews.entrySet()) {
            long timeoutLeft = timeout - waitTime;
            if (timeoutLeft <= 0L) break;
            long startWait = System.nanoTime();
            RedisFuture future = entry.getValue();
            try {
                if (!future.await(timeoutLeft, TimeUnit.NANOSECONDS)) break;
                waitTime += System.nanoTime() - startWait;
                String raw = (String)future.get();
                Partitions partitions = ClusterPartitionParser.parse(raw);
                ArrayList badNodes = Lists.newArrayList();
                for (RedisClusterNode partition : partitions) {
                    if (partition.getFlags().contains((Object)RedisClusterNode.NodeFlag.NOADDR)) {
                        badNodes.add(partition);
                    }
                    if (!partition.getFlags().contains((Object)RedisClusterNode.NodeFlag.MYSELF)) continue;
                    partition.setUri(entry.getKey());
                    latencies.put(partition.getNodeId(), entry.getValue().duration());
                }
                if (!badNodes.isEmpty()) {
                    partitions.removeAll(badNodes);
                }
                nodeSpecificViews.put(entry.getKey(), partitions);
            }
            catch (InterruptedException e) {
                Thread.interrupted();
                throw new RedisCommandInterruptedException(e);
            }
            catch (ExecutionException e) {
                logger.warn("Cannot retrieve partition view from " + entry.getKey() + ", error: " + e.toString());
            }
        }
        LatencyComparator comparator = new LatencyComparator(latencies);
        for (Partitions redisClusterNodes : nodeSpecificViews.values()) {
            Collections.sort(redisClusterNodes.getPartitions(), comparator);
        }
        return nodeSpecificViews;
    }

    private Map<RedisURI, TimedAsyncCommand<String, String, String>> requestViews(Map<RedisURI, StatefulRedisConnection<String, String>> connections) {
        TreeMap rawViews = Maps.newTreeMap((Comparator)RedisUriComparator.INSTANCE);
        for (Map.Entry<RedisURI, StatefulRedisConnection<String, String>> entry : connections.entrySet()) {
            TimedAsyncCommand<String, String, String> timed = this.createClusterNodesCommand();
            entry.getValue().dispatch(timed);
            rawViews.put(entry.getKey(), timed);
        }
        return rawViews;
    }

    protected TimedAsyncCommand<String, String, String> createClusterNodesCommand() {
        CommandArgs<String, String> args = new CommandArgs<String, String>(CODEC).add(CommandKeyword.NODES);
        Command command = new Command(CommandType.CLUSTER, new StatusOutput<String, String>(CODEC), args);
        return new TimedAsyncCommand<String, String, String>(command);
    }

    private void close(Map<RedisURI, StatefulRedisConnection<String, String>> connections) {
        for (StatefulRedisConnection<String, String> connection : connections.values()) {
            connection.close();
        }
    }

    private Map<RedisURI, StatefulRedisConnection<String, String>> getConnections(Iterable<RedisURI> seed) {
        TreeMap connections = Maps.newTreeMap((Comparator)RedisUriComparator.INSTANCE);
        for (RedisURI redisURI : seed) {
            if (redisURI.getResolvedAddress() == null) continue;
            try {
                StatefulRedisConnection<String, String> connection = this.client.connectToNode(redisURI.getResolvedAddress());
                if (redisURI.getPassword() != null && redisURI.getPassword().length != 0) {
                    connection.sync().auth(new String(redisURI.getPassword()));
                }
                connections.put(redisURI, connection);
            }
            catch (RedisConnectionException e) {
                if (logger.isDebugEnabled()) {
                    logger.debug(e.getMessage(), (Throwable)e);
                    continue;
                }
                logger.warn(e.getMessage());
            }
            catch (RuntimeException e) {
                logger.warn("Cannot connect to " + redisURI, (Throwable)e);
            }
        }
        return connections;
    }

    protected RedisURI getViewedBy(Map<RedisURI, Partitions> map, Partitions partitions) {
        for (Map.Entry<RedisURI, Partitions> entry : map.entrySet()) {
            if (entry.getValue() != partitions) continue;
            return entry.getKey();
        }
        return null;
    }

    static class LatencyComparator
    implements Comparator<RedisClusterNode> {
        private final Map<String, Long> latencies;

        public LatencyComparator(Map<String, Long> latencies) {
            this.latencies = latencies;
        }

        @Override
        public int compare(RedisClusterNode o1, RedisClusterNode o2) {
            Long latency1 = this.latencies.get(o1.getNodeId());
            Long latency2 = this.latencies.get(o2.getNodeId());
            if (latency1 != null && latency2 != null) {
                return latency1.compareTo(latency2);
            }
            if (latency1 != null && latency2 == null) {
                return -1;
            }
            if (latency1 == null && latency2 != null) {
                return 1;
            }
            return 0;
        }
    }

    static class TimedAsyncCommand<K, V, T>
    extends AsyncCommand<K, V, T> {
        long encodedAtNs = -1L;
        long completedAtNs = -1L;

        public TimedAsyncCommand(RedisCommand<K, V, T> command) {
            super(command);
        }

        @Override
        public void encode(ByteBuf buf) {
            this.completedAtNs = -1L;
            this.encodedAtNs = -1L;
            super.encode(buf);
            this.encodedAtNs = System.nanoTime();
        }

        @Override
        public void complete() {
            this.completedAtNs = System.nanoTime();
            super.complete();
        }

        public long duration() {
            if (this.completedAtNs == -1L || this.encodedAtNs == -1L) {
                return -1L;
            }
            return this.completedAtNs - this.encodedAtNs;
        }
    }

    static class RedisUriComparator
    implements Comparator<RedisURI> {
        public static final RedisUriComparator INSTANCE = new RedisUriComparator();

        RedisUriComparator() {
        }

        @Override
        public int compare(RedisURI o1, RedisURI o2) {
            String h1 = "";
            String h2 = "";
            if (o1 != null) {
                h1 = o1.getHost() + ":" + o1.getPort();
            }
            if (o2 != null) {
                h2 = o2.getHost() + ":" + o2.getPort();
            }
            return h1.compareToIgnoreCase(h2);
        }
    }
}

