/*
 * Decompiled with CFR 0.152.
 */
package ca.uhn.fhir.jpa.search.cache;

import ca.uhn.fhir.i18n.Msg;
import ca.uhn.fhir.interceptor.model.RequestPartitionId;
import ca.uhn.fhir.jpa.api.config.JpaStorageSettings;
import ca.uhn.fhir.jpa.dao.data.ISearchDao;
import ca.uhn.fhir.jpa.dao.data.ISearchIncludeDao;
import ca.uhn.fhir.jpa.dao.data.ISearchResultDao;
import ca.uhn.fhir.jpa.dao.data.SearchIdAndResultSize;
import ca.uhn.fhir.jpa.dao.tx.HapiTransactionService;
import ca.uhn.fhir.jpa.dao.tx.IHapiTransactionService;
import ca.uhn.fhir.jpa.entity.Search;
import ca.uhn.fhir.jpa.model.search.SearchStatusEnum;
import ca.uhn.fhir.jpa.search.cache.ISearchCacheSvc;
import ca.uhn.fhir.system.HapiSystemProperties;
import com.google.common.annotations.VisibleForTesting;
import java.sql.Connection;
import java.time.Instant;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;
import javax.annotation.Nonnull;
import javax.persistence.EntityManager;
import org.apache.commons.lang3.Validate;
import org.hibernate.Session;
import org.hl7.fhir.dstu3.model.InstantType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

public class DatabaseSearchCacheSvcImpl
implements ISearchCacheSvc {
    public static final int DEFAULT_MAX_RESULTS_TO_DELETE_IN_ONE_STMT = 500;
    public static final int DEFAULT_MAX_RESULTS_TO_DELETE_IN_ONE_PAS = 50000;
    public static final long SEARCH_CLEANUP_JOB_INTERVAL_MILLIS = 60000L;
    public static final int DEFAULT_MAX_DELETE_CANDIDATES_TO_FIND = 2000;
    private static final Logger ourLog = LoggerFactory.getLogger(DatabaseSearchCacheSvcImpl.class);
    private static int ourMaximumResultsToDeleteInOneStatement = 500;
    private static int ourMaximumResultsToDeleteInOneCommit = 50000;
    private static Long ourNowForUnitTests;
    private long myCutoffSlack = 60000L;
    @Autowired
    private ISearchDao mySearchDao;
    @Autowired
    private EntityManager myEntityManager;
    @Autowired
    private ISearchResultDao mySearchResultDao;
    @Autowired
    private ISearchIncludeDao mySearchIncludeDao;
    @Autowired
    private IHapiTransactionService myTransactionService;
    @Autowired
    private JpaStorageSettings myStorageSettings;

    @VisibleForTesting
    public void setCutoffSlackForUnitTest(long theCutoffSlack) {
        this.myCutoffSlack = theCutoffSlack;
    }

    @Override
    public Search save(Search theSearch, RequestPartitionId theRequestPartitionId) {
        return (Search)this.myTransactionService.withSystemRequestOnPartition(theRequestPartitionId).execute(() -> (Search)this.mySearchDao.save(theSearch));
    }

    @Override
    @Transactional(propagation=Propagation.REQUIRED)
    public Optional<Search> fetchByUuid(String theUuid, RequestPartitionId theRequestPartitionId) {
        Validate.notBlank((CharSequence)theUuid);
        return (Optional)this.myTransactionService.withSystemRequestOnPartition(theRequestPartitionId).execute(() -> this.mySearchDao.findByUuidAndFetchIncludes(theUuid));
    }

    void setSearchDaoForUnitTest(ISearchDao theSearchDao) {
        this.mySearchDao = theSearchDao;
    }

    void setTransactionServiceForUnitTest(IHapiTransactionService theTransactionService) {
        this.myTransactionService = theTransactionService;
    }

    @Override
    public Optional<Search> tryToMarkSearchAsInProgress(Search theSearch, RequestPartitionId theRequestPartitionId) {
        ourLog.trace("Going to try to change search status from {} to {}", (Object)theSearch.getStatus(), (Object)SearchStatusEnum.LOADING);
        try {
            return (Optional)this.myTransactionService.withSystemRequest().withRequestPartitionId(theRequestPartitionId).withPropagation(Propagation.REQUIRES_NEW).execute(t -> {
                Search search = this.mySearchDao.findById(theSearch.getId()).orElse(theSearch);
                if (search.getStatus() != SearchStatusEnum.PASSCMPLET) {
                    throw new IllegalStateException(Msg.code((int)1167) + "Can't change to LOADING because state is " + search.getStatus());
                }
                search.setStatus(SearchStatusEnum.LOADING);
                Search newSearch = (Search)this.mySearchDao.save(search);
                return Optional.of(newSearch);
            });
        }
        catch (Exception e) {
            ourLog.warn("Failed to activate search: {}", (Object)e.toString());
            ourLog.trace("Failed to activate search", (Throwable)e);
            return Optional.empty();
        }
    }

    @Override
    public Optional<Search> findCandidatesForReuse(String theResourceType, String theQueryString, Instant theCreatedAfter, RequestPartitionId theRequestPartitionId) {
        HapiTransactionService.requireTransaction();
        String queryString = Search.createSearchQueryStringForStorage(theQueryString, theRequestPartitionId);
        int hashCode = queryString.hashCode();
        Collection<Search> candidates = this.mySearchDao.findWithCutoffOrExpiry(theResourceType, hashCode, Date.from(theCreatedAfter));
        for (Search nextCandidateSearch : candidates) {
            if (!queryString.equals(nextCandidateSearch.getSearchQueryString()) || !nextCandidateSearch.getCreated().toInstant().isAfter(theCreatedAfter)) continue;
            return Optional.of(nextCandidateSearch);
        }
        return Optional.empty();
    }

    @Override
    public void pollForStaleSearchesAndDeleteThem(RequestPartitionId theRequestPartitionId, Instant theDeadline) {
        HapiTransactionService.noTransactionAllowed();
        if (!this.myStorageSettings.isExpireSearchResults()) {
            return;
        }
        Date cutoff = this.getCutoff();
        DeleteRun run = new DeleteRun(theDeadline, cutoff, theRequestPartitionId);
        run.run();
    }

    @Nonnull
    private Date getCutoff() {
        long cutoffMillis = this.myStorageSettings.getExpireSearchResultsAfterMillis();
        if (this.myStorageSettings.getReuseCachedSearchResultsForMillis() != null) {
            cutoffMillis += this.myStorageSettings.getReuseCachedSearchResultsForMillis().longValue();
        }
        Date cutoff = new Date(DatabaseSearchCacheSvcImpl.now() - cutoffMillis - this.myCutoffSlack);
        if (ourNowForUnitTests != null) {
            ourLog.info("Searching for searches which are before {} - now is {}", (Object)new InstantType(cutoff), (Object)new InstantType(new Date(DatabaseSearchCacheSvcImpl.now())));
        }
        return cutoff;
    }

    @VisibleForTesting
    public static void setMaximumResultsToDeleteInOnePassForUnitTest(int theMaximumResultsToDeleteInOnePass) {
        ourMaximumResultsToDeleteInOneCommit = theMaximumResultsToDeleteInOnePass;
    }

    @VisibleForTesting
    public static void setMaximumResultsToDeleteInOneStatement(int theMaximumResultsToDelete) {
        ourMaximumResultsToDeleteInOneStatement = theMaximumResultsToDelete;
    }

    @VisibleForTesting
    public static void setNowForUnitTests(Long theNowForUnitTests) {
        ourNowForUnitTests = theNowForUnitTests;
    }

    public static long now() {
        if (ourNowForUnitTests != null) {
            return ourNowForUnitTests;
        }
        return System.currentTimeMillis();
    }

    class DeleteRun {
        final RequestPartitionId myRequestPartitionId;
        final Instant myDeadline;
        final Date myCutoffForDeletion;
        final Set<Long> myUpdateDeletedFlagBatch = new HashSet<Long>();
        final Set<Long> myDeleteSearchBatch = new HashSet<Long>();
        final Set<Long> myDeleteSearchResultsBatch = new HashSet<Long>();
        private int myDeleteSearchResultsBatchCount = 0;

        DeleteRun(Instant theDeadline, Date theCutoffForDeletion, RequestPartitionId theRequestPartitionId) {
            this.myDeadline = theDeadline;
            this.myCutoffForDeletion = theCutoffForDeletion;
            this.myRequestPartitionId = theRequestPartitionId;
        }

        public void flushDeleteMarks() {
            if (this.myUpdateDeletedFlagBatch.isEmpty()) {
                return;
            }
            ourLog.debug("Marking {} searches as deleted", (Object)this.myUpdateDeletedFlagBatch.size());
            DatabaseSearchCacheSvcImpl.this.mySearchDao.updateDeleted(this.myUpdateDeletedFlagBatch, true);
            this.myUpdateDeletedFlagBatch.clear();
            this.commitOpenChanges();
        }

        private void commitOpenChanges() {
            DatabaseSearchCacheSvcImpl.this.myEntityManager.flush();
            ((Session)DatabaseSearchCacheSvcImpl.this.myEntityManager.unwrap(Session.class)).doWork(Connection::commit);
        }

        void throwIfDeadlineExpired() {
            boolean result = Instant.ofEpochMilli(DatabaseSearchCacheSvcImpl.now()).isAfter(this.myDeadline);
            if (result) {
                throw new DeadlineException(Msg.code((int)2443) + "Deadline expired while cleaning Search cache - " + this.myDeadline);
            }
        }

        private int deleteMarkedSearchesInBatches() {
            AtomicInteger deletedCounter = new AtomicInteger(0);
            try (Stream<SearchIdAndResultSize> toDelete = DatabaseSearchCacheSvcImpl.this.mySearchDao.findDeleted();){
                assert (toDelete != null);
                toDelete.forEach(nextSearchToDelete -> {
                    this.throwIfDeadlineExpired();
                    this.deleteSearchAndResults(nextSearchToDelete.searchId, nextSearchToDelete.size);
                    deletedCounter.incrementAndGet();
                });
            }
            this.flushSearchResultDeletes();
            this.flushSearchAndIncludeDeletes();
            int deletedCount = deletedCounter.get();
            ourLog.info("Deleted {} expired searches", (Object)deletedCount);
            return deletedCount;
        }

        private void deleteSearchAndResults(long theSearchPid, int theNumberOfResults) {
            ourLog.trace("Buffering deletion of search pid {} and {} results", (Object)theSearchPid, (Object)theNumberOfResults);
            this.myDeleteSearchBatch.add(theSearchPid);
            if (theNumberOfResults > ourMaximumResultsToDeleteInOneCommit) {
                this.deleteSearchResultsByChunk(theSearchPid, theNumberOfResults);
                return;
            }
            this.myDeleteSearchResultsBatch.add(theSearchPid);
            this.myDeleteSearchResultsBatchCount += theNumberOfResults;
            if (this.myDeleteSearchResultsBatchCount > ourMaximumResultsToDeleteInOneCommit) {
                this.flushSearchResultDeletes();
            }
            if (this.myDeleteSearchBatch.size() > ourMaximumResultsToDeleteInOneStatement) {
                this.flushSearchResultDeletes();
                this.flushSearchAndIncludeDeletes();
            }
        }

        private void deleteSearchResultsByChunk(long theSearchPid, int theNumberOfResults) {
            ourLog.debug("Search {} is large: has {} results.  Deleting results in chunks.", (Object)theSearchPid, (Object)theNumberOfResults);
            for (int rangeEnd = theNumberOfResults; rangeEnd >= 0; rangeEnd -= ourMaximumResultsToDeleteInOneCommit) {
                int rangeStart = rangeEnd - ourMaximumResultsToDeleteInOneCommit;
                ourLog.trace("Deleting results for search {}: {} - {}", new Object[]{theSearchPid, rangeStart, rangeEnd});
                DatabaseSearchCacheSvcImpl.this.mySearchResultDao.deleteBySearchIdInRange(theSearchPid, rangeStart, rangeEnd);
                this.commitOpenChanges();
            }
        }

        private void flushSearchAndIncludeDeletes() {
            if (this.myDeleteSearchBatch.isEmpty()) {
                return;
            }
            ourLog.debug("Deleting {} Search records", (Object)this.myDeleteSearchBatch.size());
            DatabaseSearchCacheSvcImpl.this.mySearchIncludeDao.deleteForSearch(this.myDeleteSearchBatch);
            DatabaseSearchCacheSvcImpl.this.mySearchDao.deleteByPids(this.myDeleteSearchBatch);
            this.myDeleteSearchBatch.clear();
            this.commitOpenChanges();
        }

        private void flushSearchResultDeletes() {
            if (this.myDeleteSearchResultsBatch.isEmpty()) {
                return;
            }
            ourLog.debug("Deleting {} Search Results from {} searches", (Object)this.myDeleteSearchResultsBatchCount, (Object)this.myDeleteSearchResultsBatch.size());
            DatabaseSearchCacheSvcImpl.this.mySearchResultDao.deleteBySearchIds(this.myDeleteSearchResultsBatch);
            this.myDeleteSearchResultsBatch.clear();
            this.myDeleteSearchResultsBatchCount = 0;
            this.commitOpenChanges();
        }

        IHapiTransactionService.IExecutionBuilder getTxBuilder() {
            return DatabaseSearchCacheSvcImpl.this.myTransactionService.withSystemRequest().withRequestPartitionId(this.myRequestPartitionId);
        }

        private void run() {
            ourLog.debug("Searching for searches which are before {}", (Object)this.myCutoffForDeletion);
            this.getTxBuilder().execute(theStatus -> {
                try {
                    this.markDeletedInBatches();
                    this.throwIfDeadlineExpired();
                    int deletedCount = this.deleteMarkedSearchesInBatches();
                    this.throwIfDeadlineExpired();
                    if ((ourLog.isDebugEnabled() || HapiSystemProperties.isTestModeEnabled()) && deletedCount > 0) {
                        Long total = DatabaseSearchCacheSvcImpl.this.mySearchDao.count();
                        ourLog.debug("Deleted {} searches, {} remaining", (Object)deletedCount, (Object)total);
                    }
                }
                catch (DeadlineException theTimeoutException) {
                    ourLog.warn(theTimeoutException.getMessage());
                }
                return null;
            });
        }

        private void markDeletedInBatches() {
            try (Stream<Long> toMarkDeleted = DatabaseSearchCacheSvcImpl.this.mySearchDao.findWhereCreatedBefore(this.myCutoffForDeletion, new Date(DatabaseSearchCacheSvcImpl.now()));){
                assert (toMarkDeleted != null);
                toMarkDeleted.forEach(nextSearchToDelete -> {
                    this.throwIfDeadlineExpired();
                    if (this.myUpdateDeletedFlagBatch.size() >= ourMaximumResultsToDeleteInOneStatement) {
                        this.flushDeleteMarks();
                    }
                    ourLog.trace("Marking search with PID {} as ready for deletion", nextSearchToDelete);
                    this.myUpdateDeletedFlagBatch.add((Long)nextSearchToDelete);
                });
                this.flushDeleteMarks();
            }
        }
    }

    private static class DeadlineException
    extends RuntimeException {
        public DeadlineException(String message) {
            super(message);
        }
    }
}

