/*
 * Decompiled with CFR 0.152.
 */
package co.elastic.apm.agent.log.shipper;

import co.elastic.apm.agent.log.shipper.FileChangeListener;
import co.elastic.apm.agent.shaded.slf4j.Logger;
import co.elastic.apm.agent.shaded.slf4j.LoggerFactory;
import java.io.Closeable;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Properties;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;

public class TailableFile
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(TailableFile.class);
    private static final byte NEW_LINE = 10;
    private static final int EOF = -1;
    private final File file;
    private final File stateFile;
    private final FileChannel stateFileChannel;
    private final FileLock stateFileLock;
    @Nullable
    private FileChannel fileChannel;
    private long fileCreationTime;
    private long inode;

    public TailableFile(File file) throws IOException {
        this.file = file;
        this.stateFile = new File(file + ".state");
        this.stateFileChannel = FileChannel.open(this.stateFile.toPath(), StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
        try {
            this.stateFileLock = this.stateFileChannel.tryLock();
            if (this.stateFileLock == null) {
                throw new IllegalStateException("This file is currently locked by another process: " + this.stateFile);
            }
        }
        catch (OverlappingFileLockException e) {
            throw new IllegalStateException("This file is currently locked by this process: " + this.stateFile, e);
        }
        Properties state = this.readState();
        if (!state.isEmpty()) {
            this.restoreState(state);
        } else {
            this.tryOpenFile();
        }
    }

    private Properties readState() throws IOException {
        Properties properties = new Properties();
        try (FileInputStream input = new FileInputStream(this.stateFile);){
            properties.load(input);
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
        return properties;
    }

    void deleteStateFile() {
        new File(this.file + ".state").delete();
    }

    public void deleteStateFileOnExit() {
        new File(this.file + ".state").deleteOnExit();
    }

    private void restoreState(Properties state) throws IOException {
        long inode;
        long position = Long.parseLong(state.getProperty("position", "0"));
        long creationTime = Long.parseLong(state.getProperty("creationTime", Long.toString(this.getCreationTime(this.file.toPath()))));
        if (this.hasRotated(creationTime, inode = Long.parseLong(state.getProperty("inode", Long.toString(this.getInode(this.file.toPath())))))) {
            this.openRotatedFile(position, creationTime, inode);
        } else {
            this.openExistingFile(position, this.file.toPath());
        }
    }

    private boolean hasRotated(long creationTime, long inode) throws IOException {
        if (inode != -1L) {
            return this.getInode(this.file.toPath()) != inode;
        }
        return this.getCreationTime(this.file.toPath()) != creationTime;
    }

    private void openRotatedFile(long position, long creationTime, long inode) throws IOException {
        File rotatedFile = inode == -1L ? this.findFileWithCreationDate(this.file.getParentFile(), creationTime) : this.findFileWithInode(this.file.getParentFile(), inode);
        if (rotatedFile != null && rotatedFile.length() > position) {
            this.openExistingFile(position, rotatedFile.toPath());
        }
    }

    private void saveState(FileChannel fileChannel, long fileCreationTime) throws IOException {
        Properties properties = new Properties();
        properties.put("position", Long.toString(fileChannel.position()));
        properties.put("creationTime", Long.toString(fileCreationTime));
        properties.put("inode", Long.toString(this.inode));
        try (FileOutputStream os = new FileOutputStream(this.stateFile);){
            properties.store(os, null);
        }
    }

    public void ack() {
        if (this.fileChannel == null) {
            throw new IllegalStateException("Can't acknowledge the state if the file has not been opened yet");
        }
        try {
            this.saveState(this.fileChannel, this.fileCreationTime);
        }
        catch (IOException e) {
            logger.error(e.getMessage(), e);
        }
    }

    public void nak() {
        try {
            Properties state = this.readState();
            if (!state.isEmpty()) {
                this.restoreState(state);
            }
        }
        catch (IOException e) {
            logger.error(e.getMessage(), e);
        }
    }

    @Nullable
    private File findFileWithCreationDate(File dir, final long creationTime) {
        File[] files = dir.listFiles(new FileFilter(){

            @Override
            public boolean accept(File file) {
                try {
                    return !file.getName().endsWith(".state") && TailableFile.this.getCreationTime(file.toPath()) == creationTime;
                }
                catch (IOException e) {
                    return false;
                }
            }
        });
        if (files != null && files.length == 1) {
            return files[0];
        }
        return null;
    }

    @Nullable
    private File findFileWithInode(File dir, final long inode) {
        File[] files = dir.listFiles(new FileFilter(){

            @Override
            public boolean accept(File file) {
                return TailableFile.this.getInode(file.toPath()) == inode;
            }
        });
        if (files != null && files.length == 1) {
            return files[0];
        }
        return null;
    }

    private long getCreationTime(Path path) throws IOException {
        BasicFileAttributes attr = Files.readAttributes(path, BasicFileAttributes.class, new LinkOption[0]);
        return attr.creationTime().to(TimeUnit.MILLISECONDS);
    }

    private long getInode(Path path) {
        try {
            return (Long)Files.getAttribute(path, "unix:ino", new LinkOption[0]);
        }
        catch (Exception e) {
            return -1L;
        }
    }

    public int tail(ByteBuffer buffer, FileChangeListener listener, int maxLines) throws IOException {
        FileChannel currentFile;
        int readLines;
        for (readLines = 0; readLines < maxLines; readLines += this.readFile(buffer, listener, maxLines - readLines, currentFile)) {
            currentFile = this.getFileChannel();
            if (currentFile != null && !this.isFullyRead()) continue;
            return readLines;
        }
        return readLines;
    }

    private int readFile(ByteBuffer buffer, FileChangeListener listener, int maxLines, FileChannel currentFile) throws IOException {
        int readLines;
        for (readLines = 0; readLines < maxLines; readLines += TailableFile.readLines(this, buffer, maxLines - readLines, listener)) {
            buffer.clear();
            int read = currentFile.read(buffer);
            if (read != -1) {
                buffer.flip();
                currentFile.position(currentFile.position() - (long)buffer.remaining());
                continue;
            }
            return readLines;
        }
        return readLines;
    }

    @Nullable
    private FileChannel getFileChannel() throws IOException {
        if (this.fileChannel == null || this.isFullyRead() && this.hasRotated()) {
            this.tryOpenFile();
        }
        return this.fileChannel;
    }

    private boolean hasRotated() throws IOException {
        return this.fileChannel != null && this.file.exists() && this.file.length() < this.fileChannel.position();
    }

    private boolean isFullyRead() throws IOException {
        return this.fileChannel != null && this.fileChannel.position() == this.fileChannel.size();
    }

    private void tryOpenFile() throws IOException {
        if (!this.file.exists()) {
            return;
        }
        this.openExistingFile(0L, this.file.toPath());
    }

    private void openExistingFile(long position, Path path) throws IOException {
        if (this.fileChannel != null) {
            this.fileChannel.close();
        }
        FileChannel fileChannel = FileChannel.open(path, new OpenOption[0]);
        fileChannel.position(position);
        this.fileCreationTime = this.getCreationTime(this.file.toPath());
        this.inode = this.getInode(this.file.toPath());
        if (this.fileChannel == null) {
            this.saveState(fileChannel, this.fileCreationTime);
        }
        this.fileChannel = fileChannel;
    }

    static int readLines(TailableFile file, ByteBuffer buffer, int maxLines, FileChangeListener listener) throws IOException {
        int lines = 0;
        while (buffer.hasRemaining() && lines < maxLines) {
            int startPos = buffer.position();
            boolean hasNewLine = TailableFile.skipUntil(buffer, (byte)10);
            int length = buffer.position() - startPos;
            if (hasNewLine) {
                if (--length > 0 && buffer.get(buffer.position() - 2) == 13) {
                    --length;
                }
                ++lines;
            }
            if (length <= 0) continue;
            while (!listener.onLineAvailable(file, buffer.array(), startPos, length, hasNewLine)) {
            }
        }
        return lines;
    }

    static boolean skipUntil(ByteBuffer bytes, byte b) {
        while (bytes.hasRemaining()) {
            if (bytes.get() != b) continue;
            return true;
        }
        return false;
    }

    @Override
    public void close() throws IOException {
        IOException exception = null;
        try {
            this.stateFileLock.release();
        }
        catch (IOException e) {
            exception = e;
        }
        try {
            this.stateFileChannel.close();
        }
        catch (IOException e) {
            if (exception != null) {
                e.addSuppressed(exception);
            }
            exception = e;
        }
        try {
            if (this.fileChannel != null) {
                this.fileChannel.close();
            }
        }
        catch (IOException e) {
            if (exception != null) {
                e.addSuppressed(exception);
            }
            exception = e;
        }
        if (exception != null) {
            throw exception;
        }
    }

    public File getFile() {
        return this.file;
    }

    public String toString() {
        return this.file.toString();
    }
}

