/*
 * Decompiled with CFR 0.152.
 */
package com.android.builder.png;

import com.android.annotations.NonNull;
import com.android.annotations.Nullable;
import com.android.builder.png.ByteUtils;
import com.android.builder.png.Chunk;
import com.android.builder.png.ColorType;
import com.android.builder.png.LayoutBoundChunkBuilder;
import com.android.builder.png.NinePatchChunkBuilder;
import com.android.builder.png.NinePatchException;
import com.android.builder.png.PngWriter;
import com.android.builder.png.TickException;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.io.Files;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.zip.Deflater;
import javax.imageio.ImageIO;

public class PngProcessor {
    private static final int COLOR_WHITE = -1;
    private static final int COLOR_TICK = -16777216;
    private static final int COLOR_LAYOUT_BOUNDS_TICK = -65536;
    private static final int PNG_9PATCH_NO_COLOR = 1;
    private static final int PNG_9PATCH_TRANSPARENT_COLOR = 0;
    @NonNull
    private final File mFile;
    private Chunk mIhdr;
    private Chunk mIdat;
    private List<Chunk> mOtherChunks = Lists.newArrayList();

    public static void process(@NonNull File from, @NonNull File to) throws IOException, NinePatchException {
        PngProcessor processor = new PngProcessor(from);
        processor.read();
        if (!processor.is9Patch() && processor.size() >= from.length()) {
            Files.copy((File)from, (File)to);
            return;
        }
        PngWriter writer = new PngWriter(to);
        writer.setIhdr(processor.getIhdr()).setChunks(processor.getOtherChunks()).setChunk(processor.getIdat());
        writer.write();
    }

    public static void clearCache() {
        ByteUtils.Cache.getCache().clear();
    }

    PngProcessor(@NonNull File file) {
        Preconditions.checkNotNull((Object)file);
        this.mFile = file;
    }

    @NonNull
    Chunk getIhdr() {
        return this.mIhdr;
    }

    @NonNull
    List<Chunk> getOtherChunks() {
        return this.mOtherChunks;
    }

    @NonNull
    Chunk getIdat() {
        return this.mIdat;
    }

    void read() throws IOException, NinePatchException {
        BufferedImage image = ImageIO.read(this.mFile);
        this.processImageContent(image);
    }

    private void addChunk(@NonNull Chunk chunk) {
        this.mOtherChunks.add(chunk);
    }

    long size() {
        long size = PngWriter.SIGNATURE.length;
        size += (long)this.mIhdr.size();
        size += (long)this.mIdat.size();
        for (Chunk chunk : this.mOtherChunks) {
            size += (long)chunk.size();
        }
        return size;
    }

    private void processImageContent(@NonNull BufferedImage image) throws NinePatchException, IOException {
        int width = image.getWidth();
        int height = image.getHeight();
        int[] content = new int[width * height];
        image.getRGB(0, 0, width, height, content, 0, width);
        int startX = 0;
        int startY = 0;
        int endX = width;
        int endY = height;
        if (this.is9Patch()) {
            startX = 1;
            startY = 1;
            --endX;
            --endY;
            this.processBorder(content, width, height);
        }
        ColorType colorType = this.createImage(content, width, startX, endX, startY, endY, this.is9Patch());
        this.mIhdr = this.computeIhdr(endX - startX, endY - startY, (byte)8, colorType);
    }

    private void writeIDat(byte[] data) throws IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream(data.length);
        Deflater deflater = new Deflater(8);
        deflater.setInput(data);
        deflater.finish();
        byte[] tmpBuffer = new byte[1024];
        while (!deflater.finished()) {
            int compressedLen = deflater.deflate(tmpBuffer);
            bos.write(tmpBuffer, 0, compressedLen);
        }
        bos.close();
        byte[] compressedData = bos.toByteArray();
        this.mIdat = new Chunk(PngWriter.IDAT, compressedData);
    }

    private ColorType createImage(@NonNull int[] content, int scanline, int startX, int endX, int startY, int endY, boolean is9Patch) throws IOException {
        int argb;
        int x;
        int y;
        ColorType colorType;
        int[] paletteColors = new int[256];
        int paletteColorCount = 0;
        int grayscaleTolerance = -1;
        int maxGrayDeviation = 0;
        boolean isOpaque = true;
        boolean isPalette = true;
        boolean isGrayscale = true;
        int width = endX - startX;
        int height = endY - startY;
        int rgbaLen = (1 + width * 4) * height;
        ByteBuffer rgbaBufer = ByteBuffer.allocate(rgbaLen);
        int indexedLen = (1 + width) * height;
        byte[] indexedContent = new byte[indexedLen];
        int indexedContentIndex = 0;
        for (int y2 = startY; y2 < endY; ++y2) {
            rgbaBufer.put((byte)0);
            indexedContent[indexedContentIndex++] = 0;
            for (int x2 = startX; x2 < endX; ++x2) {
                int argb2 = content[scanline * y2 + x2];
                int aa = argb2 >>> 24;
                int rr = argb2 >> 16 & 0xFF;
                int gg = argb2 >> 8 & 0xFF;
                int bb = argb2 & 0xFF;
                int odev = maxGrayDeviation;
                maxGrayDeviation = Math.max(Math.abs(rr - gg), maxGrayDeviation);
                maxGrayDeviation = Math.max(Math.abs(gg - bb), maxGrayDeviation);
                maxGrayDeviation = Math.max(Math.abs(bb - rr), maxGrayDeviation);
                if (isGrayscale && (rr != gg || rr != bb)) {
                    isGrayscale = false;
                }
                if (isOpaque && aa != 255) {
                    isOpaque = false;
                }
                if (isPalette) {
                    int idx;
                    int rgba = argb2 << 8 | aa;
                    boolean match = false;
                    for (idx = 0; idx < paletteColorCount; ++idx) {
                        if (paletteColors[idx] != rgba) continue;
                        match = true;
                        break;
                    }
                    indexedContent[indexedContentIndex++] = (byte)idx;
                    if (!match) {
                        if (paletteColorCount == 256) {
                            isPalette = false;
                        } else {
                            paletteColors[paletteColorCount++] = rgba;
                        }
                    }
                }
                rgbaBufer.putInt(argb2 << 8 | aa);
            }
        }
        boolean hasTransparency = !isOpaque;
        int bpp = isOpaque ? 3 : 4;
        int paletteSize = width * height + (isOpaque ? 3 : 4) * paletteColorCount;
        if (isGrayscale) {
            colorType = isOpaque ? ColorType.GRAY_SCALE : (isPalette && paletteSize < 2 * width * height ? ColorType.PLTE : ColorType.GRAY_SCALE_ALPHA);
        } else if (isPalette && paletteSize < bpp * width * height) {
            colorType = ColorType.PLTE;
        } else if (maxGrayDeviation <= grayscaleTolerance) {
            colorType = isOpaque ? ColorType.GRAY_SCALE : ColorType.GRAY_SCALE_ALPHA;
        } else {
            ColorType colorType2 = colorType = isOpaque ? ColorType.RGB : ColorType.RGBA;
        }
        if (is9Patch && (colorType == ColorType.RGB || colorType == ColorType.GRAY_SCALE || colorType == ColorType.PLTE)) {
            colorType = ColorType.RGBA;
        }
        if (colorType == ColorType.PLTE) {
            byte[] rgbPalette = new byte[paletteColorCount * 3];
            byte[] alphaPalette = null;
            if (hasTransparency) {
                alphaPalette = new byte[paletteColorCount];
            }
            for (int idx = 0; idx < paletteColorCount; ++idx) {
                int color = paletteColors[idx];
                rgbPalette[idx * 3] = (byte)(color >>> 24);
                rgbPalette[idx * 3 + 1] = (byte)(color >> 16 & 0xFF);
                rgbPalette[idx * 3 + 2] = (byte)(color >> 8 & 0xFF);
                if (!hasTransparency) continue;
                alphaPalette[idx] = (byte)(color & 0xFF);
            }
            this.addChunk(new Chunk(PngWriter.PLTE, rgbPalette));
            if (hasTransparency) {
                this.addChunk(new Chunk(PngWriter.TRNS, alphaPalette));
            }
            this.writeIDat(indexedContent);
        } else if (colorType == ColorType.GRAY_SCALE || colorType == ColorType.GRAY_SCALE_ALPHA) {
            int grayLen = (1 + width * (1 + (hasTransparency ? 1 : 0))) * height;
            byte[] grayContent = new byte[grayLen];
            int grayContentIndex = 0;
            for (y = startY; y < endY; ++y) {
                grayContent[grayContentIndex++] = 0;
                for (x = startX; x < endX; ++x) {
                    argb = content[scanline * y + x];
                    int rr = argb >> 16 & 0xFF;
                    if (isGrayscale) {
                        grayContent[grayContentIndex++] = (byte)rr;
                    } else {
                        int gg = argb >> 8 & 0xFF;
                        int bb = argb & 0xFF;
                        grayContent[grayContentIndex++] = (byte)((float)rr * 0.2126f + (float)gg * 0.7152f + (float)bb * 0.0722f);
                    }
                    if (!hasTransparency) continue;
                    int aa = argb >>> 24;
                    grayContent[grayContentIndex++] = (byte)aa;
                }
            }
            this.writeIDat(grayContent);
        } else if (colorType == ColorType.RGBA) {
            this.writeIDat(rgbaBufer.array());
        } else {
            int rgbLen = (1 + width * 3) * height;
            byte[] rgbContent = new byte[rgbLen];
            int rgbContentIndex = 0;
            for (y = startY; y < endY; ++y) {
                rgbContent[rgbContentIndex++] = 0;
                for (x = startX; x < endX; ++x) {
                    argb = content[scanline * y + x];
                    rgbContent[rgbContentIndex++] = (byte)(argb >> 16 & 0xFF);
                    rgbContent[rgbContentIndex++] = (byte)(argb >> 8 & 0xFF);
                    rgbContent[rgbContentIndex++] = (byte)(argb & 0xFF);
                }
            }
            this.writeIDat(rgbContent);
        }
        return colorType;
    }

    private void processBorder(int[] content, int width, int height) throws NinePatchException {
        int j;
        int paddingBottom;
        int paddingTop;
        int paddingRight;
        int paddingLeft;
        int numYDivs;
        int numXDivs;
        if (width < 3 || height < 3) {
            throw new NinePatchException(this.mFile, "Image must be at least 3x3 (1x1 without frame) pixels");
        }
        int[] xDivs = new int[width];
        int[] yDivs = new int[height];
        Arrays.fill(xDivs, -1);
        Arrays.fill(yDivs, -1);
        boolean transparent = (content[0] & 0xFF000000) == 0;
        int colorIndex = 0;
        if (!transparent && content[0] != -1) {
            throw new NinePatchException(this.mFile, "Must have one-pixel frame that is either transparent or white");
        }
        AtomicInteger outInt = new AtomicInteger(0);
        try {
            PngProcessor.getHorizontalTicks(content, 0, width, transparent, true, xDivs, 0, 1, outInt, true);
            numXDivs = outInt.get();
        }
        catch (TickException e) {
            throw new NinePatchException(this.mFile, e, "top");
        }
        outInt.set(0);
        try {
            PngProcessor.getVerticalTicks(content, 0, width, height, transparent, true, yDivs, 0, 1, outInt, true);
            numYDivs = outInt.get();
        }
        catch (TickException e) {
            throw new NinePatchException(this.mFile, e, "left");
        }
        int[] values = new int[2];
        try {
            PngProcessor.getHorizontalTicks(content, width * (height - 1), width, transparent, false, values, 0, 1, null, false);
            paddingLeft = values[0];
            paddingRight = values[1];
            values[1] = 0;
            values[0] = 0;
        }
        catch (TickException e) {
            throw new NinePatchException(this.mFile, e, "bottom");
        }
        try {
            PngProcessor.getVerticalTicks(content, width - 1, width, height, transparent, false, values, 0, 1, null, false);
            paddingTop = values[0];
            paddingBottom = values[1];
        }
        catch (TickException e) {
            throw new NinePatchException(this.mFile, e, "right");
        }
        try {
            PngProcessor.getHorizontalLayoutBoundsTicks(content, width * (height - 1), width, transparent, false, values);
        }
        catch (TickException e) {
            throw new NinePatchException(this.mFile, e, "bottom");
        }
        int[] values2 = new int[2];
        try {
            PngProcessor.getVerticalLayoutBoundsTicks(content, width - 1, width, height, transparent, false, values2);
        }
        catch (TickException e) {
            throw new NinePatchException(this.mFile, e, "right");
        }
        LayoutBoundChunkBuilder layoutBoundChunkBuilder = null;
        if (values[0] != 0 || values[1] != 0 || values2[0] != 0 || values2[1] != 0) {
            layoutBoundChunkBuilder = new LayoutBoundChunkBuilder(values[0], values2[0], values[1], values2[1]);
        }
        if (paddingLeft < 0) {
            paddingLeft = xDivs[0];
            paddingRight = width - 2 - xDivs[1];
        } else {
            paddingRight = width - 2 - paddingRight;
        }
        if (paddingTop < 0) {
            paddingTop = yDivs[0];
            paddingBottom = height - 2 - yDivs[1];
        } else {
            paddingBottom = height - 2 - paddingBottom;
        }
        width -= 2;
        height -= 2;
        int numCols = numXDivs + 1;
        if (xDivs[0] == 0) {
            --numCols;
        }
        if (xDivs[numXDivs - 1] == width) {
            --numCols;
        }
        int numRows = numYDivs + 1;
        if (yDivs[0] == 0) {
            --numRows;
        }
        if (yDivs[numYDivs - 1] == height) {
            --numRows;
        }
        if (numRows * numCols > 127) {
            throw new NinePatchException(this.mFile, "Too many rows and columns in 9-patch perimeter");
        }
        byte numColors = (byte)(numRows * numCols);
        int[] colors = new int[numColors];
        int top = 0;
        int n = j = yDivs[0] == 0 ? 1 : 0;
        while (j <= numYDivs && top < height) {
            int i;
            int bottom = j == numYDivs ? height : yDivs[j];
            int left = 0;
            int n2 = i = xDivs[0] == 0 ? 1 : 0;
            while (i <= numXDivs && left < width) {
                int right = i == numXDivs ? width : xDivs[i];
                int c = PngProcessor.getColor(content, width + 2, left, top, right - 1, bottom - 1);
                colors[colorIndex++] = c;
                left = right;
                ++i;
            }
            top = bottom;
            ++j;
        }
        NinePatchChunkBuilder ninePatchChunkBuilder = new NinePatchChunkBuilder(xDivs, numXDivs, yDivs, numYDivs, colors, paddingLeft, paddingRight, paddingTop, paddingBottom);
        this.addChunk(ninePatchChunkBuilder.getChunk());
        if (layoutBoundChunkBuilder != null) {
            this.addChunk(layoutBoundChunkBuilder.getChunk());
        }
    }

    static int getColor(@NonNull int[] content, int width, int left, int top, int right, int bottom) {
        int color = content[(top + 1) * width + left + 1];
        int alpha = color & 0xFF000000;
        if (left > right || top > bottom) {
            return 0;
        }
        while (top <= bottom) {
            for (int i = left; i <= right; ++i) {
                int c = content[(top + 1) * width + i + 1];
                if (!(alpha == 0 ? (c & 0xFF000000) != 0 : c != color)) continue;
                return 1;
            }
            ++top;
        }
        if (alpha == 0) {
            return 0;
        }
        return color;
    }

    @NonNull
    private static TickType getTickType(int color, boolean transparent) throws TickException {
        int alpha = color >>> 24;
        if (transparent) {
            if (alpha == 0) {
                return TickType.NONE;
            }
            if (color == -65536) {
                return TickType.LAYOUT_BOUNDS;
            }
            if (color == -16777216) {
                return TickType.TICK;
            }
            if (alpha != 255) {
                throw TickException.createWithColor("Frame pixels must be either solid or transparent (not intermediate alphas)", color);
            }
            if ((color & 0xFFFFFF) != 0) {
                throw TickException.createWithColor("Ticks in transparent frame must be black or red", color);
            }
            return TickType.TICK;
        }
        if (alpha != 255) {
            throw TickException.createWithColor("White frame must be a solid color (no alpha)", color);
        }
        if (color == -1) {
            return TickType.NONE;
        }
        if (color == -16777216) {
            return TickType.TICK;
        }
        if (color == -65536) {
            return TickType.LAYOUT_BOUNDS;
        }
        if ((color & 0xFFFFFF) != 0) {
            throw TickException.createWithColor("Ticks in transparent frame must be black or red", color);
        }
        return TickType.TICK;
    }

    private static void getHorizontalTicks(@NonNull int[] content, int offset, int width, boolean transparent, boolean required, @NonNull int[] divs, int left, int right, @Nullable AtomicInteger outDivs, boolean multipleAllowed) throws TickException {
        divs[right] = -1;
        divs[left] = -1;
        Tick state = Tick.START;
        boolean found = false;
        for (int i = 1; i < width - 1; ++i) {
            TickType tickType;
            try {
                tickType = PngProcessor.getTickType(content[offset + i], transparent);
            }
            catch (TickException e) {
                throw new TickException(e, i);
            }
            if (TickType.TICK == tickType) {
                if (state == Tick.START || state == Tick.OUTSIDE_1 && multipleAllowed) {
                    divs[left] = i - 1;
                    divs[right] = width - 2;
                    found = true;
                    if (outDivs != null) {
                        outDivs.addAndGet(2);
                    }
                    state = Tick.INSIDE_1;
                    continue;
                }
                if (state != Tick.OUTSIDE_1) continue;
                throw new TickException("Can't have more than one marked region along edge");
            }
            if (state != Tick.INSIDE_1) continue;
            divs[right] = i - 1;
            right += 2;
            left += 2;
            state = Tick.OUTSIDE_1;
        }
        if (required && !found) {
            throw new TickException("No marked region found along edge");
        }
    }

    private static void getVerticalTicks(@NonNull int[] content, int offset, int width, int height, boolean transparent, boolean required, @NonNull int[] divs, int top, int bottom, @Nullable AtomicInteger outDivs, boolean multipleAllowed) throws TickException {
        divs[bottom] = -1;
        divs[top] = -1;
        Tick state = Tick.START;
        boolean found = false;
        for (int i = 1; i < height - 1; ++i) {
            TickType tickType;
            try {
                tickType = PngProcessor.getTickType(content[offset + width * i], transparent);
            }
            catch (TickException e) {
                throw new TickException(e, i);
            }
            if (TickType.TICK == tickType) {
                if (state == Tick.START || state == Tick.OUTSIDE_1 && multipleAllowed) {
                    divs[top] = i - 1;
                    divs[bottom] = height - 2;
                    found = true;
                    if (outDivs != null) {
                        outDivs.addAndGet(2);
                    }
                    state = Tick.INSIDE_1;
                    continue;
                }
                if (state != Tick.OUTSIDE_1) continue;
                throw new TickException("Can't have more than one marked region along edge");
            }
            if (state != Tick.INSIDE_1) continue;
            divs[bottom] = i - 1;
            top += 2;
            bottom += 2;
            state = Tick.OUTSIDE_1;
        }
        if (required && !found) {
            throw new TickException("No marked region found along edge");
        }
    }

    private static void getHorizontalLayoutBoundsTicks(@NonNull int[] content, int offset, int width, boolean transparent, boolean required, @NonNull int[] outValues) throws TickException {
        TickType tick;
        int i;
        outValues[1] = 0;
        outValues[0] = 0;
        if (TickType.LAYOUT_BOUNDS == PngProcessor.getTickType(content[offset + 1], transparent)) {
            i = 1;
            while (i < width - 1) {
                outValues[0] = outValues[0] + 1;
                if ((tick = PngProcessor.getTickType(content[offset + ++i], transparent)) == TickType.LAYOUT_BOUNDS) continue;
                break;
            }
        }
        if (TickType.LAYOUT_BOUNDS == PngProcessor.getTickType(content[offset + (width - 2)], transparent)) {
            i = width - 2;
            while (i > 1) {
                outValues[1] = outValues[1] + 1;
                if ((tick = PngProcessor.getTickType(content[offset + --i], transparent)) == TickType.LAYOUT_BOUNDS) continue;
                break;
            }
        }
    }

    private static void getVerticalLayoutBoundsTicks(@NonNull int[] content, int offset, int width, int height, boolean transparent, boolean required, @NonNull int[] outValues) throws TickException {
        TickType tick;
        int i;
        outValues[1] = 0;
        outValues[0] = 0;
        if (TickType.LAYOUT_BOUNDS == PngProcessor.getTickType(content[offset + width], transparent)) {
            i = 1;
            while (i < height - 1) {
                outValues[0] = outValues[0] + 1;
                if ((tick = PngProcessor.getTickType(content[offset + width * ++i], transparent)) == TickType.LAYOUT_BOUNDS) continue;
                break;
            }
        }
        if (TickType.LAYOUT_BOUNDS == PngProcessor.getTickType(content[offset + width * (height - 2)], transparent)) {
            i = height - 2;
            while (i > 1) {
                outValues[1] = outValues[1] + 1;
                if ((tick = PngProcessor.getTickType(content[offset + width * --i], transparent)) == TickType.LAYOUT_BOUNDS) continue;
                break;
            }
        }
    }

    Chunk computeIhdr(int width, int height, byte bitDepth, @NonNull ColorType colorType) {
        byte[] buffer = new byte[13];
        ByteUtils utils = ByteUtils.Cache.get();
        System.arraycopy(utils.getIntAsArray(width), 0, buffer, 0, 4);
        System.arraycopy(utils.getIntAsArray(height), 0, buffer, 4, 4);
        buffer[8] = bitDepth;
        buffer[9] = colorType.getFlag();
        buffer[10] = 0;
        buffer[11] = 0;
        buffer[12] = 0;
        return new Chunk(PngWriter.IHDR, buffer);
    }

    boolean is9Patch() {
        return this.mFile.getPath().endsWith(".9.png");
    }

    private static enum Tick {
        START,
        INSIDE_1,
        OUTSIDE_1;

    }

    private static enum TickType {
        NONE,
        TICK,
        LAYOUT_BOUNDS,
        BOTH;

    }
}

