/*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.build.gradle.internal.dsl;

import com.android.annotations.Nullable;

import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.Optional;

/**
 * DSL object for configuring dx options.
 */
public class DexOptions implements com.android.builder.core.DexOptions {

    private boolean isIncrementalFlag = false;

    private boolean isPreDexLibrariesFlag = true;

    private boolean isJumboModeFlag = false;

    // By default, all dexing will happen in process to get maximum feedback quickly.
    private boolean dexInProcess = true;

    private Integer threadCount = null;

    private String javaMaxHeapSize;

    private volatile Integer maxProcessCount = null;

    public void setIncremental(boolean isIncremental) {
        // TODO: Print out a warning, that this is ignored.
        isIncrementalFlag = isIncremental;
    }

    /**
     * Ignored.
     * @deprecated
     */
    @Override
    @Input
    @Deprecated
    public boolean getIncremental() {
        return isIncrementalFlag;
    }

    public void setPreDexLibraries(boolean flag) {
        isPreDexLibrariesFlag = flag;
    }

    /**
     * Whether to pre-dex libraries. This can improve incremental builds, but clean builds may
     * be slower.
     */
    @Override
    @Input
    public boolean getPreDexLibraries() {
        return isPreDexLibrariesFlag;
    }

    public void setJumboMode(boolean flag) {
        isJumboModeFlag = flag;
    }

    /**
     * Enable jumbo mode in dx ({@code --force-jumbo}).
     */
    @Override
    @Input
    public boolean getJumboMode() {
        return isJumboModeFlag;
    }

    public void setDexInProcess(boolean dexInProcess) {
        this.dexInProcess = dexInProcess;
    }

    /**
     * Whether to run the {@code dx} compiler as a separate process or inside the Gradle daemon JVM.
     *
     * <p>Running {@code dx} in-process can greatly improve performance, but is still experimental.
     */
    @Override
    public boolean getDexInProcess() {
        return dexInProcess;
    }

    public void setJavaMaxHeapSize(String theJavaMaxHeapSize) {
        if (theJavaMaxHeapSize.matches("\\d+[kKmMgGtT]?")) {
            javaMaxHeapSize = theJavaMaxHeapSize;
        } else {
            throw new IllegalArgumentException(
                    "Invalid max heap size DexOption. See `man java` for valid -Xmx arguments.");
        }
    }

    /**
     * Specifies the {@code -Xmx} value when calling dx. Example value is {@code "2048m"}.
     */
    @Override
    @Optional @Input
    @Nullable
    public String getJavaMaxHeapSize() {
        return javaMaxHeapSize;
    }

    public void setThreadCount(int threadCount) {
        this.threadCount = threadCount;
    }

    /**
     * Number of threads to use when running dx. Defaults to 4.
     */
    @Override
    @Nullable
    public Integer getThreadCount() {
        return threadCount;
    }


    /**
     * Returns the maximum number of concurrent processes that can be used to dex. Defaults to 4.
     *
     * <p>Be aware that the number of concurrent process times the memory requirement represent the
     * minimum amount of memory that will be used by the dx processes:
     *
     * <p>{@code Total Memory = getMaxProcessCount() * getJavaMaxHeapSize()}
     *
     * <p>To avoid trashing, keep these two settings appropriate for your configuration.
     * @return the max number of concurrent dx processes.
     */
    @Nullable
    @Override
    public Integer getMaxProcessCount() {
        return maxProcessCount;
    }

    public void setMaxProcessCount(int maxProcessCount) {
        this.maxProcessCount = maxProcessCount;
    }
}
