/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.BoundStatement;
import com.datastax.driver.core.BusyConnectionException;
import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.Configuration;
import com.datastax.driver.core.Connection;
import com.datastax.driver.core.ConnectionException;
import com.datastax.driver.core.ConsistencyLevel;
import com.datastax.driver.core.DefaultResultSetFuture;
import com.datastax.driver.core.Host;
import com.datastax.driver.core.HostConnectionPool;
import com.datastax.driver.core.HostDistance;
import com.datastax.driver.core.PooledConnection;
import com.datastax.driver.core.PreparedStatement;
import com.datastax.driver.core.Query;
import com.datastax.driver.core.RequestHandler;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.ResultSetFuture;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.SimpleStatement;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.exceptions.AuthenticationException;
import com.datastax.driver.core.exceptions.DriverInternalError;
import com.datastax.driver.core.policies.LoadBalancingPolicy;
import com.datastax.driver.core.policies.ReconnectionPolicy;
import com.google.common.base.Function;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.Uninterruptibles;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.cassandra.transport.Message;
import org.apache.cassandra.transport.messages.ErrorMessage;
import org.apache.cassandra.transport.messages.ExecuteMessage;
import org.apache.cassandra.transport.messages.PrepareMessage;
import org.apache.cassandra.transport.messages.QueryMessage;
import org.apache.cassandra.transport.messages.ResultMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class SessionManager
implements Session {
    private static final Logger logger = LoggerFactory.getLogger(Session.class);
    final Cluster cluster;
    final ConcurrentMap<Host, HostConnectionPool> pools;
    final HostConnectionPool.PoolState poolsState;
    final AtomicBoolean isShutdown = new AtomicBoolean(false);

    SessionManager(Cluster cluster, Collection<Host> hosts) {
        this.cluster = cluster;
        this.pools = new ConcurrentHashMap<Host, HostConnectionPool>(hosts.size());
        this.poolsState = new HostConnectionPool.PoolState();
        for (Host host : hosts) {
            try {
                this.addOrRenewPool(host, false).get();
            }
            catch (ExecutionException e) {
                throw new DriverInternalError(e);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    @Override
    public String getLoggedKeyspace() {
        return this.poolsState.keyspace;
    }

    @Override
    public ResultSet execute(String query) {
        return this.execute(new SimpleStatement(query));
    }

    @Override
    public ResultSet execute(Query query) {
        return this.executeAsync(query).getUninterruptibly();
    }

    @Override
    public ResultSetFuture executeAsync(String query) {
        return this.executeAsync(new SimpleStatement(query));
    }

    @Override
    public ResultSetFuture executeAsync(Query query) {
        if (query instanceof Statement) {
            return this.executeQuery((Message.Request)new QueryMessage(((Statement)query).getQueryString(), ConsistencyLevel.toCassandraCL(query.getConsistencyLevel())), query);
        }
        assert (query instanceof BoundStatement) : query;
        BoundStatement bs = (BoundStatement)query;
        return this.executeQuery((Message.Request)new ExecuteMessage(bs.statement.id, Arrays.asList(bs.values), ConsistencyLevel.toCassandraCL(query.getConsistencyLevel())), query);
    }

    @Override
    public PreparedStatement prepare(String query) {
        try {
            return (PreparedStatement)Uninterruptibles.getUninterruptibly(this.prepareAsync(query));
        }
        catch (ExecutionException e) {
            throw DefaultResultSetFuture.extractCauseFromExecutionException(e);
        }
    }

    @Override
    public PreparedStatement prepare(Statement statement) {
        try {
            return (PreparedStatement)Uninterruptibles.getUninterruptibly(this.prepareAsync(statement));
        }
        catch (ExecutionException e) {
            throw DefaultResultSetFuture.extractCauseFromExecutionException(e);
        }
    }

    @Override
    public ListenableFuture<PreparedStatement> prepareAsync(String query) {
        Connection.Future future = new Connection.Future((Message.Request)new PrepareMessage(query));
        this.execute(future, Query.DEFAULT);
        return this.toPreparedStatement(query, future);
    }

    @Override
    public ListenableFuture<PreparedStatement> prepareAsync(final Statement statement) {
        ListenableFuture<PreparedStatement> prepared = this.prepareAsync(statement.getQueryString());
        return Futures.transform(prepared, (Function)new Function<PreparedStatement, PreparedStatement>(){

            public PreparedStatement apply(PreparedStatement prepared) {
                ByteBuffer routingKey = statement.getRoutingKey();
                if (routingKey != null) {
                    prepared.setRoutingKey(routingKey);
                }
                prepared.setConsistencyLevel(statement.getConsistencyLevel());
                if (statement.isTracing()) {
                    prepared.enableTracing();
                }
                prepared.setRetryPolicy(statement.getRetryPolicy());
                return prepared;
            }
        });
    }

    @Override
    public void shutdown() {
        this.shutdown(Long.MAX_VALUE, TimeUnit.MILLISECONDS);
    }

    @Override
    public boolean shutdown(long timeout, TimeUnit unit) {
        if (!this.isShutdown.compareAndSet(false, true)) {
            return true;
        }
        try {
            long start = System.nanoTime();
            boolean success = true;
            for (HostConnectionPool pool : this.pools.values()) {
                success &= pool.shutdown(timeout - Cluster.timeSince(start, unit), unit);
            }
            return success;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            return false;
        }
    }

    @Override
    public Cluster getCluster() {
        return this.cluster;
    }

    private ListenableFuture<PreparedStatement> toPreparedStatement(final String query, final Connection.Future future) {
        return Futures.transform((ListenableFuture)future, (Function)new Function<Message.Response, PreparedStatement>(){

            public PreparedStatement apply(Message.Response response) {
                switch (response.type) {
                    case RESULT: {
                        ResultMessage rm = (ResultMessage)response;
                        switch (rm.kind) {
                            case PREPARED: {
                                ResultMessage.Prepared pmsg = (ResultMessage.Prepared)rm;
                                PreparedStatement stmt = PreparedStatement.fromMessage(pmsg, SessionManager.this.cluster.getMetadata(), query, SessionManager.this.poolsState.keyspace);
                                stmt = SessionManager.this.cluster.manager.addPrepared(stmt);
                                try {
                                    SessionManager.this.prepare(stmt.getQueryString(), future.getAddress());
                                }
                                catch (InterruptedException e) {
                                    Thread.currentThread().interrupt();
                                }
                                return stmt;
                            }
                        }
                        throw new DriverInternalError(String.format("%s response received when prepared statement was expected", rm.kind));
                    }
                    case ERROR: {
                        DefaultResultSetFuture.extractCause(DefaultResultSetFuture.convertException(((ErrorMessage)response).error));
                        throw new AssertionError();
                    }
                }
                throw new DriverInternalError(String.format("%s response received when prepared statement was expected", response.type));
            }
        }, (Executor)this.executor());
    }

    Connection.Factory connectionFactory() {
        return this.cluster.manager.connectionFactory;
    }

    Configuration configuration() {
        return this.cluster.manager.configuration;
    }

    LoadBalancingPolicy loadBalancingPolicy() {
        return this.cluster.manager.loadBalancingPolicy();
    }

    ReconnectionPolicy reconnectionPolicy() {
        return this.cluster.manager.reconnectionPolicy();
    }

    ListeningExecutorService executor() {
        return this.cluster.manager.executor;
    }

    ListeningExecutorService blockingExecutor() {
        return this.cluster.manager.blockingTasksExecutor;
    }

    ListenableFuture<Boolean> addOrRenewPool(final Host host, final boolean isHostAddition) {
        final HostDistance distance = this.cluster.manager.loadBalancingPolicy().distance(host);
        if (distance == HostDistance.IGNORED) {
            return Futures.immediateFuture((Object)true);
        }
        return this.executor().submit((Callable)new Callable<Boolean>(){

            @Override
            public Boolean call() {
                logger.debug("Adding {} to list of queried hosts", (Object)host);
                try {
                    HostConnectionPool previous = SessionManager.this.pools.put(host, new HostConnectionPool(host, distance, SessionManager.this));
                    if (previous != null) {
                        previous.shutdown();
                    }
                    return true;
                }
                catch (AuthenticationException e) {
                    logger.error("Error creating pool to {} ({})", (Object)host, (Object)e.getMessage());
                    SessionManager.this.cluster.manager.signalConnectionFailure(host, new ConnectionException(e.getHost(), e.getMessage()), isHostAddition);
                    return false;
                }
                catch (ConnectionException e) {
                    logger.debug("Error creating pool to {} ({})", (Object)host, (Object)e.getMessage());
                    SessionManager.this.cluster.manager.signalConnectionFailure(host, e, isHostAddition);
                    return false;
                }
            }
        });
    }

    ListenableFuture<?> removePool(Host host) {
        final HostConnectionPool pool = (HostConnectionPool)this.pools.remove(host);
        if (pool == null) {
            return Futures.immediateFuture(null);
        }
        return this.executor().submit(new Runnable(){

            @Override
            public void run() {
                pool.shutdown();
            }
        });
    }

    void updateCreatedPools() {
        for (Host h : this.cluster.getMetadata().allHosts()) {
            HostDistance dist = this.loadBalancingPolicy().distance(h);
            HostConnectionPool pool = (HostConnectionPool)this.pools.get(h);
            if (pool == null) {
                if (dist == HostDistance.IGNORED || !h.isUp()) continue;
                this.addOrRenewPool(h, false);
                continue;
            }
            if (dist == pool.hostDistance) continue;
            if (dist == HostDistance.IGNORED) {
                this.removePool(h);
                continue;
            }
            pool.hostDistance = dist;
        }
    }

    void onDown(Host host) {
        this.removePool(host).addListener(new Runnable(){

            @Override
            public void run() {
                SessionManager.this.updateCreatedPools();
            }
        }, (Executor)MoreExecutors.sameThreadExecutor());
    }

    void onRemove(Host host) {
        this.onDown(host);
    }

    void setKeyspace(String keyspace) {
        long timeout = this.configuration().getSocketOptions().getConnectTimeoutMillis();
        try {
            ResultSetFuture future = this.executeQuery((Message.Request)new QueryMessage("use " + keyspace, ConsistencyLevel.DEFAULT_CASSANDRA_CL), Query.DEFAULT);
            Uninterruptibles.getUninterruptibly((Future)((Object)future), (long)timeout, (TimeUnit)TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            throw new DriverInternalError(String.format("No responses after %d milliseconds while setting current keyspace. This should not happen, unless you have setup a very low connection timeout.", timeout));
        }
        catch (ExecutionException e) {
            DefaultResultSetFuture.extractCauseFromExecutionException(e);
        }
    }

    void execute(RequestHandler.Callback callback, Query query) {
        new RequestHandler(this, callback, query).sendRequest();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private void prepare(String query, InetAddress toExclude) throws InterruptedException {
        Iterator i$ = this.pools.entrySet().iterator();
        while (i$.hasNext()) {
            Map.Entry entry = i$.next();
            if (((Host)entry.getKey()).getAddress().equals(toExclude)) continue;
            PooledConnection c = null;
            try {
                c = ((HostConnectionPool)entry.getValue()).borrowConnection(200L, TimeUnit.MILLISECONDS);
                c.write((Message.Request)new PrepareMessage(query)).get();
                continue;
            }
            catch (ConnectionException e) {
                continue;
            }
            catch (BusyConnectionException e) {
                continue;
            }
            catch (TimeoutException e) {
                continue;
            }
            catch (ExecutionException e) {
                logger.error(String.format("Unexpected error while preparing query (%s) on %s", query, entry.getKey()), (Throwable)e);
                continue;
            }
            finally {
                if (c == null) continue;
                c.release();
                continue;
            }
            break;
        }
        return;
    }

    ResultSetFuture executeQuery(Message.Request msg, Query query) {
        if (query.isTracing()) {
            msg.setTracingRequested();
        }
        DefaultResultSetFuture future = new DefaultResultSetFuture(this, msg);
        this.execute(future, query);
        return future;
    }
}

