/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.DataType;
import com.datastax.driver.core.TupleType;
import com.datastax.driver.core.TupleValue;
import com.datastax.driver.core.TypeCodec;
import com.datastax.driver.core.UDTValue;
import com.datastax.driver.core.UserType;
import com.datastax.driver.core.exceptions.CodecNotFoundException;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.cache.Weigher;
import com.google.common.collect.ImmutableSet;
import com.google.common.reflect.TypeToken;
import com.google.common.util.concurrent.UncheckedExecutionException;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class CodecRegistry {
    private static final Logger logger = LoggerFactory.getLogger(CodecRegistry.class);
    private static final ImmutableSet<TypeCodec<?>> PRIMITIVE_CODECS = ImmutableSet.of((Object)TypeCodec.BlobCodec.instance, (Object)TypeCodec.BooleanCodec.instance, (Object)TypeCodec.SmallIntCodec.instance, (Object)TypeCodec.TinyIntCodec.instance, (Object)TypeCodec.IntCodec.instance, (Object)TypeCodec.BigintCodec.instance, (Object[])new TypeCodec[]{TypeCodec.CounterCodec.instance, TypeCodec.DoubleCodec.instance, TypeCodec.FloatCodec.instance, TypeCodec.VarintCodec.instance, TypeCodec.DecimalCodec.instance, TypeCodec.VarcharCodec.instance, TypeCodec.AsciiCodec.instance, TypeCodec.TimestampCodec.instance, TypeCodec.DateCodec.instance, TypeCodec.TimeCodec.instance, TypeCodec.UUIDCodec.instance, TypeCodec.TimeUUIDCodec.instance, TypeCodec.InetCodec.instance});
    public static final CodecRegistry DEFAULT_INSTANCE = new CodecRegistry();
    private final CopyOnWriteArrayList<TypeCodec<?>> codecs = new CopyOnWriteArrayList((Collection<TypeCodec<?>>)PRIMITIVE_CODECS);
    private final LoadingCache<CacheKey, TypeCodec<?>> cache = CacheBuilder.newBuilder().initialCapacity(100).weigher((Weigher)new TypeCodecWeigher()).maximumWeight(1000L).concurrencyLevel(Runtime.getRuntime().availableProcessors() * 4).removalListener((RemovalListener)new TypeCodecRemovalListener()).build(new CacheLoader<CacheKey, TypeCodec<?>>(){

        public TypeCodec<?> load(CacheKey cacheKey) {
            return CodecRegistry.this.findCodec(cacheKey.cqlType, (Object)cacheKey.javaType);
        }
    });

    public CodecRegistry register(TypeCodec<?> codec) {
        return this.register(Collections.singleton(codec));
    }

    public CodecRegistry register(TypeCodec<?> ... codecs) {
        return this.register(Arrays.asList(codecs));
    }

    public CodecRegistry register(Iterable<? extends TypeCodec<?>> codecs) {
        for (TypeCodec<?> codec : codecs) {
            this.codecs.add(codec);
        }
        return this;
    }

    public <T> TypeCodec<T> codecFor(T value) {
        Preconditions.checkNotNull(value, (Object)"Parameter value cannot be null");
        TypeCodec<T> codec = this.findCodec(null, value);
        return codec;
    }

    public <T> TypeCodec<T> codecFor(DataType cqlType) throws CodecNotFoundException {
        return this.lookupCodec(cqlType, null);
    }

    public <T> TypeCodec<T> codecFor(DataType cqlType, Class<T> javaType) throws CodecNotFoundException {
        return this.codecFor(cqlType, (T)TypeToken.of(javaType));
    }

    public <T> TypeCodec<T> codecFor(DataType cqlType, TypeToken<T> javaType) throws CodecNotFoundException {
        Preconditions.checkNotNull((Object)cqlType, (Object)"Parameter cqlType cannot be null");
        Preconditions.checkNotNull(javaType, (Object)"Parameter javaType cannot be null");
        return this.lookupCodec(cqlType, javaType);
    }

    public <T> TypeCodec<T> codecFor(DataType cqlType, T value) {
        Preconditions.checkNotNull((Object)cqlType, (Object)"Parameter cqlType cannot be null");
        Preconditions.checkNotNull(value, (Object)"Parameter value cannot be null");
        return this.findCodec(cqlType, value);
    }

    private <T> TypeCodec<T> lookupCodec(DataType cqlType, TypeToken<T> javaType) {
        Preconditions.checkNotNull((Object)cqlType, (Object)"Parameter cqlType cannot be null");
        logger.trace("Querying cache for codec [{} <-> {}]", (Object)cqlType, javaType);
        CacheKey cacheKey = new CacheKey(cqlType, javaType);
        try {
            TypeCodec codec = (TypeCodec)this.cache.get((Object)cacheKey);
            logger.trace("Returning cached codec [{} <-> {}]", (Object)cqlType, javaType);
            return codec;
        }
        catch (UncheckedExecutionException e) {
            if (e.getCause() instanceof CodecNotFoundException) {
                throw (CodecNotFoundException)e.getCause();
            }
            throw new CodecNotFoundException(e.getCause(), cqlType, javaType);
        }
        catch (ExecutionException e) {
            throw new CodecNotFoundException(e.getCause(), cqlType, javaType);
        }
    }

    private <T> TypeCodec<T> findCodec(DataType cqlType, TypeToken<T> javaType) {
        Preconditions.checkNotNull((Object)cqlType, (Object)"Parameter cqlType cannot be null");
        logger.trace("Looking for codec [{} <-> {}]", cqlType == null ? "ANY" : cqlType, javaType == null ? "ANY" : javaType);
        for (TypeCodec<?> codec : this.codecs) {
            if (cqlType != null && !codec.accepts(cqlType) || javaType != null && !codec.accepts(javaType)) continue;
            logger.trace("Codec found: {}", codec);
            return codec;
        }
        return this.createCodec(cqlType, (T)javaType);
    }

    private <T> TypeCodec<T> findCodec(DataType cqlType, T value) {
        Preconditions.checkNotNull(value, (Object)"Parameter value cannot be null");
        logger.trace("Looking for codec [{} <-> {}]", cqlType == null ? "ANY" : cqlType, value.getClass());
        for (TypeCodec<?> codec : this.codecs) {
            if (cqlType != null && !codec.accepts(cqlType) || !codec.accepts(value)) continue;
            logger.trace("Codec found: {}", codec);
            return codec;
        }
        return this.createCodec(cqlType, value);
    }

    private <T> TypeCodec<T> createCodec(DataType cqlType, TypeToken<T> javaType) {
        TypeCodec<TypeToken<T>> codec = this.maybeCreateCodec(cqlType, (T)javaType);
        if (codec == null) {
            throw CodecRegistry.newException(cqlType, javaType);
        }
        if (!codec.accepts(cqlType) || javaType != null && !codec.accepts(javaType)) {
            throw CodecRegistry.newException(cqlType, javaType);
        }
        logger.trace("Codec created: {}", codec);
        return codec;
    }

    private <T> TypeCodec<T> createCodec(DataType cqlType, T value) {
        TypeCodec<T> codec = this.maybeCreateCodec(cqlType, value);
        if (codec == null) {
            throw CodecRegistry.newException(cqlType, TypeToken.of(value.getClass()));
        }
        if (cqlType != null && !codec.accepts(cqlType) || !codec.accepts(value)) {
            throw CodecRegistry.newException(cqlType, TypeToken.of(value.getClass()));
        }
        logger.trace("Codec created: {}", codec);
        return codec;
    }

    private <T> TypeCodec<T> maybeCreateCodec(DataType cqlType, TypeToken<T> javaType) {
        Preconditions.checkNotNull((Object)cqlType);
        if ((cqlType.getName() == DataType.Name.VARCHAR || cqlType.getName() == DataType.Name.TEXT) && javaType != null && Enum.class.isAssignableFrom(javaType.getRawType())) {
            return new TypeCodec.EnumStringCodec(javaType.getRawType());
        }
        if (cqlType.getName() == DataType.Name.LIST && (javaType == null || List.class.isAssignableFrom(javaType.getRawType()))) {
            TypeToken elementType = null;
            if (javaType != null && javaType.getType() instanceof ParameterizedType) {
                Type[] typeArguments = ((ParameterizedType)javaType.getType()).getActualTypeArguments();
                elementType = TypeToken.of((Type)typeArguments[0]);
            }
            TypeCodec<Object> eltCodec = this.findCodec(cqlType.getTypeArguments().get(0), (T)elementType);
            return new TypeCodec.ListCodec<Object>(eltCodec);
        }
        if (cqlType.getName() == DataType.Name.SET && (javaType == null || Set.class.isAssignableFrom(javaType.getRawType()))) {
            TypeToken elementType = null;
            if (javaType != null && javaType.getType() instanceof ParameterizedType) {
                Type[] typeArguments = ((ParameterizedType)javaType.getType()).getActualTypeArguments();
                elementType = TypeToken.of((Type)typeArguments[0]);
            }
            TypeCodec<Object> eltCodec = this.findCodec(cqlType.getTypeArguments().get(0), (T)elementType);
            return new TypeCodec.SetCodec<Object>(eltCodec);
        }
        if (cqlType.getName() == DataType.Name.MAP && (javaType == null || Map.class.isAssignableFrom(javaType.getRawType()))) {
            TypeToken keyType = null;
            TypeToken valueType = null;
            if (javaType != null && javaType.getType() instanceof ParameterizedType) {
                Type[] typeArguments = ((ParameterizedType)javaType.getType()).getActualTypeArguments();
                keyType = TypeToken.of((Type)typeArguments[0]);
                valueType = TypeToken.of((Type)typeArguments[1]);
            }
            TypeCodec<Object> keyCodec = this.findCodec(cqlType.getTypeArguments().get(0), (T)keyType);
            TypeCodec<Object> valueCodec = this.findCodec(cqlType.getTypeArguments().get(1), (T)valueType);
            return new TypeCodec.MapCodec<Object, Object>(keyCodec, valueCodec);
        }
        if (cqlType instanceof TupleType && (javaType == null || TupleValue.class.isAssignableFrom(javaType.getRawType()))) {
            return new TypeCodec.TupleCodec((TupleType)cqlType);
        }
        if (cqlType instanceof UserType && (javaType == null || UDTValue.class.isAssignableFrom(javaType.getRawType()))) {
            return new TypeCodec.UDTCodec((UserType)cqlType);
        }
        return null;
    }

    private <T> TypeCodec<T> maybeCreateCodec(DataType cqlType, T value) {
        Preconditions.checkNotNull(value);
        if ((cqlType == null || cqlType.getName() == DataType.Name.VARCHAR || cqlType.getName() == DataType.Name.TEXT) && value instanceof Enum) {
            return new TypeCodec.EnumStringCodec(value.getClass());
        }
        if ((cqlType == null || cqlType.getName() == DataType.Name.LIST) && value instanceof List) {
            List list = (List)value;
            if (list.isEmpty()) {
                DataType elementType = cqlType == null || cqlType.getTypeArguments().isEmpty() ? DataType.blob() : cqlType.getTypeArguments().get(0);
                return new TypeCodec.ListCodec<TypeToken>(this.findCodec(elementType, (T)null));
            }
            DataType elementType = cqlType == null || cqlType.getTypeArguments().isEmpty() ? null : cqlType.getTypeArguments().get(0);
            return new TypeCodec.ListCodec(this.findCodec(elementType, list.iterator().next()));
        }
        if ((cqlType == null || cqlType.getName() == DataType.Name.SET) && value instanceof Set) {
            Set set = (Set)value;
            if (set.isEmpty()) {
                DataType elementType = cqlType == null || cqlType.getTypeArguments().isEmpty() ? DataType.blob() : cqlType.getTypeArguments().get(0);
                return new TypeCodec.SetCodec<TypeToken>(this.findCodec(elementType, (T)null));
            }
            DataType elementType = cqlType == null || cqlType.getTypeArguments().isEmpty() ? null : cqlType.getTypeArguments().get(0);
            return new TypeCodec.SetCodec(this.findCodec(elementType, set.iterator().next()));
        }
        if ((cqlType == null || cqlType.getName() == DataType.Name.MAP) && value instanceof Map) {
            Map map = (Map)value;
            if (map.isEmpty()) {
                DataType keyType = cqlType == null || cqlType.getTypeArguments().size() < 1 ? DataType.blob() : cqlType.getTypeArguments().get(0);
                DataType valueType = cqlType == null || cqlType.getTypeArguments().size() < 2 ? DataType.blob() : cqlType.getTypeArguments().get(1);
                return new TypeCodec.MapCodec<TypeToken, TypeToken>(this.findCodec(keyType, (T)null), this.findCodec(valueType, (T)null));
            }
            DataType keyType = cqlType == null || cqlType.getTypeArguments().size() < 1 ? null : cqlType.getTypeArguments().get(0);
            DataType valueType = cqlType == null || cqlType.getTypeArguments().size() < 2 ? null : cqlType.getTypeArguments().get(1);
            Map.Entry entry = map.entrySet().iterator().next();
            return new TypeCodec.MapCodec(this.findCodec(keyType, entry.getKey()), this.findCodec(valueType, entry.getValue()));
        }
        if ((cqlType == null || cqlType.getName() == DataType.Name.TUPLE) && value instanceof TupleValue) {
            return new TypeCodec.TupleCodec(cqlType == null ? ((TupleValue)value).getType() : (TupleType)cqlType);
        }
        if ((cqlType == null || cqlType.getName() == DataType.Name.UDT) && value instanceof UDTValue) {
            return new TypeCodec.UDTCodec(cqlType == null ? ((UDTValue)value).getType() : (UserType)cqlType);
        }
        return null;
    }

    private static CodecNotFoundException newException(DataType cqlType, TypeToken<?> javaType) {
        String msg = String.format("Codec not found for requested operation: [%s <-> %s]", cqlType == null ? "ANY" : cqlType, javaType == null ? "ANY" : javaType);
        return new CodecNotFoundException(msg, cqlType, javaType);
    }

    private class TypeCodecRemovalListener
    implements RemovalListener<CacheKey, TypeCodec<?>> {
        private TypeCodecRemovalListener() {
        }

        public void onRemoval(RemovalNotification<CacheKey, TypeCodec<?>> notification) {
            logger.trace("Evicting codec from cache: {} (cause: {})", notification.getValue(), (Object)notification.getCause());
        }
    }

    private class TypeCodecWeigher
    implements Weigher<CacheKey, TypeCodec<?>> {
        private TypeCodecWeigher() {
        }

        public int weigh(CacheKey key, TypeCodec<?> value) {
            return CodecRegistry.this.codecs.contains(value) ? 0 : this.weigh(key.cqlType, 0);
        }

        private int weigh(DataType cqlType, int level) {
            switch (cqlType.getName()) {
                case LIST: 
                case SET: 
                case MAP: {
                    int weight = level;
                    for (DataType eltType : cqlType.getTypeArguments()) {
                        weight += this.weigh(eltType, level + 1);
                    }
                    return weight;
                }
                case UDT: {
                    int weight = level;
                    for (UserType.Field field : (UserType)cqlType) {
                        weight += this.weigh(field.getType(), level + 1);
                    }
                    return weight == 0 ? 1 : weight;
                }
                case TUPLE: {
                    int weight = level;
                    for (DataType componentType : ((TupleType)cqlType).getComponentTypes()) {
                        weight += this.weigh(componentType, level + 1);
                    }
                    return weight == 0 ? 1 : weight;
                }
                case CUSTOM: {
                    return 1;
                }
            }
            return 0;
        }
    }

    private static final class CacheKey {
        private final DataType cqlType;
        private final TypeToken<?> javaType;

        public CacheKey(DataType cqlType, TypeToken<?> javaType) {
            this.javaType = javaType;
            this.cqlType = cqlType;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CacheKey cacheKey = (CacheKey)o;
            return Objects.equal((Object)this.cqlType, (Object)cacheKey.cqlType) && Objects.equal(this.javaType, cacheKey.javaType);
        }

        public int hashCode() {
            return Objects.hashCode((Object[])new Object[]{this.javaType, this.cqlType});
        }
    }
}

