/*
 * Decompiled with CFR 0.152.
 */
package com.github.adejanovski.cassandra.jdbc;

import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.SocketOptions;
import com.datastax.driver.core.exceptions.DriverException;
import com.datastax.driver.core.policies.LoadBalancingPolicy;
import com.datastax.driver.core.policies.RoundRobinPolicy;
import com.datastax.driver.core.policies.TokenAwarePolicy;
import com.github.adejanovski.cassandra.jdbc.Utils;
import com.google.common.cache.LoadingCache;
import java.sql.SQLException;
import java.sql.SQLNonTransientConnectionException;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class SessionHolder {
    private static final Logger logger = LoggerFactory.getLogger(SessionHolder.class);
    static final String URL_KEY = "url";
    private final LoadingCache<Map<String, String>, SessionHolder> parentCache;
    private final Map<String, String> cacheKey;
    private final AtomicInteger references = new AtomicInteger();
    final Session session;
    final Properties properties;

    SessionHolder(Map<String, String> params, LoadingCache<Map<String, String>, SessionHolder> parentCache) throws SQLException {
        this.cacheKey = params;
        this.parentCache = parentCache;
        String url = params.get(URL_KEY);
        this.properties = Utils.parseURL(url.replace("\"", "'"));
        for (String key : params.keySet()) {
            if (URL_KEY.equals(key)) continue;
            this.properties.put(key, params.get(key));
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Final Properties to Connection: {}", (Object)this.properties);
        }
        this.session = this.createSession(this.properties);
    }

    void release() {
        int newRef;
        int ref;
        while (!this.references.compareAndSet(ref, newRef = (ref = this.references.get()) == 1 ? -1 : ref - 1)) {
        }
        if (newRef == -1) {
            logger.debug("Released last reference to {}, closing Session", (Object)this.cacheKey.get(URL_KEY));
            this.dispose();
        } else {
            logger.debug("Released reference to {}, new count = {}", (Object)this.cacheKey.get(URL_KEY), (Object)newRef);
        }
    }

    boolean acquire() {
        int ref;
        do {
            if ((ref = this.references.get()) >= 0) continue;
            logger.debug("Failed to acquire reference to {}", (Object)this.cacheKey.get(URL_KEY));
            return false;
        } while (!this.references.compareAndSet(ref, ref + 1));
        logger.debug("Acquired reference to {}, new count = {}", (Object)this.cacheKey.get(URL_KEY), (Object)(ref + 1));
        return true;
    }

    private Session createSession(Properties properties) throws SQLException {
        String hosts = properties.getProperty("serverName");
        int port = Integer.parseInt(properties.getProperty("portNumber"));
        String keyspace = properties.getProperty("databaseName");
        String username = properties.getProperty("user", "");
        String password = properties.getProperty("password", "");
        String loadBalancingPolicy = properties.getProperty("loadBalancing", "");
        String retryPolicy = properties.getProperty("retry", "");
        String reconnectPolicy = properties.getProperty("reconnection", "");
        boolean debugMode = properties.getProperty("debug", "").equals("true");
        Cluster.Builder builder = Cluster.builder();
        builder.addContactPoints(hosts.split("--")).withPort(port);
        builder.withSocketOptions(new SocketOptions().setKeepAlive(true));
        if (username.length() > 0) {
            builder.withCredentials(username, password);
        }
        if (loadBalancingPolicy.length() > 0) {
            try {
                builder.withLoadBalancingPolicy(Utils.parseLbPolicy(loadBalancingPolicy));
            }
            catch (Exception e) {
                if (debugMode) {
                    throw new SQLNonTransientConnectionException(e);
                }
                logger.warn("Error occured while parsing load balancing policy :" + e.getMessage() + " / Forcing to TokenAwarePolicy...");
                builder.withLoadBalancingPolicy((LoadBalancingPolicy)new TokenAwarePolicy((LoadBalancingPolicy)new RoundRobinPolicy()));
            }
        }
        if (retryPolicy.length() > 0) {
            try {
                builder.withRetryPolicy(Utils.parseRetryPolicy(retryPolicy));
            }
            catch (Exception e) {
                if (debugMode) {
                    throw new SQLNonTransientConnectionException(e);
                }
                logger.warn("Error occured while parsing retry policy :" + e.getMessage() + " / skipping...");
            }
        }
        if (reconnectPolicy.length() > 0) {
            try {
                builder.withReconnectionPolicy(Utils.parseReconnectionPolicy(reconnectPolicy));
            }
            catch (Exception e) {
                if (debugMode) {
                    throw new SQLNonTransientConnectionException(e);
                }
                logger.warn("Error occured while parsing reconnection policy :" + e.getMessage() + " / skipping...");
            }
        }
        Cluster cluster = null;
        try {
            cluster = builder.build();
            return cluster.connect(keyspace);
        }
        catch (DriverException e) {
            if (cluster != null) {
                cluster.close();
            }
            throw new SQLNonTransientConnectionException(e);
        }
    }

    private void dispose() {
        this.session.getCluster().close();
        this.parentCache.invalidate(this.cacheKey);
    }
}

