/*
 * FindBugs - Find Bugs in Java programs
 * Copyright (C) 2006, University of Maryland
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package edu.umd.cs.findbugs.util;

import java.util.Collections;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;

import edu.umd.cs.findbugs.internalAnnotations.StaticConstant;

/**
 * Utility methods for working with zip/jar archives.
 *
 * @author David Hovemeyer
 */
public class Archive {

    /**
     * File extensions that indicate an archive (zip, jar, or similar).
     */
    @StaticConstant
    public static final Set<String> ARCHIVE_EXTENSION_SET;
    static {
        Set<String> extensions = new HashSet<>();
        extensions.add(".jar");
        extensions.add(".zip");
        extensions.add(".war");
        extensions.add(".ear");
        extensions.add(".sar");
        ARCHIVE_EXTENSION_SET = Collections.unmodifiableSet(extensions);
    }

    /**
     * Determine whether or not the given filename appears to identify a zip/jar
     * archive.
     *
     * @param fileName
     *            the filename
     * @return true if the filename appears to identify an archive, false
     *         otherwise
     */
    public static boolean isArchiveFileName(String fileName) {
        String extension = getExtension(fileName);
        return ARCHIVE_EXTENSION_SET.contains(extension);
    }

    private static String getExtension(String fileName) {
        int lastDot = fileName.lastIndexOf('.');
        if (lastDot < 0) {
            return fileName;
        }
        String extension = fileName.substring(lastDot).toLowerCase(Locale.ENGLISH);
        return extension;
    }

    public static boolean isLibraryFileName(String fileName) {
        String extension = getExtension(fileName);
        return ".jar".equals(extension);
    }
}
