/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.fs.gcs;

import com.google.cloud.hadoop.fs.gcs.GhfsStatistic;
import com.google.cloud.hadoop.fs.gcs.GoogleHadoopFileSystemBase;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageStatistics;
import com.google.cloud.hadoop.gcsio.StatisticTypeEnum;
import com.google.cloud.hadoop.util.ITraceFactory;
import com.google.cloud.hadoop.util.ITraceOperation;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.flogger.GoogleLogger;
import com.google.common.util.concurrent.AtomicDouble;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.Nonnull;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.fs.StorageStatistics;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public class GhfsStorageStatistics
extends StorageStatistics {
    private static final GoogleLogger logger = GoogleLogger.forEnclosingClass();
    public static final String NAME = "GhfsStorageStatistics";
    public static final int LATENCY_LOGGING_THRESHOLD_MS = 500;
    static final GhfsStorageStatistics DUMMY_INSTANCE = new GhfsStorageStatistics();
    private final Map<String, AtomicLong> opsCount = new HashMap<String, AtomicLong>();
    private final Map<String, AtomicLong> minimums = new HashMap<String, AtomicLong>();
    private final Map<String, AtomicLong> maximums = new HashMap<String, AtomicLong>();
    private final Map<String, MeanStatistic> means = new HashMap<String, MeanStatistic>();
    private final Map<String, AtomicDouble> total = new HashMap<String, AtomicDouble>();

    public GhfsStorageStatistics() {
        super(NAME);
        String symbol;
        for (GoogleCloudStorageStatistics googleCloudStorageStatistics : GoogleCloudStorageStatistics.values()) {
            symbol = googleCloudStorageStatistics.getSymbol();
            this.opsCount.put(symbol, new AtomicLong(0L));
        }
        for (GhfsStatistic ghfsStatistic : GhfsStatistic.values()) {
            symbol = ghfsStatistic.getSymbol();
            this.opsCount.put(symbol, new AtomicLong(0L));
            if (ghfsStatistic.getType() != StatisticTypeEnum.TYPE_DURATION && ghfsStatistic.getType() != StatisticTypeEnum.TYPE_DURATION_TOTAL) continue;
            this.minimums.put(this.getMinKey(symbol), null);
            this.maximums.put(this.getMaxKey(symbol), new AtomicLong(0L));
            this.means.put(this.getMeanKey(symbol), new MeanStatistic());
            if (ghfsStatistic.getType() != StatisticTypeEnum.TYPE_DURATION_TOTAL) continue;
            this.total.put(this.getTimeKey(symbol), new AtomicDouble(0.0));
        }
    }

    /*
     * Loose catch block
     */
    static <B> B trackDuration(@Nonnull GhfsStorageStatistics stats, GhfsStatistic statistic, Object context, ITraceFactory traceFactory, GoogleHadoopFileSystemBase.InvocationRaisingIOE<B> operation) throws IOException {
        Stopwatch stopwatch = Stopwatch.createStarted();
        try {
            B b;
            try (ITraceOperation op = traceFactory.createRootWithLogging(statistic.getSymbol(), context);){
                stats.increment(statistic);
                b = operation.apply();
            }
            return b;
            {
                catch (Throwable throwable) {
                    throw throwable;
                }
            }
        }
        finally {
            stats.updateStats(statistic, stopwatch.elapsed().toMillis(), context);
        }
    }

    private long increment(GhfsStatistic statistic) {
        return this.incrementCounter(statistic, 1L);
    }

    private void increment(GoogleCloudStorageStatistics statistic) {
        this.incrementCounter(statistic, 1L);
    }

    long incrementCounter(GhfsStatistic op, long count) {
        return this.opsCount.get(op.getSymbol()).addAndGet(count);
    }

    void incrementCounter(GoogleCloudStorageStatistics op, long count) {
        this.opsCount.get(op.getSymbol()).addAndGet(count);
    }

    public void reset() {
        this.resetLongMetrics(this.opsCount);
        this.resetLongMetrics(this.maximums);
        this.resetDoubleMetrics(this.total);
        for (String ms : this.means.keySet()) {
            this.means.get(ms).reset();
        }
        for (String ms : this.minimums.keySet()) {
            this.minimums.put(ms, null);
        }
    }

    private void resetLongMetrics(Map<String, AtomicLong> metrics) {
        for (AtomicLong value : metrics.values()) {
            value.set(0L);
        }
    }

    private void resetDoubleMetrics(Map<String, AtomicDouble> metrics) {
        for (AtomicDouble value : metrics.values()) {
            value.set(0.0);
        }
    }

    void updateStats(GhfsStatistic statistic, long durationMs, Object context) {
        Preconditions.checkArgument((statistic.getType() == StatisticTypeEnum.TYPE_DURATION ? 1 : 0) != 0, (Object)String.format("Unexpected instrumentation type %s", new Object[]{statistic}));
        this.updateMinMaxStats(statistic, durationMs, durationMs, context);
        this.addMeanStatistic(statistic, durationMs, 1);
    }

    private void addMeanStatistic(GhfsStatistic statistic, long totalDurationMs, int count) {
        String meanKey = this.getMeanKey(statistic.getSymbol());
        if (this.means.containsKey(meanKey)) {
            this.means.get(meanKey).addSample(totalDurationMs, count);
        }
    }

    protected void addTotalTimeStatistic(String statistic) {
        assert (statistic.contains("_duration"));
        String parentCounterKey = statistic.replace("_duration", "");
        String parentMeanKey = this.getMeanKey(parentCounterKey);
        assert (this.means.containsKey(parentMeanKey) && this.opsCount.containsKey(parentCounterKey));
        double meanValue = this.means.get(parentMeanKey).getValue();
        long operationValue = this.opsCount.get(parentCounterKey).get();
        this.total.get(statistic).set(1.0 * meanValue * (double)operationValue);
    }

    void updateStats(GhfsStatistic statistic, long minLatency, long maxLatency, long totalDuration, int count, Object context) {
        this.updateMinMaxStats(statistic, minLatency, maxLatency, context);
        this.addMeanStatistic(statistic, totalDuration, count);
        this.opsCount.get(statistic.getSymbol()).addAndGet(count);
    }

    private void updateMinMaxStats(GhfsStatistic statistic, long minDurationMs, long maxDurationMs, Object context) {
        String minKey = this.getMinKey(statistic.getSymbol());
        AtomicLong minVal = this.minimums.get(minKey);
        if (minVal == null) {
            this.minimums.put(minKey, new AtomicLong(minDurationMs));
        } else if (minDurationMs < minVal.get()) {
            minVal.set(minDurationMs);
        }
        String maxKey = this.getMaxKey(statistic.getSymbol());
        AtomicLong maxVal = this.maximums.get(maxKey);
        if (maxDurationMs > maxVal.get()) {
            if (maxDurationMs > 500L) {
                ((GoogleLogger.Api)logger.atInfo()).log("Detected potential high latency for operation %s. latencyMs=%s; previousMaxLatencyMs=%s; operationCount=%s; context=%s", (Object)statistic, (Object)maxDurationMs, (Object)maxVal.get(), (Object)this.opsCount.get(statistic.getSymbol()), context);
            }
            maxVal.set(maxDurationMs);
        }
    }

    void streamReadBytes(int bytesRead) {
        this.incrementCounter(GhfsStatistic.STREAM_READ_BYTES, (long)bytesRead);
    }

    void streamReadOperationInComplete(int requested, int actual) {
        if (requested > actual) {
            this.increment(GhfsStatistic.STREAM_READ_OPERATIONS_INCOMPLETE);
        }
    }

    void streamReadSeekBackward(long negativeOffset) {
        this.increment(GhfsStatistic.STREAM_READ_SEEK_BACKWARD_OPERATIONS);
        this.incrementCounter(GhfsStatistic.STREAM_READ_SEEK_BYTES_BACKWARDS, -negativeOffset);
    }

    void streamReadSeekForward(long skipped) {
        if (skipped > 0L) {
            this.incrementCounter(GhfsStatistic.STREAM_READ_SEEK_BYTES_SKIPPED, skipped);
        }
        this.increment(GhfsStatistic.STREAM_READ_SEEK_FORWARD_OPERATIONS);
    }

    void streamWriteBytes(int bytesWritten) {
        this.incrementCounter(GhfsStatistic.STREAM_WRITE_BYTES, (long)bytesWritten);
    }

    void filesCreated() {
        this.increment(GhfsStatistic.FILES_CREATED);
    }

    void getFileCheckSum() {
        this.increment(GhfsStatistic.INVOCATION_GET_FILE_CHECKSUM);
    }

    void incrementGcsExceptionCount() {
        this.increment(GoogleCloudStorageStatistics.EXCEPTION_COUNT);
    }

    void incrementGcsTotalRequestCount() {
        this.increment(GoogleCloudStorageStatistics.GCS_API_REQUEST_COUNT);
    }

    void incrementRateLimitingCounter() {
        this.increment(GoogleCloudStorageStatistics.GCS_API_CLIENT_RATE_LIMIT_COUNT);
    }

    void incrementGcsClientSideCounter() {
        this.increment(GoogleCloudStorageStatistics.GCS_API_CLIENT_SIDE_ERROR_COUNT);
    }

    void incrementGcsServerSideCounter() {
        this.increment(GoogleCloudStorageStatistics.GCS_API_SERVER_SIDE_ERROR_COUNT);
    }

    void incrementGcsClientBadRequestCount() {
        this.increment(GoogleCloudStorageStatistics.GCS_API_CLIENT_BAD_REQUEST_COUNT);
    }

    void incrementGcsClientUnauthorizedResponseCount() {
        this.increment(GoogleCloudStorageStatistics.GCS_API_CLIENT_UNAUTHORIZED_RESPONSE_COUNT);
    }

    void incrementGcsClientNotFoundResponseCount() {
        this.increment(GoogleCloudStorageStatistics.GCS_API_CLIENT_NOT_FOUND_RESPONSE_COUNT);
    }

    void incrementGcsClientRequestTimeoutCount() {
        this.increment(GoogleCloudStorageStatistics.GCS_API_CLIENT_REQUEST_TIMEOUT_COUNT);
    }

    void incrementGcsClientGoneResponseCount() {
        this.increment(GoogleCloudStorageStatistics.GCS_API_CLIENT_GONE_RESPONSE_COUNT);
    }

    void incrementGcsClientPreconditionFailedResponseCount() {
        this.increment(GoogleCloudStorageStatistics.GCS_API_CLIENT_PRECONDITION_FAILED_RESPONSE_COUNT);
    }

    void incrementGcsClientRequestedRangeNotSatisfiableCount() {
        this.increment(GoogleCloudStorageStatistics.GCS_API_CLIENT_REQUESTED_RANGE_NOT_SATISFIABLE_COUNT);
    }

    void incrementGcsServerInternalErrorCount() {
        this.increment(GoogleCloudStorageStatistics.GCS_API_SERVER_INTERNAL_ERROR_COUNT);
    }

    void incrementGcsServerNotImplementedErrorCount() {
        this.increment(GoogleCloudStorageStatistics.GCS_API_SERVER_NOT_IMPLEMENTED_ERROR_COUNT);
    }

    void incrementGcsServerBadGatewayCount() {
        this.increment(GoogleCloudStorageStatistics.GCS_API_SERVER_BAD_GATEWAY_COUNT);
    }

    void incrementGcsServerServiceUnavailableCount() {
        this.increment(GoogleCloudStorageStatistics.GCS_API_SERVER_SERVICE_UNAVAILABLE_COUNT);
    }

    void incrementGcsServerTimeoutCount() {
        this.increment(GoogleCloudStorageStatistics.GCS_API_SERVER_TIMEOUT_COUNT);
    }

    private long getValue(String key) {
        if (this.opsCount.containsKey(key)) {
            return this.opsCount.get(key).longValue();
        }
        if (this.maximums.containsKey(key)) {
            return this.maximums.get(key).longValue();
        }
        if (this.minimums.containsKey(key) && this.minimums.get(key) != null) {
            return this.minimums.get(key).longValue();
        }
        if (this.means.containsKey(key)) {
            return Math.round(this.means.get(key).getValue());
        }
        if (this.total.containsKey(key)) {
            return this.total.get(key).longValue();
        }
        return 0L;
    }

    public Iterator<StorageStatistics.LongStatistic> getLongStatistics() {
        return new LongIterator();
    }

    public Long getLong(String key) {
        return this.getValue(key);
    }

    public boolean isTracked(String key) {
        return this.opsCount.containsKey(key) || this.maximums.containsKey(key) || this.minimums.containsKey(key) || this.means.containsKey(key) || this.total.containsKey(key);
    }

    public Long getMin(String symbol) {
        AtomicLong minValue = this.minimums.get(this.getMinKey(symbol));
        if (minValue == null) {
            return 0L;
        }
        return minValue.longValue();
    }

    private String getMinKey(String symbol) {
        return symbol + "_min";
    }

    private String getMaxKey(String symbol) {
        return symbol + "_max";
    }

    private String getMeanKey(String symbol) {
        return symbol + "_mean";
    }

    private String getTimeKey(String symbol) {
        return symbol + "_duration";
    }

    public Long getMax(String symbol) {
        AtomicLong maxValue = this.maximums.get(this.getMaxKey(symbol));
        if (maxValue == null) {
            return 0L;
        }
        return maxValue.longValue();
    }

    public double getMean(String key) {
        MeanStatistic val = this.means.get(this.getMeanKey(key));
        if (val == null) {
            return 0.0;
        }
        return val.getValue();
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        Iterator<StorageStatistics.LongStatistic> it = this.getLongStatistics();
        while (it.hasNext()) {
            StorageStatistics.LongStatistic statistic = it.next();
            if (sb.length() != 0) {
                sb.append(", ");
            }
            sb.append(String.format("%s=%s", statistic.getName(), statistic.getValue()));
        }
        return String.format("[%s]", sb);
    }

    static class MeanStatistic {
        private int sample;
        private long sum;

        MeanStatistic() {
        }

        synchronized void addSample(long val) {
            this.addSample(val, 1);
        }

        synchronized void addSample(long val, int count) {
            this.sample += count;
            this.sum += val;
        }

        double getValue() {
            if (this.sample == 0) {
                return 0.0;
            }
            return this.sum / (long)Math.max(1, this.sample);
        }

        public void reset() {
            this.sum = 0L;
            this.sample = 0;
        }
    }

    private class LongIterator
    implements Iterator<StorageStatistics.LongStatistic> {
        private Iterator<String> iterator = this.getMetricNames();

        private LongIterator() {
        }

        private Iterator<String> getMetricNames() {
            ArrayList metrics = new ArrayList();
            metrics.addAll(GhfsStorageStatistics.this.opsCount.keySet());
            metrics.addAll(GhfsStorageStatistics.this.minimums.keySet());
            metrics.addAll(GhfsStorageStatistics.this.maximums.keySet());
            metrics.addAll(GhfsStorageStatistics.this.means.keySet());
            for (String statistic : GhfsStorageStatistics.this.total.keySet()) {
                GhfsStorageStatistics.this.addTotalTimeStatistic(statistic);
            }
            metrics.addAll(GhfsStorageStatistics.this.total.keySet());
            return metrics.iterator();
        }

        @Override
        public boolean hasNext() {
            return this.iterator.hasNext();
        }

        @Override
        public StorageStatistics.LongStatistic next() {
            if (!this.iterator.hasNext()) {
                throw new NoSuchElementException();
            }
            String entry = this.iterator.next();
            return new StorageStatistics.LongStatistic(entry, GhfsStorageStatistics.this.getValue(entry));
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }
}

