/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.sql.core;

import com.google.api.client.googleapis.json.GoogleJsonError;
import com.google.api.client.googleapis.json.GoogleJsonResponseException;
import com.google.api.services.sqladmin.SQLAdmin;
import com.google.api.services.sqladmin.model.ConnectSettings;
import com.google.api.services.sqladmin.model.GenerateEphemeralCertRequest;
import com.google.api.services.sqladmin.model.GenerateEphemeralCertResponse;
import com.google.api.services.sqladmin.model.IpMapping;
import com.google.auth.oauth2.AccessToken;
import com.google.cloud.sql.AuthType;
import com.google.cloud.sql.core.AccessTokenSupplier;
import com.google.cloud.sql.core.CloudSqlInstanceName;
import com.google.cloud.sql.core.DefaultAccessTokenSupplier;
import com.google.cloud.sql.core.InstanceData;
import com.google.cloud.sql.core.InstanceDataSupplier;
import com.google.cloud.sql.core.Metadata;
import com.google.cloud.sql.core.SslData;
import com.google.common.base.CharMatcher;
import com.google.common.io.BaseEncoding;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningScheduledExecutorService;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Date;
import java.util.HashMap;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.logging.Logger;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;

class SqlAdminApiFetcher
implements InstanceDataSupplier {
    private static final Logger logger = Logger.getLogger(SqlAdminApiFetcher.class.getName());
    private final SQLAdmin apiClient;

    SqlAdminApiFetcher(SQLAdmin apiClient) {
        this.apiClient = apiClient;
    }

    private void checkDatabaseCompatibility(ConnectSettings instanceMetadata, AuthType authType, String connectionName) {
        if (authType == AuthType.IAM && instanceMetadata.getDatabaseVersion().contains("SQLSERVER")) {
            throw new IllegalArgumentException(String.format("[%s] IAM Authentication is not supported for SQL Server instances.", connectionName));
        }
    }

    private Certificate createCertificate(String cert) throws CertificateException {
        byte[] certBytes = cert.getBytes(StandardCharsets.UTF_8);
        ByteArrayInputStream certStream = new ByteArrayInputStream(certBytes);
        return CertificateFactory.getInstance("X.509").generateCertificate(certStream);
    }

    private String generatePublicKeyCert(KeyPair keyPair) {
        return "-----BEGIN RSA PUBLIC KEY-----\n" + BaseEncoding.base64().withSeparator("\n", 64).encode(keyPair.getPublic().getEncoded()) + "\n-----END RSA PUBLIC KEY-----\n";
    }

    @Override
    public InstanceData getInstanceData(CloudSqlInstanceName instanceName, AccessTokenSupplier accessTokenSupplier, AuthType authType, ListeningScheduledExecutorService executor, ListenableFuture<KeyPair> keyPair) throws ExecutionException, InterruptedException {
        ListenableFuture token = executor.submit(accessTokenSupplier::get);
        ListenableFuture metadataFuture = executor.submit(() -> this.fetchMetadata(instanceName, authType));
        ListenableFuture ephemeralCertificateFuture = Futures.whenAllComplete((ListenableFuture[])new ListenableFuture[]{keyPair, token}).call(() -> this.fetchEphemeralCertificate((KeyPair)Futures.getDone((Future)keyPair), instanceName, (Optional)Futures.getDone((Future)token), authType), (Executor)executor);
        ListenableFuture sslContextFuture = Futures.whenAllComplete((ListenableFuture[])new ListenableFuture[]{metadataFuture, ephemeralCertificateFuture}).call(() -> this.createSslData((KeyPair)Futures.getDone((Future)keyPair), (Metadata)Futures.getDone((Future)metadataFuture), (Certificate)Futures.getDone((Future)ephemeralCertificateFuture), instanceName, authType), (Executor)executor);
        ListenableFuture done = Futures.whenAllComplete((ListenableFuture[])new ListenableFuture[]{metadataFuture, ephemeralCertificateFuture, sslContextFuture}).call(() -> {
            Certificate ephemeralCertificate = (Certificate)Futures.getDone((Future)ephemeralCertificateFuture);
            X509Certificate x509Certificate = (X509Certificate)ephemeralCertificate;
            Date expiration = x509Certificate.getNotAfter();
            if (authType == AuthType.IAM) {
                expiration = DefaultAccessTokenSupplier.getTokenExpirationTime((Optional)Futures.getDone((Future)token)).filter(tokenExpiration -> x509Certificate.getNotAfter().after((Date)tokenExpiration)).orElse(x509Certificate.getNotAfter());
            }
            logger.fine(String.format("[%s] INSTANCE DATA DONE", instanceName));
            return new InstanceData((Metadata)Futures.getDone((Future)metadataFuture), (SslData)Futures.getDone((Future)sslContextFuture), expiration);
        }, (Executor)executor);
        InstanceData instanceData = (InstanceData)done.get();
        logger.fine(String.format("[%s] ALL FUTURES DONE", instanceName));
        return instanceData;
    }

    String getApplicationName() {
        return this.apiClient.getApplicationName();
    }

    private Metadata fetchMetadata(CloudSqlInstanceName instanceName, AuthType authType) {
        try {
            ConnectSettings instanceMetadata = (ConnectSettings)this.apiClient.connect().get(instanceName.getProjectId(), instanceName.getInstanceId()).execute();
            if (!instanceMetadata.getRegion().equals(instanceName.getRegionId())) {
                throw new IllegalArgumentException(String.format("[%s] The region specified for the Cloud SQL instance is incorrect. Please verify the instance connection name.", instanceName.getConnectionName()));
            }
            if (!instanceMetadata.getBackendType().equals("SECOND_GEN")) {
                throw new IllegalArgumentException(String.format("[%s] Connections to Cloud SQL instance not supported - not a Second Generation instance.", instanceName.getConnectionName()));
            }
            this.checkDatabaseCompatibility(instanceMetadata, authType, instanceName.getConnectionName());
            HashMap<String, String> ipAddrs = new HashMap<String, String>();
            if (instanceMetadata.getIpAddresses() != null) {
                for (IpMapping addr : instanceMetadata.getIpAddresses()) {
                    ipAddrs.put(addr.getType(), addr.getIpAddress());
                }
            }
            if (instanceMetadata.getDnsName() != null && !instanceMetadata.getDnsName().isEmpty()) {
                ipAddrs.put("PSC", instanceMetadata.getDnsName());
            }
            if (ipAddrs.isEmpty()) {
                throw new IllegalStateException(String.format("[%s] Unable to connect to Cloud SQL instance: instance does not have an assigned IP address.", instanceName.getConnectionName()));
            }
            try {
                Certificate instanceCaCertificate = this.createCertificate(instanceMetadata.getServerCaCert().getCert());
                logger.fine(String.format("[%s] METADATA DONE", instanceName));
                return new Metadata(ipAddrs, instanceCaCertificate);
            }
            catch (CertificateException ex) {
                throw new RuntimeException(String.format("[%s] Unable to parse the server CA certificate for the Cloud SQL instance.", instanceName.getConnectionName()), ex);
            }
        }
        catch (IOException ex) {
            throw this.addExceptionContext(ex, String.format("[%s] Failed to update metadata for Cloud SQL instance.", instanceName.getConnectionName()), instanceName);
        }
    }

    private Certificate fetchEphemeralCertificate(KeyPair keyPair, CloudSqlInstanceName instanceName, Optional<AccessToken> accessTokenOptional, AuthType authType) {
        Certificate ephemeralCertificate;
        GenerateEphemeralCertResponse response;
        GenerateEphemeralCertRequest request = new GenerateEphemeralCertRequest().setPublicKey(this.generatePublicKeyCert(keyPair));
        if (authType == AuthType.IAM && accessTokenOptional.isPresent()) {
            AccessToken accessToken = accessTokenOptional.get();
            String token = accessToken.getTokenValue();
            request.setAccessToken(CharMatcher.is((char)'.').trimTrailingFrom((CharSequence)token));
        }
        try {
            response = (GenerateEphemeralCertResponse)this.apiClient.connect().generateEphemeralCert(instanceName.getProjectId(), instanceName.getInstanceId(), request).execute();
        }
        catch (IOException ex) {
            throw this.addExceptionContext(ex, String.format("[%s] Failed to create ephemeral certificate for the Cloud SQL instance.", instanceName.getConnectionName()), instanceName);
        }
        try {
            ephemeralCertificate = this.createCertificate(response.getEphemeralCert().getCert());
        }
        catch (CertificateException ex) {
            throw new RuntimeException(String.format("[%s] Unable to parse the ephemeral certificate for the Cloud SQL instance.", instanceName.getConnectionName()), ex);
        }
        logger.fine(String.format("[%s %d] CERT DONE", instanceName, Thread.currentThread().getId()));
        return ephemeralCertificate;
    }

    private SslData createSslData(KeyPair keyPair, Metadata metadata, Certificate ephemeralCertificate, CloudSqlInstanceName instanceName, AuthType authType) {
        try {
            SSLContext sslContext;
            KeyStore authKeyStore = KeyStore.getInstance(KeyStore.getDefaultType());
            authKeyStore.load(null, null);
            KeyStore.PrivateKeyEntry privateKey = new KeyStore.PrivateKeyEntry(keyPair.getPrivate(), new Certificate[]{ephemeralCertificate});
            authKeyStore.setEntry("ephemeral", privateKey, new KeyStore.PasswordProtection(new char[0]));
            KeyManagerFactory kmf = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
            kmf.init(authKeyStore, new char[0]);
            KeyStore trustedKeyStore = KeyStore.getInstance(KeyStore.getDefaultType());
            trustedKeyStore.load(null, null);
            trustedKeyStore.setCertificateEntry("instance", metadata.getInstanceCaCertificate());
            TrustManagerFactory tmf = TrustManagerFactory.getInstance("X.509");
            tmf.init(trustedKeyStore);
            try {
                sslContext = SSLContext.getInstance("TLSv1.3");
            }
            catch (NoSuchAlgorithmException ex) {
                if (authType == AuthType.IAM) {
                    throw new RuntimeException(String.format("[%s] Unable to create a SSLContext for the Cloud SQL instance.", instanceName.getConnectionName()) + " TLSv1.3 is not supported for your Java version and is required to connect using IAM authentication", ex);
                }
                logger.warning("TLSv1.3 is not supported for your Java version, fallback to TLSv1.2");
                sslContext = SSLContext.getInstance("TLSv1.2");
            }
            sslContext.init(kmf.getKeyManagers(), tmf.getTrustManagers(), new SecureRandom());
            logger.fine(String.format("[%s %d] SSL CONTEXT", instanceName, Thread.currentThread().getId()));
            return new SslData(sslContext, kmf, tmf);
        }
        catch (IOException | GeneralSecurityException ex) {
            throw new RuntimeException(String.format("[%s] Unable to create a SSLContext for the Cloud SQL instance.", instanceName.getConnectionName()), ex);
        }
    }

    private RuntimeException addExceptionContext(IOException ex, String fallbackDesc, CloudSqlInstanceName instanceName) {
        GoogleJsonResponseException gjrEx;
        GoogleJsonResponseException googleJsonResponseException = gjrEx = ex instanceof GoogleJsonResponseException ? (GoogleJsonResponseException)ex : null;
        if (gjrEx == null || gjrEx.getDetails() == null || gjrEx.getDetails().getErrors() == null || gjrEx.getDetails().getErrors().isEmpty()) {
            return new RuntimeException(fallbackDesc, ex);
        }
        String reason = ((GoogleJsonError.ErrorInfo)gjrEx.getDetails().getErrors().get(0)).getReason();
        if ("accessNotConfigured".equals(reason)) {
            String apiLink = "https://console.cloud.google.com/apis/api/sqladmin/overview?project=" + instanceName.getProjectId();
            return new RuntimeException(String.format("[%s] The Google Cloud SQL Admin API is not enabled for the project \"%s\". Please use the Google Developers Console to enable it: %s", instanceName.getConnectionName(), instanceName.getProjectId(), apiLink), ex);
        }
        if ("notAuthorized".equals(reason)) {
            return new RuntimeException(String.format("[%s] The Cloud SQL Instance does not exist or your account is not authorized to access it. Please verify the instance connection name and check the IAM permissions for project \"%s\" ", instanceName.getConnectionName(), instanceName.getProjectId()), ex);
        }
        return new RuntimeException(fallbackDesc, ex);
    }
}

