/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.sql.core;

import com.google.auth.oauth2.AccessToken;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.cloud.sql.CredentialFactory;
import com.google.cloud.sql.core.AccessTokenSupplier;
import com.google.cloud.sql.core.RetryingCallable;
import java.io.IOException;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.Optional;
import java.util.logging.Logger;

class DefaultAccessTokenSupplier
implements AccessTokenSupplier {
    private static final Logger logger = Logger.getLogger(DefaultAccessTokenSupplier.class.getName());
    private static final String SQL_LOGIN_SCOPE = "https://www.googleapis.com/auth/sqlservice.login";
    private final CredentialFactory credentialFactory;
    private final int retryCount;
    private final Duration retryDuration;

    DefaultAccessTokenSupplier(CredentialFactory tokenSource) {
        this(tokenSource, 3, Duration.ofSeconds(3L));
    }

    DefaultAccessTokenSupplier(CredentialFactory tokenSource, int retryCount, Duration retryDuration) {
        this.credentialFactory = tokenSource;
        this.retryCount = retryCount;
        this.retryDuration = retryDuration;
    }

    @Override
    public Optional<AccessToken> get() throws IOException {
        if (this.credentialFactory == null) {
            return Optional.empty();
        }
        RetryingCallable<Optional> retries = new RetryingCallable<Optional>(() -> {
            GoogleCredentials credentials = this.credentialFactory.getCredentials();
            try {
                credentials.refreshIfExpired();
            }
            catch (IllegalStateException e) {
                throw new IllegalStateException("Error refreshing credentials " + credentials, e);
            }
            if (credentials.getAccessToken() == null || "".equals(credentials.getAccessToken().getTokenValue())) {
                String errorMessage = "Access Token has length of zero";
                logger.warning(errorMessage);
                throw new IllegalStateException(errorMessage);
            }
            this.validateAccessTokenExpiration(credentials.getAccessToken());
            GoogleCredentials downscoped = DefaultAccessTokenSupplier.getDownscopedCredentials(credentials);
            if (downscoped.getAccessToken() == null || "".equals(downscoped.getAccessToken().getTokenValue())) {
                try {
                    downscoped.refreshIfExpired();
                }
                catch (Exception e) {
                    throw new IllegalStateException("Error refreshing downscoped credentials " + credentials, e);
                }
                if (downscoped.getAccessToken() == null || "".equals(downscoped.getAccessToken().getTokenValue())) {
                    String errorMessage = "Downscoped access token has length of zero";
                    logger.warning(errorMessage);
                    throw new IllegalStateException(errorMessage + ": " + downscoped.getClass().getName() + " from " + credentials.getClass().getName());
                }
                this.validateAccessTokenExpiration(downscoped.getAccessToken());
            }
            return Optional.of(downscoped.getAccessToken());
        }, this.retryCount, this.retryDuration);
        try {
            return retries.call();
        }
        catch (IOException | RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new RuntimeException("Unexpected exception refreshing authentication token", e);
        }
    }

    private void validateAccessTokenExpiration(AccessToken accessToken) {
        Instant now;
        Instant expirationTime;
        Date expirationTimeDate = accessToken.getExpirationTime();
        if (expirationTimeDate != null && ((expirationTime = expirationTimeDate.toInstant()).isBefore(now = Instant.now()) || expirationTime.equals(now))) {
            DateTimeFormatter formatter = DateTimeFormatter.ISO_INSTANT.withZone(ZoneId.of("UTC"));
            String nowFormat = formatter.format(now);
            String expirationFormat = formatter.format(expirationTime);
            String errorMessage = "Access Token expiration time is in the past. Now = " + nowFormat + " Expiration = " + expirationFormat;
            logger.warning(errorMessage);
            throw new IllegalStateException(errorMessage);
        }
    }

    static Optional<Instant> getTokenExpirationTime(Optional<AccessToken> token) {
        return token.flatMap(at -> Optional.ofNullable(at.getExpirationTime())).map(Date::toInstant);
    }

    static GoogleCredentials getDownscopedCredentials(GoogleCredentials credentials) {
        return credentials.createScoped(new String[]{SQL_LOGIN_SCOPE});
    }
}

