/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.sql.core;

import com.google.api.client.googleapis.json.GoogleJsonResponseException;
import com.google.api.services.sqladmin.SQLAdmin;
import com.google.api.services.sqladmin.model.ConnectSettings;
import com.google.api.services.sqladmin.model.GenerateEphemeralCertRequest;
import com.google.api.services.sqladmin.model.GenerateEphemeralCertResponse;
import com.google.api.services.sqladmin.model.IpMapping;
import com.google.auth.oauth2.AccessToken;
import com.google.cloud.sql.AuthType;
import com.google.cloud.sql.IpType;
import com.google.cloud.sql.core.AccessTokenSupplier;
import com.google.cloud.sql.core.CloudSqlInstanceName;
import com.google.cloud.sql.core.ConnectionInfo;
import com.google.cloud.sql.core.ConnectionInfoRepository;
import com.google.cloud.sql.core.DefaultAccessTokenSupplier;
import com.google.cloud.sql.core.InstanceCheckingTrustManagerFactory;
import com.google.cloud.sql.core.InstanceMetadata;
import com.google.cloud.sql.core.SslData;
import com.google.cloud.sql.core.TerminalException;
import com.google.common.io.BaseEncoding;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningScheduledExecutorService;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.time.Instant;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DefaultConnectionInfoRepository
implements ConnectionInfoRepository {
    private static final String USER_PROJECT_HEADER_NAME = "X-Goog-User-Project";
    private static final Logger logger = LoggerFactory.getLogger(DefaultConnectionInfoRepository.class);
    private final SQLAdmin apiClient;
    private static final List<Integer> TERMINAL_STATUS_CODES = Arrays.asList(400, 401, 403, 404);

    DefaultConnectionInfoRepository(SQLAdmin apiClient) {
        this.apiClient = apiClient;
    }

    private void checkDatabaseCompatibility(ConnectSettings instanceMetadata, AuthType authType, String connectionName) {
        if (authType == AuthType.IAM && instanceMetadata.getDatabaseVersion().contains("SQLSERVER")) {
            throw new IllegalArgumentException(String.format("[%s] IAM Authentication is not supported for SQL Server instances.", connectionName));
        }
    }

    private Certificate createCertificate(String cert) throws CertificateException {
        byte[] certBytes = cert.getBytes(StandardCharsets.UTF_8);
        ByteArrayInputStream certStream = new ByteArrayInputStream(certBytes);
        return CertificateFactory.getInstance("X.509").generateCertificate(certStream);
    }

    private String generatePublicKeyCert(KeyPair keyPair) {
        return "-----BEGIN RSA PUBLIC KEY-----\n" + BaseEncoding.base64().withSeparator("\n", 64).encode(keyPair.getPublic().getEncoded()) + "\n-----END RSA PUBLIC KEY-----\n";
    }

    @Override
    public ConnectionInfo getConnectionInfoSync(CloudSqlInstanceName instanceName, AccessTokenSupplier accessTokenSupplier, AuthType authType, KeyPair keyPair) {
        Optional<AccessToken> token = null;
        try {
            token = accessTokenSupplier.get();
        }
        catch (IOException e) {
            throw new RuntimeException("Unable to create IAM Auth access token", e);
        }
        InstanceMetadata metadata = this.fetchMetadata(instanceName, authType);
        Certificate ephemeralCertificate = this.fetchEphemeralCertificate(keyPair, instanceName, token, authType);
        SslData sslContext = this.createSslData(keyPair, metadata, ephemeralCertificate, instanceName, authType);
        return DefaultConnectionInfoRepository.createConnectionInfo(instanceName, authType, token, metadata, ephemeralCertificate, sslContext);
    }

    @Override
    public ListenableFuture<ConnectionInfo> getConnectionInfo(CloudSqlInstanceName instanceName, AccessTokenSupplier accessTokenSupplier, AuthType authType, ListeningScheduledExecutorService executor, ListenableFuture<KeyPair> keyPair) {
        ListenableFuture token = executor.submit(accessTokenSupplier::get);
        ListenableFuture metadataFuture = executor.submit(() -> this.fetchMetadata(instanceName, authType));
        ListenableFuture ephemeralCertificateFuture = Futures.whenAllComplete((ListenableFuture[])new ListenableFuture[]{keyPair, token}).call(() -> this.fetchEphemeralCertificate((KeyPair)Futures.getDone((Future)keyPair), instanceName, (Optional)Futures.getDone((Future)token), authType), (Executor)executor);
        ListenableFuture sslContextFuture = Futures.whenAllComplete((ListenableFuture[])new ListenableFuture[]{metadataFuture, ephemeralCertificateFuture}).call(() -> this.createSslData((KeyPair)Futures.getDone((Future)keyPair), (InstanceMetadata)Futures.getDone((Future)metadataFuture), (Certificate)Futures.getDone((Future)ephemeralCertificateFuture), instanceName, authType), (Executor)executor);
        ListenableFuture done = Futures.whenAllComplete((ListenableFuture[])new ListenableFuture[]{metadataFuture, ephemeralCertificateFuture, sslContextFuture}).call(() -> DefaultConnectionInfoRepository.createConnectionInfo(instanceName, authType, (Optional)Futures.getDone((Future)token), (InstanceMetadata)Futures.getDone((Future)metadataFuture), (Certificate)Futures.getDone((Future)ephemeralCertificateFuture), (SslData)Futures.getDone((Future)sslContextFuture)), (Executor)executor);
        done.addListener(() -> logger.debug(String.format("[%s] ALL FUTURES DONE", instanceName)), (Executor)executor);
        return done;
    }

    private static ConnectionInfo createConnectionInfo(CloudSqlInstanceName instanceName, AuthType authType, Optional<AccessToken> token, InstanceMetadata metadata, Certificate ephemeralCertificate, SslData sslContext) {
        X509Certificate x509Certificate = (X509Certificate)ephemeralCertificate;
        Instant expiration = x509Certificate.getNotAfter().toInstant();
        if (authType == AuthType.IAM) {
            expiration = DefaultAccessTokenSupplier.getTokenExpirationTime(token).filter(tokenExpiration -> x509Certificate.getNotAfter().toInstant().isAfter((Instant)tokenExpiration)).orElse(x509Certificate.getNotAfter().toInstant());
        }
        logger.debug(String.format("[%s] INSTANCE DATA DONE", instanceName));
        return new ConnectionInfo(metadata, sslContext, expiration);
    }

    String getApplicationName() {
        return this.apiClient.getApplicationName();
    }

    String getQuotaProject(String connectionName) {
        CloudSqlInstanceName instanceName = new CloudSqlInstanceName(connectionName);
        try {
            List values = this.apiClient.connect().get(instanceName.getProjectId(), instanceName.getInstanceId()).getRequestHeaders().getHeaderStringValues(USER_PROJECT_HEADER_NAME);
            if (!values.isEmpty()) {
                return (String)values.get(0);
            }
            return null;
        }
        catch (IOException ex) {
            throw this.addExceptionContext(ex, "[%s] Failed to get Quota Project", instanceName);
        }
    }

    private InstanceMetadata fetchMetadata(CloudSqlInstanceName instanceName, AuthType authType) {
        try {
            ConnectSettings instanceMetadata = (ConnectSettings)this.apiClient.connect().get(instanceName.getProjectId(), instanceName.getInstanceId()).execute();
            if (!instanceMetadata.getRegion().equals(instanceName.getRegionId())) {
                throw new TerminalException(String.format("[%s] The region specified for the Cloud SQL instance is incorrect. Please verify the instance connection name.", instanceName.getConnectionName()));
            }
            if (!instanceMetadata.getBackendType().equals("SECOND_GEN")) {
                throw new TerminalException(String.format("[%s] Connections to Cloud SQL instance not supported - not a Second Generation instance.", instanceName.getConnectionName()));
            }
            this.checkDatabaseCompatibility(instanceMetadata, authType, instanceName.getConnectionName());
            HashMap<IpType, String> ipAddrs = new HashMap<IpType, String>();
            if (instanceMetadata.getIpAddresses() != null) {
                for (IpMapping addr : instanceMetadata.getIpAddresses()) {
                    if ("PRIVATE".equals(addr.getType())) {
                        ipAddrs.put(IpType.PRIVATE, addr.getIpAddress());
                        continue;
                    }
                    if (!"PRIMARY".equals(addr.getType())) continue;
                    ipAddrs.put(IpType.PUBLIC, addr.getIpAddress());
                }
            }
            if (instanceMetadata.getDnsName() != null && !instanceMetadata.getDnsName().isEmpty()) {
                ipAddrs.put(IpType.PSC, instanceMetadata.getDnsName());
            }
            if (ipAddrs.isEmpty()) {
                throw new TerminalException(String.format("[%s] Unable to connect to Cloud SQL instance: instance does not have an assigned IP address.", instanceName.getConnectionName()));
            }
            try {
                Certificate instanceCaCertificate = this.createCertificate(instanceMetadata.getServerCaCert().getCert());
                logger.debug(String.format("[%s] METADATA DONE", instanceName));
                return new InstanceMetadata(ipAddrs, instanceCaCertificate);
            }
            catch (CertificateException ex) {
                throw new RuntimeException(String.format("[%s] Unable to parse the server CA certificate for the Cloud SQL instance.", instanceName.getConnectionName()), ex);
            }
        }
        catch (IOException ex) {
            throw this.addExceptionContext(ex, String.format("[%s] Failed to update metadata for Cloud SQL instance.", instanceName.getConnectionName()), instanceName);
        }
    }

    private Certificate fetchEphemeralCertificate(KeyPair keyPair, CloudSqlInstanceName instanceName, Optional<AccessToken> accessTokenOptional, AuthType authType) {
        Certificate ephemeralCertificate;
        GenerateEphemeralCertResponse response;
        GenerateEphemeralCertRequest request = new GenerateEphemeralCertRequest().setPublicKey(this.generatePublicKeyCert(keyPair));
        if (authType == AuthType.IAM && accessTokenOptional.isPresent()) {
            AccessToken accessToken = accessTokenOptional.get();
            String token = accessToken.getTokenValue();
            request.setAccessToken(token);
        }
        try {
            response = (GenerateEphemeralCertResponse)this.apiClient.connect().generateEphemeralCert(instanceName.getProjectId(), instanceName.getInstanceId(), request).execute();
        }
        catch (IOException ex) {
            throw this.addExceptionContext(ex, String.format("[%s] Failed to create ephemeral certificate for the Cloud SQL instance.", instanceName.getConnectionName()), instanceName);
        }
        try {
            ephemeralCertificate = this.createCertificate(response.getEphemeralCert().getCert());
        }
        catch (CertificateException ex) {
            throw new RuntimeException(String.format("[%s] Unable to parse the ephemeral certificate for the Cloud SQL instance.", instanceName.getConnectionName()), ex);
        }
        logger.debug(String.format("[%s %d] CERT DONE", instanceName, Thread.currentThread().getId()));
        return ephemeralCertificate;
    }

    private SslData createSslData(KeyPair keyPair, InstanceMetadata instanceMetadata, Certificate ephemeralCertificate, CloudSqlInstanceName instanceName, AuthType authType) {
        try {
            SSLContext sslContext;
            KeyStore authKeyStore = KeyStore.getInstance(KeyStore.getDefaultType());
            authKeyStore.load(null, null);
            KeyStore.PrivateKeyEntry privateKey = new KeyStore.PrivateKeyEntry(keyPair.getPrivate(), new Certificate[]{ephemeralCertificate});
            authKeyStore.setEntry("ephemeral", privateKey, new KeyStore.PasswordProtection(new char[0]));
            KeyManagerFactory kmf = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
            kmf.init(authKeyStore, new char[0]);
            InstanceCheckingTrustManagerFactory tmf = InstanceCheckingTrustManagerFactory.newInstance(instanceName, instanceMetadata);
            try {
                sslContext = SSLContext.getInstance("TLSv1.3");
            }
            catch (NoSuchAlgorithmException ex) {
                if (authType == AuthType.IAM) {
                    throw new RuntimeException(String.format("[%s] Unable to create a SSLContext for the Cloud SQL instance.", instanceName.getConnectionName()) + " TLSv1.3 is not supported for your Java version and is required to connect using IAM authentication", ex);
                }
                logger.debug("TLSv1.3 is not supported for your Java version, fallback to TLSv1.2");
                sslContext = SSLContext.getInstance("TLSv1.2");
            }
            sslContext.init(kmf.getKeyManagers(), tmf.getTrustManagers(), new SecureRandom());
            logger.debug(String.format("[%s %d] SSL CONTEXT", instanceName, Thread.currentThread().getId()));
            return new SslData(sslContext, kmf, tmf);
        }
        catch (IOException | GeneralSecurityException ex) {
            throw new RuntimeException(String.format("[%s] Unable to create a SSLContext for the Cloud SQL instance.", instanceName.getConnectionName()), ex);
        }
    }

    private RuntimeException addExceptionContext(IOException ex, String fallbackDesc, CloudSqlInstanceName instanceName) {
        String reason = fallbackDesc;
        int statusCode = 0;
        if (ex instanceof GoogleJsonResponseException) {
            GoogleJsonResponseException gjrEx = (GoogleJsonResponseException)ex;
            reason = gjrEx.getMessage();
            statusCode = gjrEx.getStatusCode();
        } else if (ex instanceof UnknownHostException) {
            statusCode = 404;
            reason = String.format("Host \"%s\" not found", ex.getMessage());
        } else {
            Matcher matcher = Pattern.compile("Error code (\\d+)").matcher(ex.getMessage());
            reason = ex.getMessage();
            if (matcher.find()) {
                statusCode = Integer.parseInt(matcher.group(1));
            }
        }
        String message = String.format("[%s] The Google Cloud SQL Admin API failed for the project \"%s\". Reason: %s", instanceName.getConnectionName(), instanceName.getProjectId(), reason);
        if (TERMINAL_STATUS_CODES.contains(statusCode)) {
            return new TerminalException(message, ex);
        }
        return new RuntimeException(message, ex);
    }
}

