/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.sql.core;

import com.google.api.client.googleapis.json.GoogleJsonError;
import com.google.api.client.googleapis.json.GoogleJsonResponseException;
import com.google.api.services.sqladmin.SQLAdmin;
import com.google.api.services.sqladmin.model.DatabaseInstance;
import com.google.api.services.sqladmin.model.IpMapping;
import com.google.api.services.sqladmin.model.SslCert;
import com.google.api.services.sqladmin.model.SslCertsCreateEphemeralRequest;
import com.google.cloud.sql.core.SslData;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.io.BaseEncoding;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningScheduledExecutorService;
import com.google.common.util.concurrent.RateLimiter;
import com.google.common.util.concurrent.SettableFuture;
import com.google.common.util.concurrent.Uninterruptibles;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.SecureRandom;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.TrustManagerFactory;
import org.checkerframework.checker.nullness.compatqual.NullableDecl;

class CloudSqlInstance {
    private static final Logger logger = Logger.getLogger(CloudSqlInstance.class.getName());
    private static final Pattern CONNECTION_NAME = Pattern.compile("([^:]+(:[^:]+)?):([^:]+):([^:]+)");
    private final ListeningScheduledExecutorService executor;
    private final SQLAdmin apiClient;
    private final String connectionName;
    private final String projectId;
    private final String regionId;
    private final String instanceId;
    private final String regionalizedInstanceId;
    private final ListenableFuture<KeyPair> keyPair;
    private final Object instanceDataGuard = new Object();
    @GuardedBy(value="instanceDataGuard")
    private ListenableFuture<InstanceData> currentInstanceData;
    @GuardedBy(value="instanceDataGuard")
    private ListenableFuture<ListenableFuture<InstanceData>> nextInstanceData;
    private RateLimiter forcedRenewRateLimiter = RateLimiter.create((double)0.016666666666666666);

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    CloudSqlInstance(String connectionName, SQLAdmin apiClient, ListeningScheduledExecutorService executor, ListenableFuture<KeyPair> keyPair) {
        this.connectionName = connectionName;
        Matcher matcher = CONNECTION_NAME.matcher(connectionName);
        Preconditions.checkArgument((boolean)matcher.matches(), (Object)"[%s] Cloud SQL connection name is invalid, expected string in the form of \"<PROJECT_ID>:<REGION_ID>:<INSTANCE_ID>\".");
        this.projectId = matcher.group(1);
        this.regionId = matcher.group(3);
        this.instanceId = matcher.group(4);
        this.regionalizedInstanceId = String.format("%s~%s", this.regionId, this.instanceId);
        this.apiClient = apiClient;
        this.executor = executor;
        this.keyPair = keyPair;
        Object object = this.instanceDataGuard;
        synchronized (object) {
            this.currentInstanceData = this.performRefresh();
            this.nextInstanceData = Futures.immediateFuture(this.currentInstanceData);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private InstanceData getInstanceData() {
        ListenableFuture<InstanceData> instanceData;
        Object object = this.instanceDataGuard;
        synchronized (object) {
            instanceData = this.currentInstanceData;
        }
        try {
            return (InstanceData)Uninterruptibles.getUninterruptibly(instanceData);
        }
        catch (ExecutionException ex) {
            Throwable cause = ex.getCause();
            Throwables.throwIfUnchecked((Throwable)cause);
            throw new RuntimeException(cause);
        }
    }

    SSLSocket createSslSocket() throws IOException {
        return (SSLSocket)this.getInstanceData().getSslContext().getSocketFactory().createSocket();
    }

    String getPreferredIp(List<String> preferredTypes) {
        Map<String, String> ipAddrs = this.getInstanceData().getIpAddrs();
        for (String ipType : preferredTypes) {
            String preferredIp = ipAddrs.get(ipType);
            if (preferredIp == null) continue;
            return preferredIp;
        }
        throw new IllegalArgumentException(String.format("[%s] Cloud SQL instance  does not have any IP addresses matching preferences (%s)", this.connectionName, String.join((CharSequence)", ", preferredTypes)));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean forceRefresh() {
        if (!this.forcedRenewRateLimiter.tryAcquire()) {
            return false;
        }
        Object object = this.instanceDataGuard;
        synchronized (object) {
            if (this.nextInstanceData.cancel(false)) {
                this.currentInstanceData = this.performRefresh();
                this.nextInstanceData = Futures.immediateFuture(this.currentInstanceData);
            } else {
                this.currentInstanceData = CloudSqlInstance.blockOnNestedFuture(this.nextInstanceData, (ScheduledExecutorService)this.executor);
            }
            return true;
        }
    }

    private ListenableFuture<InstanceData> performRefresh() {
        ListenableFuture metadataFuture = this.executor.submit(this::fetchMetadata);
        ListenableFuture<Certificate> ephemeralCertificateFuture = CloudSqlInstance.whenAllSucceed(() -> this.fetchEphemeralCertificate((KeyPair)Futures.getDone(this.keyPair)), this.executor, this.keyPair);
        ListenableFuture<SslData> sslContextFuture = CloudSqlInstance.whenAllSucceed(() -> this.createSslData((KeyPair)Futures.getDone(this.keyPair), (Metadata)Futures.getDone((Future)metadataFuture), (Certificate)Futures.getDone((Future)ephemeralCertificateFuture)), this.executor, this.keyPair, metadataFuture, ephemeralCertificateFuture);
        ListenableFuture<InstanceData> refreshFuture = CloudSqlInstance.whenAllSucceed(() -> new InstanceData((Metadata)Futures.getDone((Future)metadataFuture), (SslData)Futures.getDone((Future)sslContextFuture)), this.executor, metadataFuture, sslContextFuture);
        refreshFuture.addListener(() -> {
            Object object = this.instanceDataGuard;
            synchronized (object) {
                this.currentInstanceData = refreshFuture;
                this.nextInstanceData = this.executor.schedule(this::performRefresh, 55L, TimeUnit.MINUTES);
            }
        }, (Executor)this.executor);
        return refreshFuture;
    }

    private SslData createSslData(KeyPair keyPair, Metadata metadata, Certificate ephemeralCertificate) {
        try {
            KeyStore authKeyStore = KeyStore.getInstance(KeyStore.getDefaultType());
            authKeyStore.load(null, null);
            KeyStore.PrivateKeyEntry privateKey = new KeyStore.PrivateKeyEntry(keyPair.getPrivate(), new Certificate[]{ephemeralCertificate});
            authKeyStore.setEntry("ephemeral", privateKey, new KeyStore.PasswordProtection(new char[0]));
            KeyManagerFactory kmf = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
            kmf.init(authKeyStore, new char[0]);
            KeyStore trustedKeyStore = KeyStore.getInstance(KeyStore.getDefaultType());
            trustedKeyStore.load(null, null);
            trustedKeyStore.setCertificateEntry("instance", metadata.getInstanceCaCertificate());
            TrustManagerFactory tmf = TrustManagerFactory.getInstance("X.509");
            tmf.init(trustedKeyStore);
            SSLContext sslContext = SSLContext.getInstance("TLSv1.2");
            sslContext.init(kmf.getKeyManagers(), tmf.getTrustManagers(), new SecureRandom());
            return new SslData(sslContext, kmf, tmf);
        }
        catch (IOException | GeneralSecurityException ex) {
            throw new RuntimeException(String.format("[%s] Unable to create a SSLContext for the Cloud SQL instance.", this.connectionName), ex);
        }
    }

    private Metadata fetchMetadata() {
        try {
            DatabaseInstance instanceMetadata = (DatabaseInstance)this.apiClient.instances().get(this.projectId, this.regionalizedInstanceId).execute();
            if (!instanceMetadata.getRegion().equals(this.regionId)) {
                throw new IllegalArgumentException(String.format("[%s] The region specified for the Cloud SQL instance is incorrect. Please verify the instance connection name.", this.connectionName));
            }
            if (!instanceMetadata.getBackendType().equals("SECOND_GEN")) {
                throw new IllegalArgumentException(String.format("[%s] Connections to Cloud SQL instance not supported - not a Second Generation instance.", this.connectionName));
            }
            if (instanceMetadata.getIpAddresses().isEmpty()) {
                throw new IllegalStateException(String.format("[%s] Unable to connect to Cloud SQL instance: instance does not have an assigned IP address.", this.connectionName));
            }
            HashMap<String, String> ipAddrs = new HashMap<String, String>();
            for (IpMapping addr : instanceMetadata.getIpAddresses()) {
                ipAddrs.put(addr.getType(), addr.getIpAddress());
            }
            try {
                Certificate instanceCaCertificate = CloudSqlInstance.createCertificate(instanceMetadata.getServerCaCert().getCert());
                return new Metadata(ipAddrs, instanceCaCertificate);
            }
            catch (CertificateException ex) {
                throw new RuntimeException(String.format("[%s] Unable to parse the server CA certificate for the Cloud SQL instance.", this.connectionName), ex);
            }
        }
        catch (IOException ex) {
            throw this.addExceptionContext(ex, String.format("[%s] Failed to update metadata for Cloud SQL instance.", this.connectionName));
        }
    }

    private Certificate fetchEphemeralCertificate(KeyPair keyPair) {
        Certificate ephemeralCertificate;
        SslCert response;
        SslCertsCreateEphemeralRequest request = new SslCertsCreateEphemeralRequest().setPublicKey(CloudSqlInstance.generatePublicKeyCert(keyPair));
        try {
            response = (SslCert)this.apiClient.sslCerts().createEphemeral(this.projectId, this.regionalizedInstanceId, request).execute();
        }
        catch (IOException ex) {
            throw this.addExceptionContext(ex, String.format("[%s] Failed to create ephemeral certificate for the Cloud SQL instance.", this.connectionName));
        }
        try {
            ephemeralCertificate = CloudSqlInstance.createCertificate(response.getCert());
        }
        catch (CertificateException ex) {
            throw new RuntimeException(String.format("[%s] Unable to parse the ephemeral certificate for the Cloud SQL instance.", this.connectionName), ex);
        }
        return ephemeralCertificate;
    }

    private static String generatePublicKeyCert(KeyPair keyPair) {
        return "-----BEGIN RSA PUBLIC KEY-----\n" + BaseEncoding.base64().withSeparator("\n", 64).encode(keyPair.getPublic().getEncoded()) + "\n-----END RSA PUBLIC KEY-----\n";
    }

    private static <T> ListenableFuture<T> whenAllSucceed(final Callable<T> task, final ListeningScheduledExecutorService executor, ListenableFuture<?> ... futures) {
        final SettableFuture taskFuture = SettableFuture.create();
        final AtomicInteger countDown = new AtomicInteger(futures.length);
        FutureCallback<Object> runWhenInputAreComplete = new FutureCallback<Object>(){

            public void onSuccess(@NullableDecl Object o) {
                if (countDown.decrementAndGet() == 0) {
                    taskFuture.setFuture(executor.submit(task));
                }
            }

            public void onFailure(Throwable throwable) {
                if (!taskFuture.setException(throwable)) {
                    String msg = "Got more than one input failure. Logging failures after the first";
                    logger.log(Level.SEVERE, msg, throwable);
                }
            }
        };
        for (ListenableFuture<?> future : futures) {
            Futures.addCallback(future, (FutureCallback)runWhenInputAreComplete, (Executor)executor);
        }
        return taskFuture;
    }

    private static <T> ListenableFuture<T> blockOnNestedFuture(ListenableFuture<ListenableFuture<T>> nestedFuture, ScheduledExecutorService executor) {
        final SettableFuture blockedFuture = SettableFuture.create();
        Futures.addCallback(nestedFuture, (FutureCallback)new FutureCallback<ListenableFuture<T>>(){

            public void onSuccess(ListenableFuture<T> result) {
                blockedFuture.setFuture(result);
            }

            public void onFailure(Throwable throwable) {
                blockedFuture.setException(throwable);
            }
        }, (Executor)executor);
        return blockedFuture;
    }

    private static Certificate createCertificate(String cert) throws CertificateException {
        byte[] certBytes = cert.getBytes(StandardCharsets.UTF_8);
        ByteArrayInputStream certStream = new ByteArrayInputStream(certBytes);
        return CertificateFactory.getInstance("X.509").generateCertificate(certStream);
    }

    private RuntimeException addExceptionContext(IOException ex, String fallbackDesc) {
        GoogleJsonResponseException gjrEx;
        GoogleJsonResponseException googleJsonResponseException = gjrEx = ex instanceof GoogleJsonResponseException ? (GoogleJsonResponseException)ex : null;
        if (gjrEx == null || gjrEx.getDetails() == null || gjrEx.getDetails().getErrors() == null || gjrEx.getDetails().getErrors().isEmpty()) {
            return new RuntimeException(fallbackDesc, ex);
        }
        String reason = ((GoogleJsonError.ErrorInfo)gjrEx.getDetails().getErrors().get(0)).getReason();
        if ("accessNotConfigured".equals(reason)) {
            String apiLink = "https://console.cloud.google.com/apis/api/sqladmin/overview?project=" + this.projectId;
            return new RuntimeException(String.format("[%s] The Google Cloud SQL Admin API is not enabled for the project \"%s\". Please use the Google Developers Console to enable it: %s", this.connectionName, this.projectId, apiLink), ex);
        }
        if ("notAuthorized".equals(reason)) {
            return new RuntimeException(String.format("[%s] The Cloud SQL Instance does not exist or your account is not authorized to access it. Please verify the instance connection name and check the IAM permissions for project \"%s\" ", this.connectionName, this.projectId), ex);
        }
        return new RuntimeException(fallbackDesc, ex);
    }

    SslData getSslData() {
        return this.getInstanceData().getSslData();
    }

    private static class Metadata {
        private final Map<String, String> ipAddrs;
        private final Certificate instanceCaCertificate;

        Metadata(Map<String, String> ipAddrs, Certificate instanceCaCertificate) {
            this.ipAddrs = ipAddrs;
            this.instanceCaCertificate = instanceCaCertificate;
        }

        Map<String, String> getIpAddrs() {
            return this.ipAddrs;
        }

        Certificate getInstanceCaCertificate() {
            return this.instanceCaCertificate;
        }
    }

    private static class InstanceData {
        private final Metadata metadata;
        private final SSLContext sslContext;
        private final SslData sslData;

        InstanceData(Metadata metadata, SslData sslData) {
            this.metadata = metadata;
            this.sslData = sslData;
            this.sslContext = sslData.getSslContext();
        }

        SSLContext getSslContext() {
            return this.sslContext;
        }

        Map<String, String> getIpAddrs() {
            return this.metadata.getIpAddrs();
        }

        SslData getSslData() {
            return this.sslData;
        }
    }
}

