/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.core.internal;

import joptsimple.OptionSet;
import joptsimple.OptionSpec;
import protoj.core.Command;
import protoj.core.CommandStore;
import protoj.core.CompileFeature;
import protoj.core.ProjectLayout;

/**
 * Command handler that delegates to the compile functionality. See
 * {@link CompileFeature}.
 * 
 * @author Ashley Williams
 * 
 */
public final class CompileCommand {
	private final class Body implements Runnable {
		public void run() {
			OptionSet options = delegate.getOptions();

			String includes = options.valueOf(getIncludesOption());
			if (includes == null) {
				includes = "*";
			}
			String excludes = options.valueOf(getExcludesOption());
			parent.getCompileFeature().compile(includes, excludes);
		}
	}

	/**
	 * Provides the basic command functionality.
	 */
	private Command delegate;

	/**
	 * The parent of this command.
	 */
	private final CoreProject parent;

	private OptionSpec<String> includesOption;

	private OptionSpec<String> excludesOption;

	public CompileCommand(CoreProject core) {
		this.parent = core;
		ProjectLayout layout = core.getLayout();
		CommandStore store = core.getCommandStore();

		StringBuilder description = new StringBuilder();

		description
				.append("Compiles the project source Java (.java) files. Compilation of AspectJ (.aj)");
		description
				.append("\nfiles is also supported without any additional software.");
		description.append("\n");
		description.append("\nThe following options are supported:");
		description.append("\n");
		description
				.append("\n   1. -includes: the ant-style pattern that describes the source files that are");
		description
				.append("\n      to be included in the compilation. If not specified then all source files are");
		description.append("\n      included in the compilation.");
		description
				.append("\n   2. -excludes: the ant-style pattern that describes the source files that are");
		description
				.append("\n      to be excluded in the compilation. If not specified then no source file is");
		description.append("\n      excluded in the compilation.");
		description.append("\n");
		description.append(String.format("\nExample: %s compile", layout
				.getScriptName()));
		description.append("\n");
		description.append(String.format(
				"\nExample: %s \"compile -i org/acme/util\"", layout
						.getScriptName()));
		delegate = store.addCommand("compile", description.toString(), "16m",
				new Body());
		delegate.initBootstrapCurrentVm();
		delegate.initAliases("build");
		includesOption = delegate.getParser().accepts("includesOption")
				.withRequiredArg().ofType(String.class);
		excludesOption = delegate.getParser().accepts("excludesOption")
				.withRequiredArg().ofType(String.class);
	}

	public Command getDelegate() {
		return delegate;
	}

	public OptionSpec<String> getIncludesOption() {
		return includesOption;
	}

	public OptionSpec<String> getExcludesOption() {
		return excludesOption;
	}

}
