/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.core.internal;

import java.util.List;

import joptsimple.OptionSet;
import joptsimple.OptionSpec;

import org.apache.commons.beanutils.PropertyUtils;

import protoj.core.ArgRunnable;
import protoj.core.Command;
import protoj.core.CommandStore;
import protoj.core.DispatchFeature;
import protoj.core.InstructionChain;

/**
 * Used to delegate to a new vm. Since ProtoJ already provides the ability to
 * launch a new vm then this is an ideal way of replacing an os specific command
 * in the startup script for a project.
 * <p>
 * This class is tightly coupled to {@link DispatchFeature} since it requires
 * special treatment to terminate the dispatch loop early. It can be thought of
 * as the part of the DispatchFeature implementation that needs to exist in
 * command form.
 * 
 * @author Ashley Williams
 * 
 */
public final class JavaCommand {
	private final class Body implements Runnable {
		public void run() {
			OptionSet options = delegate.getOptions();
			if (options.has(optOption)) {
				List<String> javaOptions = options.valuesOf(optOption);
				startVm(javaOptions);
			}
		}
	}

	/**
	 * Bootstraps to a specified java class. The options parameter contains name
	 * value pairs to be applied to the underlying ant task by beans
	 * introspection, so that the vm launcher can easily be configured from the
	 * command line.
	 * <p>
	 * The bean being configured is the StartVmCommand instance. For example a
	 * string with the content "javaTask.classname=com.acme.MyMain" is
	 * equivalent to calling
	 * <code>StartVmCommand.getJavaTask().setClassname("javaTask.classname=com.acme.MyMain")</code>.
	 * 
	 * @param opts
	 */
	public void startVm(final List<String> opts) {
		final CoreProject project = parent.getProject();
		InstructionChain chain = project.getInstructionChain();
		String[] args = chain.createArgsAndRemove(true);

		DispatchFeature dispatchFeature = project.getDispatchFeature();
		dispatchFeature.startVm(null, args, null,
				new ArgRunnable<StartVmCommand>() {
					public void run(StartVmCommand command) {
						project.getLogger().info("applying java options:");
						for (String opt : opts) {
							String[] property = opt.split("=");
							String name = property[0];
							String value = property[1];
							project.getLogger().info(
									String.format("%s=%s", name, value));
							PropertyUtils.setProperty(command, name, value);
						}
					}
				});
		// ensure remaining commands don't get executed in this vm since they
		// have been bootstrapped to a new vm - we can assume we are in the
		// DispatchFeature dispatch loop since this command class is a helper of
		// that class and not useable anywhere else
		chain.breakVisit();
	}

	/**
	 * Provides the basic command functionality.
	 */
	private Command delegate;

	/**
	 * The parent of this command.
	 */
	private final DispatchFeature parent;

	private OptionSpec<String> optOption;

	public JavaCommand(DispatchFeature dispatchFeature) {
		this.parent = dispatchFeature;
		CommandStore store = dispatchFeature.getProject().getCommandStore();

		StringBuilder description = new StringBuilder();
		delegate = store.addCommand("java", description.toString(), "16m",
				new Body());
		delegate.initBootstrapCurrentVm();
		optOption = delegate.getParser().accepts("optOption").withRequiredArg();
	}

	public Command getDelegate() {
		return delegate;
	}

}
