/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang;

import java.io.File;
import java.util.Collection;
import java.util.Set;
import java.util.TreeMap;

import org.apache.commons.io.FileUtils;
import org.apache.tools.ant.types.DirSet;

import protoj.core.ArgRunnable;
import protoj.core.ProjectLayout;
import protoj.core.internal.InformationException;
import protoj.lang.internal.ant.AssembleTask;
import protoj.lang.internal.ant.JavadocTask;

/**
 * Responsible for creating one or more archives from the parent javadoc files.
 * Call {@link #addArchive(String, String, String, String, String, ArgRunnable)}
 * to configure each additional archive with the specified name and
 * {@link #createArchive(String)} to create one of the added archive. During
 * creation, the <code>config</code> argument specified in the constructor will
 * be called back to give the caller a chance to provide further archive
 * configuration.
 * 
 * @author Ashley Williams
 * 
 */
public final class JavadocArchive {
	/**
	 * Adds javadoc specific information to a wrapped {@link ArchiveEntry}
	 * instance.
	 * 
	 * @author Ashley Williams
	 * 
	 */
	public static final class JavadocEntry {
		/**
		 * See {@link #getArchiveEntry()}.
		 */
		private ArchiveEntry<JavadocArchive> archiveEntry;

		/**
		 * See {@link #getMemory()}.
		 */
		private String memory;

		/**
		 * See the {@link ArchiveEntry} accessors.
		 * 
		 * @param parent
		 * @param name
		 * @param fileName
		 * @param manifest
		 * @param includes
		 * @param excludes
		 * @param config
		 */
		public JavadocEntry(ArchiveFeature parent, String name,
				String fileName, String manifest, String includes,
				String excludes, String memory,
				ArgRunnable<JavadocArchive> config) {
			this.archiveEntry = new ArchiveEntry<JavadocArchive>(name, parent,
					fileName, manifest, includes, excludes, config);
			this.memory = memory;
		}

		/**
		 * The maximum amount of memory that the javadoc vm should allocate.
		 * 
		 * @return
		 */
		public String getMemory() {
			return memory;
		}

		/**
		 * The wrapped helper.
		 * 
		 * @return
		 */
		public ArchiveEntry<JavadocArchive> getArchiveEntry() {
			return archiveEntry;
		}

	}

	/**
	 * The parent that owns this instance lifecycle.
	 */
	private final ArchiveFeature parent;

	/**
	 * The configuration information for each added archive.
	 */
	private TreeMap<String, JavadocEntry> entries = new TreeMap<String, JavadocEntry>();

	/**
	 * See {@link #getCurrentAssembleTask()}.
	 */
	private AssembleTask currentAssembleTask;

	/**
	 * See {@link #getCurrentEntry()}.
	 */
	private ArchiveEntry<JavadocArchive> currentEntry;

	/**
	 * See {@link #getWorkingDir()}.
	 */
	private File workingDir;

	/**
	 * See {@link #getJavadocTask()}.
	 */
	private JavadocTask javadocTask;

	/**
	 * Creates with the parent feature.
	 * 
	 * @param parent
	 */
	public JavadocArchive(ArchiveFeature parent) {
		this.parent = parent;
		this.workingDir = new File(parent.getProject().getLayout()
				.getArchiveDir(), "javadoc-archive");
	}

	/**
	 * Use to enable creation of an additional archive.
	 * 
	 * @param name
	 *            the name of the jar file to be created, without the extension.
	 * @param manifest
	 *            the name of the manifest to be used from the manifest
	 *            directory, without the extension. Can be null if a default
	 *            manifest is required.
	 * @param includes
	 *            the resources that should be included in the archive, can be
	 *            null to include all resources
	 * @param excludes
	 *            the resources that should be excluded in the archive, can be
	 *            null to exclude no resources
	 * @param memory
	 *            the maximum amount of memory to be used by the javadoc vm, eg
	 *            "16m"
	 * @param config
	 *            this will be called back during the invocatin of
	 *            {@link #createArchive(String)} if the caller wishes to provide
	 *            further configuration. Can be null if the defaults are ok.
	 */

	public void addArchive(String name, String manifest, String includes,
			String excludes, String memory, ArgRunnable<JavadocArchive> config) {
		if (memory == null) {
			throw new InformationException(
					"please specify a non null value for javadoc memory");
		}
		String fileName = String.format("%s-%s.jar", name, parent.getLayout()
				.getJavadocPostfix());
		JavadocEntry entry = new JavadocEntry(parent, name, fileName, manifest,
				includes, excludes, memory, config);
		entries.put(name, entry);
	}

	/**
	 * Delegates to {@link #createArchive(String)} for each archive that was
	 * added through a call to
	 * {@link #addArchive(String, String, String, String, String, ArgRunnable)}.
	 */
	public void createArchives() {
		Set<String> keys = entries.keySet();
		for (String key : keys) {
			createArchive(key);
		}
	}

	/**
	 * Creates the jar for the given name under
	 * {@link ProjectLayout#getArchiveDir()}. The jar file will have a name of
	 * [name]-javadoc.jar.
	 * 
	 * @param name
	 *            the name as specified in the call to
	 *            {@link #addArchive(String, String, String, String, String, ArgRunnable)}
	 *            .
	 */
	public void createArchive(String name) {
		ProjectLayout layout = parent.getProject().getLayout();

		// creates javadocs in working directory
		getWorkingDir().mkdirs();
		JavadocEntry entry = entries.get(name);
		String memory = entry.getMemory();
		File srcDir = layout.getJavaDir();
		javadocTask = new JavadocTask(srcDir, getWorkingDir());
		javadocTask.initLogging(layout.getLogFile());
		javadocTask.getjavadoc().setMaxmemory(memory);
		DirSet packageSet = javadocTask.getPackageSet();
		packageSet.setDefaultexcludes(true);
		javadocTask.execute();

		currentAssembleTask = entry.getArchiveEntry().createAssembleTask(
				getWorkingDir());
		ArgRunnable<JavadocArchive> config = entry.getArchiveEntry()
				.getConfig();
		if (config != null) {
			config.run(this);
		}
		currentAssembleTask.execute();
		FileUtils.deleteDirectory(getWorkingDir());
	}

	/**
	 * Convenient method that visits each entry in this archive.
	 * 
	 * @param visitor
	 */
	public void visit(ArgRunnable<JavadocEntry> visitor) {
		Collection<JavadocEntry> values = entries.values();
		for (JavadocEntry entry : values) {
			visitor.run(entry);
		}
	}

	/**
	 * Accessor for the entry corresponding to the given name.
	 * 
	 * @param name
	 * @return
	 */
	public JavadocEntry getEntry(String name) {
		return entries.get(name);
	}

	/**
	 * The ant task responsible for creating the javadocs during the call to
	 * {@link #createArchive(String)}.
	 * 
	 * @return
	 */
	public JavadocTask getJavadocTask() {
		return javadocTask;
	}

	/**
	 * Used to create the javadoc files before archiving.
	 * 
	 * @return
	 */
	public File getWorkingDir() {
		return workingDir;
	}

	/**
	 * The ant task responsible for creating the jar during the call to
	 * {@link #createArchive(String)}.
	 * 
	 * @return
	 */
	public AssembleTask getCurrentAssembleTask() {
		return currentAssembleTask;
	}

	/**
	 * The instance used to hold information about the the archive being created
	 * during the call to {@link #createArchive(String)}.
	 * 
	 * @return
	 */
	public ArchiveEntry<JavadocArchive> getCurrentEntry() {
		return currentEntry;
	}
}