/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang;

import java.io.File;

import org.apache.tools.ant.taskdefs.optional.ssh.Scp;

import protoj.core.ProjectLayout;
import protoj.core.internal.InformationException;
import protoj.lang.internal.ant.ScpTask;

/**
 * This feature is used to remotely copy the project archive generated by
 * {@link ProjectArchive}.
 * 
 * @author Ashley Williams
 * 
 */
public final class ScpFeature {
	private ScpTask scpTask;

	/**
	 * See {@link #getProject()}.
	 */
	private final StandardProject project;

	/**
	 * See {@link #getToDir()}.
	 */
	private String toDir;

	/**
	 * See {@link #getKeyFile()}.
	 */
	private String keyFile;

	/**
	 * Creates an instance with the owning parent project and also the plug-in
	 * runnable responsible for further configuration.
	 * 
	 * @param project
	 */
	public ScpFeature(StandardProject project) {
		this.project = project;
	}

	/**
	 * Copies the project tar file to the remote location using simple password
	 * authentication. The <code>toDir</code> parameter takes the form
	 * <code><username>@<host>:<dest></code>.
	 * 
	 * @param toDir
	 * @param password
	 */
	public void copyProjectArchive(String toDir, String password) {
		copyProjectArchive(false, toDir, null, password);
	}

	/**
	 * Copies the project tar file to the remote location using key based
	 * authentication. The <code>toDir</code> parameter takes the form
	 * <code><username>@<host>:<dest></code>.
	 * 
	 * @param toDir
	 * @param passphrase
	 * @param keyFile
	 */
	public void copyProjectArchive(String toDir, String passphrase,
			String keyFile) {
		copyProjectArchive(true, toDir, keyFile, passphrase);
	}

	private void copyProjectArchive(boolean isKeyAuth, String toDir,
			String keyFile, String passText) {
		this.toDir = toDir;
		this.keyFile = keyFile;
		this.scpTask = createScpTask();
		if (isKeyAuth) {
			this.scpTask.initKeyAuthentication(passText, keyFile);
		} else {
			this.scpTask.initPasswordAuthentication(passText);
		}
		this.scpTask.execute();
	}

	/**
	 * Creates the default scp task instance ready for further customization.
	 * true.
	 * 
	 * @return
	 */
	private ScpTask createScpTask() {
		String tarPath = project.getArchiveFeature().getProjectArchive()
				.getArchivePath();
		File tarFile = new File(tarPath);
		if (!tarFile.exists()) {
			throw new InformationException(
					"no tar file: scp won't work until a tar file has been generated with the tar command");
		}
		ScpTask scp = new ScpTask(toDir, tarFile);
		ProjectLayout layout = project.getLayout();
		File logFile = layout.getLogFile();
		scp.initLogging(logFile);
		return scp;
	}

	/**
	 * The parent project of this feature.
	 * 
	 * @return
	 */
	public StandardProject getProject() {
		return project;
	}

	/**
	 * The underlying scp ant object for fine grained customization.
	 * 
	 * @return
	 */
	public Scp getAntScp() {
		return scpTask.getScp();
	}

	/**
	 * The toDir of the remote machine.
	 * 
	 * @return
	 */
	public String getToDir() {
		return toDir;
	}

	/**
	 * The location of the private key when key based authentication is used.
	 * 
	 * @return
	 */
	public String getKeyFile() {
		return keyFile;
	}

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("host: ");
		builder.append(scpTask.getScp().getHost());
		return builder.toString();
	}

}
