/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang;

import java.io.File;

import org.apache.log4j.Logger;

import protoj.core.CommandStore;
import protoj.core.CompileFeature;
import protoj.core.DispatchFeature;
import protoj.core.InstructionChain;
import protoj.core.ProjectLayout;
import protoj.core.PropertyStore;
import protoj.core.ProtoLogger;
import protoj.core.ResourceFeature;
import protoj.core.RetrieveFeature;
import protoj.core.StandardProperties;
import protoj.core.internal.CoreProject;

/**
 * This is the starting point class for all projects that wish to take advantage
 * of ProtoJ pure java use-cases from compilation and javadoc generation to
 * installation and configuration. There is no need to construct any other
 * protoj class other than this one as it is the domain object that aggregates
 * all the others. Neither is there any need to write even one line of xml -
 * that is unless you take advantage of the dependency download feature or
 * artifact publish feature, which use the xml based ivy/maven API.
 * <p>
 * Although there are many classes in the ProtoJ library many of them are inside
 * packages qualified with *internal* which means they are implementation
 * classes. This leaves little more than a couple of dozen pojos to get to grips
 * with in order to gain an understanding of ProtoJ. You can find these classes
 * directly under the <code>protoj.lang</code> and <code>protoj.core</code>
 * packages.
 * <p>
 * To get started simply embed an instance of StandardProject in your own pojo
 * and invoke the initXXX() methods for features that you wish to use and use
 * the {@link CommandStore} domain object to add your own project specific
 * commands. Here is a simple but complete example of a class called
 * <code>foo.Foo</code>:
 * 
 * <pre>
 * package protoj.lang.internal.sample;
 * 
 * import protoj.lang.StandardProject;
 * 
 * final class Foo {
 * 	public static void main(String[] args) {
 * 		StandardProject project = new StandardProject(args, &quot;Foo Version 1.0&quot;);
 * 		project.initCompile(false, new ArgRunnable&lt;CompileFeature&gt;() {
 * 			public void run(CompileFeature feature) {
 * 				feature.getAjcCompileTask().getAjcTask().setMaxmem(&quot;16m&quot;);
 * 				feature.getAjcCompileTask().getAjcTask().setSource(&quot;1.5&quot;);
 * 			}
 * 		});
 * 		project.initJunit(&quot;16m&quot;);
 * 		project.getArchiveFeature().initProjectArchive(&quot;foo&quot;, &quot;foo-1.0-RC1&quot;);
 * 		project.getArchiveFeature().addClasses(&quot;foo-1.0&quot;);
 * 		project.getArchiveFeature().addSources(&quot;foo-1.0&quot;);
 * 		project.getArchiveFeature().addJavadoc(&quot;foo-1.0&quot;, &quot;16m&quot;);
 * 		project.getCommandStore().addCommand(&quot;hello-foo&quot;,
 * 				&quot;writes to the console&quot;, &quot;16m&quot;, new Runnable() {
 * 					public void run() {
 * 						System.out.println(&quot;Hello Foo!&quot;);
 * 					}
 * 				});
 * 		project.getDispatchFeature().dispatchCommands();
 * 	}
 * }
 * </pre>
 * 
 * Take a look at {@link ProjectLayout} to find out more about the directory
 * structure of a ProtoJ project and where the java code in particular should be
 * placed. Be sure to modify the demo project though rather than creating a new
 * project from scratch each time - do this by executing the no-dependencies
 * ProtoJ jar file with <code>java -jar</code>.
 * <p>
 * In order to get to this world of pure java, some mechanism must take
 * responsibility for compiling and running the application in the first place
 * and one solution is to use a shell script. Check out <code>demo.sh</code> and
 * <code>demo.bat</code> from the demo project for some examples. Here is the
 * call to java from a typical bash shell script:
 * 
 * <pre>
 * java ... helloworld.HelloWorld &quot;init --rootDir `pwd`/.. --scriptName $0&quot; &quot;$@&quot;
 * </pre>
 * 
 * Note the special <code>init</code> command is specified as an argument to
 * java, with the mandatory <code>--rootDir</code> and <code>--scriptName</code>
 * options that protoj requires in order to function:
 * 
 * And also note the use of the bash <code>$@</code> place-holder so that
 * additional commands such as <code>compile</code>, <code>jar</code> and
 * <code>test</code> can be specified at the command line.
 * 
 * @author Ashley Williams
 * 
 */
public final class StandardProject {

	/**
	 * Handles the standard project use-cases, some of which core back to this
	 * class.
	 */
	private CoreProject core;

	/**
	 * See {@link #getSampleProjectFeature()}.
	 */
	private SampleProjectFeature sampleProjectFeature;

	/**
	 * See {@link #getInfoFeature()}.
	 */
	private InfoFeature infoFeature;

	/**
	 * See {@link #getPublishFeature()}.
	 */
	private PublishFeature publishFeature;

	/**
	 * See {@link #getJunitFeature()}.
	 */
	private JunitFeature junitFeature;

	/**
	 * See {@link #getArchiveFeature()}.
	 */
	private ArchiveFeature archiveFeature;

	/**
	 * See {@link #getUploadGoogleCodeFeature()}.
	 */
	private UploadGoogleCodeFeature uploadGoogleCodeFeature;

	/**
	 * See {@link #getScpFeature()}.
	 */
	private ScpFeature scpFeature;

	/**
	 * See {@link #getConfigureFeature()}.
	 */
	private ConfigureFeature configureFeature;

	/**
	 * See {@link #getVerifyTarFeature()}.
	 */
	private VerifyTarFeature verifyTarFeature;

	/**
	 * See {@link #getCommands()}.
	 */
	private StandardCommands commands;

	/**
	 * Parses arguments originating from <code>main()</code>. The easiest way to
	 * describe the format is with an example:
	 * 
	 * <pre>
	 * compile &quot;find-alien -gx milky-way&quot; --rootDir &tilde;/dev/myproj/ --scriptName alien.sh -Dprotoj.debug
	 * </pre>
	 * 
	 * which results in the following args array elements:
	 * <p>
	 * <ul>
	 * <li><code>arg[0]="compile"</code> - command with no arguments</li>
	 * <li><code>arg[1]="find-alien -gx milky-way"</code> - command with a
	 * single argument</li>
	 * <li><code>arg[2]="--rootDir"</code> - the project root directory option
	 * name</li>
	 * <li><code>arg[3]="~/dev/myproj/"</code> - the project root directory
	 * option value</li>
	 * <li><code>arg[4]="--scriptName"</code> - the project script name option
	 * name</li>
	 * <li><code>arg[5]="alien.sh"</code> - the project script name option value
	 * </li>
	 * <li><code>arg[6]="-Dprotoj.debug"</code> - system property</li>
	 * </ul>
	 * So in other words the two types of argument are commands and system
	 * properties. Also the rootDir and scriptName options are mandatory and
	 * used to calculate the project layout - see {@link InstructionChain} for
	 * more information
	 * <p>
	 * Note that although <code>-D</code> syntax is used for the vm arguments,
	 * these are not the same as the genuine vm arguments that won't appear in
	 * the <code>args</code> parameter at all. This is because we need to store
	 * them so that they can successfully be passed to any new virtual machines
	 * that need starting.
	 * 
	 * @param args
	 *            these should be the arguments from <code>main()</code>
	 * @param versionInfo
	 *            any information to be reported when the version command is
	 *            invoked
	 */
	public StandardProject(String[] args, String versionInfo) {
		this(new CoreProject(args), versionInfo);
	}

	/**
	 * This constructor is useful in situations where the commands have to built
	 * up in-code rather than parsed from a command line. For example in unit
	 * tests and also in {@link VerifyTarFeature}.
	 * 
	 * @param rootDir
	 *            the root directory of the project
	 * @param scriptName
	 *            the name of the script responsible for issuing commands - if
	 *            no extension is given then ".bat" is assumed for windows and
	 *            ".sh" otherwise
	 * @param versionInfo
	 *            any information to be reported when the version command is
	 *            invoked
	 */
	public StandardProject(File rootDir, String scriptName, String versionInfo) {
		this(new CoreProject(rootDir, scriptName), versionInfo);
	}

	/**
	 * Create a default instance of the project. Only useful for a limited
	 * number of features since it isn't tied to any particular directory or
	 * launch script.
	 */
	public StandardProject() {
		this(new File("."), "unspecified", null);
	}

	/**
	 * Intended for internal use only.
	 * 
	 * @param core
	 * @param versionInfo
	 */
	public StandardProject(CoreProject core, String versionInfo) {
		this.core = core;
		this.sampleProjectFeature = new SampleProjectFeature();
		this.infoFeature = new InfoFeature(this, versionInfo);
		this.scpFeature = new ScpFeature(this);
		this.verifyTarFeature = new VerifyTarFeature(this);
		this.archiveFeature = new ArchiveFeature(this);
		this.commands = new StandardCommands(this);
	}

	/**
	 * This method must be called if you wish to be able to publish project
	 * artifacts created with the {@link ArchiveFeature} class.
	 * 
	 * @param url
	 */
	public void initPublish(String url) {
		this.publishFeature = new PublishFeature(this, url);
	}

	/**
	 * This method must be called if junit tests are required, in order to
	 * specify the amount of memory required. Invoke junit funtionality by first
	 * obtaining the delegate with a call to {@link #getJunitFeature()}.
	 * 
	 * @param memory
	 *            uses standard java tool notation, e.g. "32m"
	 */
	public void initJunit(String memory) {
		this.junitFeature = new JunitFeature(this, memory);
	}

	/**
	 * This method must be called if configuration with properties files is to
	 * be supported. Broadly speaking, setting interpolated to true means that
	 * property placeholders of the form ${var} in config files will get
	 * replaced with their real values. This is useful if you use libraries that
	 * can't handle property placeholders at runtime. See
	 * {@link ConfigureFeature} for more information.
	 * 
	 * @param interpolated
	 */
	public void initConfig(boolean interpolated) {
		this.configureFeature = new ConfigureFeature(this, interpolated);
	}

	/**
	 * This method must be called if the ability to upload the project tar file
	 * to google code is required. This is a pass-through method to
	 * {@link UploadGoogleCodeFeature#UploadGoogleCodeFeature(StandardProject, String)}
	 * , so see that constructor for a description of the parameters.
	 * <p>
	 * Note that there is no corresponding out of the box command for this
	 * feature. That's because it would be too generic to be useful due to the
	 * many pieces of information that would need to be specified at the command
	 * line. Projects would be best served by writing their own commands with
	 * the information hardcoded, such as the artifact name etc.
	 * 
	 * @param googleProjectName
	 * @return
	 */
	public UploadGoogleCodeFeature initUploadGoogleCode(String googleProjectName) {
		uploadGoogleCodeFeature = new UploadGoogleCodeFeature(this,
				googleProjectName);
		return uploadGoogleCodeFeature;
	}

	/**
	 * Creates a helper that is able to drive the project represented by this
	 * class instance through its command line interface. Very useful for tests
	 * for example that need to go through the script as the user would.
	 * 
	 * @return
	 */
	public ScriptSession createScriptSession() {
		return new ScriptSession(getLayout().getShellScript());
	}

	/**
	 * A reference to the delegate object used to create a sample project. See
	 * {@link SampleProjectFeature}.
	 * 
	 * @return
	 */
	public SampleProjectFeature getSampleProjectFeature() {
		return sampleProjectFeature;
	}

	/**
	 * A reference to the delegate object used to extract resources to the
	 * filing system for APIs that can only work in this way and also apply
	 * variable ${var} substitutions. See {@link ResourceFeature}.
	 * 
	 * @return
	 */
	public ResourceFeature getResourceFeature() {
		return core.getResourceFeature();
	}

	/**
	 * A reference to the delegate object used to dispatch the instructions
	 * usually specified at the command line. See {@link DispatchFeature}.
	 * 
	 * @return
	 */
	public DispatchFeature getDispatchFeature() {
		return core.getDispatchFeature();
	}

	/**
	 * A reference to the delegate object used to perform in-code compilation .
	 * See {@link CompileFeature}.
	 * 
	 * @return
	 */
	public CompileFeature getCompileFeature() {
		return core.getCompileFeature();
	}

	/**
	 * A reference to the delegate object used to download project dependencies.
	 * See {@link RetrieveFeature}.
	 * 
	 * @return
	 */
	public RetrieveFeature getRetrieveFeature() {
		return core.getRetrieveFeature();
	}

	/**
	 * A reference to the delegate object used to provide configuration
	 * functionality. See {@link JunitFeature}.
	 * 
	 * @return
	 */
	public JunitFeature getJunitFeature() {
		return junitFeature;
	}

	/**
	 * A reference to the delegate object used to provide archive creation
	 * functionality. See {@link ArchiveFeature}.
	 * 
	 * @return
	 */
	public ArchiveFeature getArchiveFeature() {
		return archiveFeature;
	}

	/**
	 * A reference to the delegate object used to provide configuration
	 * functionality. See {@link VerifyTarFeature}.
	 * 
	 * @return
	 */
	public VerifyTarFeature getVerifyTarFeature() {
		return verifyTarFeature;
	}

	/**
	 * A reference to the delegate object used to provide the google code upload
	 * functionality. See {@link UploadGoogleCodeFeature}.
	 * 
	 * @return
	 */
	public UploadGoogleCodeFeature getUploadGoogleCodeFeature() {
		return uploadGoogleCodeFeature;
	}

	/**
	 * A reference to the delegate object used to provide scp functionality. See
	 * {@link ScpFeature}.
	 * 
	 * @return
	 */
	public ScpFeature getScpFeature() {
		return scpFeature;
	}

	/**
	 * A reference to the delegate object used to provide configuration
	 * functionality. See {@link ConfigureFeature}.
	 * 
	 * @return
	 */
	public ConfigureFeature getConfigureFeature() {
		return configureFeature;
	}

	/**
	 * A reference to the delegate object used to provide project information
	 * functionality. See {@link InfoFeature}.
	 * 
	 * @return
	 */
	public InfoFeature getInfoFeature() {
		return infoFeature;
	}

	/**
	 * A reference to the delegate object used to publish project artifacts. See
	 * {@link PublishFeature}.
	 * 
	 * @return
	 */
	public PublishFeature getPublishFeature() {
		return publishFeature;
	}

	/**
	 * Access to the class responsible for containing all the instructions to be
	 * dispatched.
	 * 
	 * @return
	 */
	public InstructionChain getInstructionChain() {
		return core.getInstructionChain();
	}

	/**
	 * Delegated functionality, see {@link CommandStore}.
	 * 
	 * @return
	 */
	public CommandStore getCommandStore() {
		return core.getCommandStore();
	}

	/**
	 * A container of references of those commands belonging to this
	 * StandardProject that have been added to the {@link CommandStore}.
	 * 
	 * @return
	 */
	public StandardCommands getCommands() {
		return commands;
	}

	/**
	 * Delegated functionality, see {@link PropertyStore}.
	 * 
	 * @return
	 */
	public PropertyStore getPropertyStore() {
		return core.getPropertyStore();
	}

	/**
	 * A container of references of those properties belonging to this
	 * StandardProject that have been added to the {@link PropertyStore}.
	 * 
	 * @return
	 */
	public StandardProperties getProperties() {
		return core.getProperties();
	}

	/**
	 * Delegated functionality, see {@link ProtoLogger}.
	 * 
	 * @return
	 */
	public ProtoLogger getProtoLogger() {
		return core.getProtoLogger();
	}

	/**
	 * Access to the protoj log4j instance. Convenient short-hand for
	 * <code>getProtoLogger().getProtoLogger()</code>.
	 * 
	 * @return
	 */
	public Logger getLogger() {
		return core.getLogger();
	}

	/**
	 * Delegated functionality, see {@link ProjectLayout}.
	 * 
	 * @return
	 */
	public ProjectLayout getLayout() {
		return core.getLayout();
	}

	/**
	 * Provides much of the implementation of this project.
	 * 
	 * @return
	 */
	CoreProject getCore() {
		return core;
	}

}
