/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal.acme;

import java.io.File;

import org.junit.Assert;

import protoj.core.ArgRunnable;
import protoj.core.ProjectLayout;
import protoj.lang.ScriptSession;

/**
 * See the conditional logic for the different types of tag name.
 * 
 * @author Ashley Williams
 * 
 */
final class AssertCompile implements ArgRunnable<ScriptSession> {
	public static final String EMPTY = "empty";
	public static final String ALL = "all";
	public static final String UTIL = "util";
	public static final String RETRIEVE = "retrieve";
	private final AcmeSession acmeSession;

	public AssertCompile(AcmeSession acmeSession) {
		this.acmeSession = acmeSession;
	}

	public void run(ScriptSession projectSession) {
		ProjectLayout acmeLayout = acmeSession.getProject().getLayout();
		String log = acmeSession.getProject().getLayout().loadLog();

		String tag = projectSession.getCurrentTag();
		File acmeProjectClass = new File(acmeLayout.getClassesDir(),
				"acme/AcmeProject.class");
		File acmeUtilClass = new File(acmeLayout.getClassesDir(),
				"acme/util/AcmeUtil.class");
		File sourcesJar = new File(acmeLayout.getLibDir(),
				"junit-4.4-sources.jar");
		File resource = new File(acmeLayout.getClassesDir(), "test.txt");
		if (tag.equals(EMPTY)) {
			// checks that before the compiler is executed, just the
			// core classes are already compiled
			Assert.assertFalse("shouldn't find "
					+ acmeUtilClass.getAbsolutePath(), acmeUtilClass.exists());
			Assert.assertFalse("shouldn't find "
					+ acmeProjectClass.getAbsolutePath(), acmeProjectClass
					.exists());
			Assert.assertFalse(
					"shouldn't find " + sourcesJar.getAbsolutePath(),
					sourcesJar.exists());
		} else if (tag.equals(RETRIEVE)) {
			// checks that when the ivy command is executed, the
			// junit source jar should be downloaded
			Assert.assertTrue("couldn't find " + sourcesJar.getAbsolutePath(),
					sourcesJar.exists());
		} else if (tag.equals(UTIL)) {
			// checks that when the util includes pattern is specified
			// then just that package should be compiled
			Assert.assertTrue("couldn't find "
					+ acmeUtilClass.getAbsolutePath(), acmeUtilClass.exists());
			Assert.assertFalse("shouldn't find "
					+ acmeProjectClass.getAbsolutePath(), acmeProjectClass
					.exists());
		} else if (tag.equals(ALL)) {
			// checks that when no includes pattern is provided,
			// all the classes should be compiled and also that
			// the resources are copied into the classes directory
			Assert.assertTrue("couldn't find "
					+ acmeUtilClass.getAbsolutePath(), acmeUtilClass.exists());
			Assert.assertTrue("couldn't find "
					+ acmeProjectClass.getAbsolutePath(), acmeProjectClass
					.exists());
			Assert.assertTrue("couldn't find " + resource.getAbsolutePath(),
					resource.exists());
			Assert.assertTrue(log, log.contains("compile"));
			Assert.assertTrue(log, log.contains("Task \"ajc\" started"));
			Assert.assertTrue(log, log.contains("Task \"ajc\" finished"));
		}
		Assert.assertTrue(projectSession.getCurrentExec().isSuccess());
		acmeLayout.getLogFile().delete();
	}
}