/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal.acme;

import java.io.File;

import org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.commons.io.FileUtils;
import org.junit.Assert;

import protoj.core.ArgRunnable;
import protoj.core.ProjectLayout;
import protoj.lang.ScriptSession;
import protoj.lang.internal.ant.CommandTask;

/**
 * These assertions trigger when the configure command is invoked and ensures
 * that interpolated properties work ok. In other words that an ordinary file is
 * copied correctly, a properties file is merged and that in all cases variable
 * substitution works. Additionally the undo functionality is tested so that we
 * check all the matching files in the project directory get deleted.
 * 
 * @author Ashley Williams
 * 
 */
public final class AssertConfigure implements ArgRunnable<ScriptSession> {

	public static final String CONFIG = "config";
	public static final String UNDO = "undo";
	private final AcmeSession acmeSession;

	public AssertConfigure(AcmeSession acmeSession) {
		this.acmeSession = acmeSession;
	}

	public void run(ScriptSession projectSession) {
		ProjectLayout layout = acmeSession.getProject().getLayout();
		File propertiesFile = layout.getPropertiesFile();
		File sampleTextFile = new File(layout.getDocsDir(), "sample.txt");
		File samplePropertiesFile = new File(layout.getDocsDir(),
				"sample.properties");

		String tag = projectSession.getCurrentTag();
		if (tag.equals(CONFIG)) {
			Assert
					.assertTrue("couldn't find "
							+ propertiesFile.getAbsolutePath(), propertiesFile
							.exists());
			Assert.assertTrue("couldn't find "
					+ samplePropertiesFile.getAbsolutePath(),
					samplePropertiesFile.exists());
			// check property merge and interpolation worked
			PropertiesConfiguration config = new PropertiesConfiguration(
					propertiesFile);
			Assert.assertTrue(config.containsKey("color"));
			Assert.assertEquals(config.getString("color"), "red");
			Assert.assertTrue(config.containsKey("greeting"));
			Assert.assertEquals(config.getString("greeting"), "hi");
			Assert.assertTrue(config.containsKey("foo"));
			Assert.assertEquals(config.getString("foo"), "hi");
			Assert.assertTrue(config.containsKey("drink"));
			Assert.assertEquals(config.getString("drink"), "coffee");
			Assert.assertTrue(config.containsKey("random"));
			Assert.assertEquals(config.getString("random"), "coffee");
			Assert.assertTrue(config.containsKey("word"));
			Assert.assertEquals(config.getString("word"), "hi");
			Assert.assertTrue("couldn't find "
					+ samplePropertiesFile.getAbsolutePath(), sampleTextFile
					.exists());
			String sampleText = FileUtils.readFileToString(sampleTextFile);
			Assert.assertTrue(sampleText, sampleText
					.contains("My favorite color is red"));
			Assert
					.assertTrue(
							sampleText,
							sampleText
									.contains("My favorite acme constant is AcmeUtil constant"));
			Assert.assertTrue(sampleText, sampleText
					.contains("My favorite version of Java is "
							+ System.getProperty("java.version")));
		} else if (tag.equals(UNDO)) {
			Assert
					.assertFalse("shouldn't find "
							+ propertiesFile.getAbsolutePath(), propertiesFile
							.exists());
			Assert
					.assertFalse("shouldn't find "
							+ sampleTextFile.getAbsolutePath(), sampleTextFile
							.exists());
		}
		// check command exited ok
		CommandTask exec = projectSession.getCurrentExec();
		Assert.assertTrue(exec.getResult(), exec.isSuccess());

		layout.getLogFile().delete();
	}
}
